package io.apicurio.registry.rest.v2;

import io.apicurio.registry.rest.v2.beans.ArtifactTypeInfo;
import io.apicurio.registry.rest.v2.beans.ConfigurationProperty;
import io.apicurio.registry.rest.v2.beans.LogConfiguration;
import io.apicurio.registry.rest.v2.beans.NamedLogConfiguration;
import io.apicurio.registry.rest.v2.beans.RoleMapping;
import io.apicurio.registry.rest.v2.beans.Rule;
import io.apicurio.registry.rest.v2.beans.UpdateConfigurationProperty;
import io.apicurio.registry.rest.v2.beans.UpdateRole;
import io.apicurio.registry.types.RuleType;
import jakarta.validation.constraints.NotNull;
import jakarta.ws.rs.Consumes;
import jakarta.ws.rs.DELETE;
import jakarta.ws.rs.GET;
import jakarta.ws.rs.HeaderParam;
import jakarta.ws.rs.POST;
import jakarta.ws.rs.PUT;
import jakarta.ws.rs.Path;
import jakarta.ws.rs.PathParam;
import jakarta.ws.rs.Produces;
import jakarta.ws.rs.QueryParam;
import jakarta.ws.rs.core.Response;
import java.io.InputStream;
import java.util.List;

/**
 * A JAX-RS interface. An implementation of this interface must be provided.
 */
@Path("/apis/registry/v2/admin")
public interface AdminResource {
  /**
   * <p>
   * Gets a list of all the configured artifact types.
   * </p>
   * <p>
   * This operation can fail for the following reasons:
   * </p>
   * <ul>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/artifactTypes")
  @GET
  @Produces("application/json")
  List<ArtifactTypeInfo> listArtifactTypes();

  /**
   * <p>
   * Gets a list of all the currently configured global rules (if any).
   * </p>
   * <p>
   * This operation can fail for the following reasons:
   * </p>
   * <ul>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/rules")
  @GET
  @Produces("application/json")
  List<RuleType> listGlobalRules();

  /**
   * <p>
   * Adds a rule to the list of globally configured rules.
   * </p>
   * <p>
   * This operation can fail for the following reasons:
   * </p>
   * <ul>
   * <li>The rule type is unknown (HTTP error <code>400</code>)</li>
   * <li>The rule already exists (HTTP error <code>409</code>)</li>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/rules")
  @POST
  @Consumes("application/json")
  void createGlobalRule(@NotNull Rule data);

  /**
   * <p>
   * Deletes all globally configured rules.
   * </p>
   * <p>
   * This operation can fail for the following reasons:
   * </p>
   * <ul>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/rules")
  @DELETE
  void deleteAllGlobalRules();

  /**
   * <p>
   * Returns information about the named globally configured rule.
   * </p>
   * <p>
   * This operation can fail for the following reasons:
   * </p>
   * <ul>
   * <li>Invalid rule name/type (HTTP error <code>400</code>)</li>
   * <li>No rule with name/type <code>rule</code> exists (HTTP error
   * <code>404</code>)</li>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/rules/{rule}")
  @GET
  @Produces("application/json")
  Rule getGlobalRuleConfig(@PathParam("rule") RuleType rule);

  /**
   * <p>
   * Updates the configuration for a globally configured rule.
   * </p>
   * <p>
   * This operation can fail for the following reasons:
   * </p>
   * <ul>
   * <li>Invalid rule name/type (HTTP error <code>400</code>)</li>
   * <li>No rule with name/type <code>rule</code> exists (HTTP error
   * <code>404</code>)</li>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/rules/{rule}")
  @PUT
  @Produces("application/json")
  @Consumes("application/json")
  Rule updateGlobalRuleConfig(@PathParam("rule") RuleType rule, @NotNull Rule data);

  /**
   * <p>
   * Deletes a single global rule. If this is the only rule configured, this is
   * the same as deleting <strong>all</strong> rules.
   * </p>
   * <p>
   * This operation can fail for the following reasons:
   * </p>
   * <ul>
   * <li>Invalid rule name/type (HTTP error <code>400</code>)</li>
   * <li>No rule with name/type <code>rule</code> exists (HTTP error
   * <code>404</code>)</li>
   * <li>Rule cannot be deleted (HTTP error <code>409</code>)</li>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/rules/{rule}")
  @DELETE
  void deleteGlobalRule(@PathParam("rule") RuleType rule);

  /**
   * <p>
   * List all of the configured logging levels. These override the default logging
   * configuration.
   * </p>
   * 
   */
  @Path("/loggers")
  @GET
  @Produces("application/json")
  List<NamedLogConfiguration> listLogConfigurations();

  /**
   * <p>
   * Returns the configured logger configuration for the provided logger name, if
   * no logger configuration is persisted it will return the current default log
   * configuration in the system.
   * </p>
   * 
   */
  @Path("/loggers/{logger}")
  @GET
  @Produces("application/json")
  NamedLogConfiguration getLogConfiguration(@PathParam("logger") String logger);

  /**
   * <p>
   * Configures the logger referenced by the provided logger name with the given
   * configuration.
   * </p>
   * 
   */
  @Path("/loggers/{logger}")
  @PUT
  @Produces("application/json")
  @Consumes("application/json")
  NamedLogConfiguration setLogConfiguration(@PathParam("logger") String logger, @NotNull LogConfiguration data);

  /**
   * <p>
   * Removes the configured logger configuration (if any) for the given logger.
   * </p>
   * 
   */
  @Path("/loggers/{logger}")
  @DELETE
  @Produces("application/json")
  NamedLogConfiguration removeLogConfiguration(@PathParam("logger") String logger);

  /**
   * <p>
   * Exports registry data as a ZIP archive.
   * </p>
   * 
   */
  @Path("/export")
  @GET
  @Produces({"application/json", "application/zip"})
  Response exportData(@QueryParam("forBrowser") Boolean forBrowser);

  /**
   * <p>
   * Imports registry data that was previously exported using the
   * <code>/admin/export</code> operation.
   * </p>
   * 
   */
  @Path("/import")
  @POST
  @Consumes("application/zip")
  void importData(@HeaderParam("X-Registry-Preserve-GlobalId") Boolean xRegistryPreserveGlobalId,
      @HeaderParam("X-Registry-Preserve-ContentId") Boolean xRegistryPreserveContentId, @NotNull InputStream data);

  /**
   * <p>
   * Gets the details of a single role mapping (by <code>principalId</code>).
   * </p>
   * <p>
   * This operation can fail for the following reasons:
   * </p>
   * <ul>
   * <li>No role mapping for the <code>principalId</code> exists (HTTP error
   * <code>404</code>)</li>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/roleMappings/{principalId}")
  @GET
  @Produces("application/json")
  RoleMapping getRoleMapping(@PathParam("principalId") String principalId);

  /**
   * <p>
   * Updates a single role mapping for one user/principal.
   * </p>
   * <p>
   * This operation can fail for the following reasons:
   * </p>
   * <ul>
   * <li>No role mapping for the principalId exists (HTTP error
   * <code>404</code>)</li>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/roleMappings/{principalId}")
  @PUT
  @Consumes("application/json")
  void updateRoleMapping(@PathParam("principalId") String principalId, @NotNull UpdateRole data);

  /**
   * <p>
   * Deletes a single role mapping, effectively denying access to a
   * user/principal.
   * </p>
   * <p>
   * This operation can fail for the following reasons:
   * </p>
   * <ul>
   * <li>No role mapping for the principalId exists (HTTP error
   * <code>404</code>)</li>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/roleMappings/{principalId}")
  @DELETE
  void deleteRoleMapping(@PathParam("principalId") String principalId);

  /**
   * <p>
   * Gets a list of all role mappings configured in the registry (if any).
   * </p>
   * <p>
   * This operation can fail for the following reasons:
   * </p>
   * <ul>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/roleMappings")
  @GET
  @Produces("application/json")
  List<RoleMapping> listRoleMappings();

  /**
   * <p>
   * Creates a new mapping between a user/principal and a role.
   * </p>
   * <p>
   * This operation can fail for the following reasons:
   * </p>
   * <ul>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/roleMappings")
  @POST
  @Consumes("application/json")
  void createRoleMapping(@NotNull RoleMapping data);

  /**
   * <p>
   * Returns a list of all configuration properties that have been set. The list
   * is not paged.
   * </p>
   * <p>
   * This operation may fail for one of the following reasons:
   * </p>
   * <ul>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/config/properties")
  @GET
  @Produces("application/json")
  List<ConfigurationProperty> listConfigProperties();

  /**
   * <p>
   * Returns the value of a single configuration property.
   * </p>
   * <p>
   * This operation may fail for one of the following reasons:
   * </p>
   * <ul>
   * <li>Property not found or not configured (HTTP error <code>404</code>)</li>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/config/properties/{propertyName}")
  @GET
  @Produces("application/json")
  ConfigurationProperty getConfigProperty(@PathParam("propertyName") String propertyName);

  /**
   * <p>
   * Updates the value of a single configuration property.
   * </p>
   * <p>
   * This operation may fail for one of the following reasons:
   * </p>
   * <ul>
   * <li>Property not found or not configured (HTTP error <code>404</code>)</li>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/config/properties/{propertyName}")
  @PUT
  @Consumes("application/json")
  void updateConfigProperty(@PathParam("propertyName") String propertyName, @NotNull UpdateConfigurationProperty data);

  /**
   * <p>
   * Resets the value of a single configuration property. This will return the
   * property to its default value (see external documentation for supported
   * properties and their default values).
   * </p>
   * <p>
   * This operation may fail for one of the following reasons:
   * </p>
   * <ul>
   * <li>Property not found or not configured (HTTP error <code>404</code>)</li>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/config/properties/{propertyName}")
  @DELETE
  void resetConfigProperty(@PathParam("propertyName") String propertyName);
}
