package io.apicurio.registry.rest.v3;

import io.apicurio.registry.rest.v3.beans.AddVersionToBranch;
import io.apicurio.registry.rest.v3.beans.ArtifactMetaData;
import io.apicurio.registry.rest.v3.beans.ArtifactReference;
import io.apicurio.registry.rest.v3.beans.ArtifactSearchResults;
import io.apicurio.registry.rest.v3.beans.ArtifactSortBy;
import io.apicurio.registry.rest.v3.beans.BranchMetaData;
import io.apicurio.registry.rest.v3.beans.BranchSearchResults;
import io.apicurio.registry.rest.v3.beans.Comment;
import io.apicurio.registry.rest.v3.beans.CreateArtifact;
import io.apicurio.registry.rest.v3.beans.CreateArtifactResponse;
import io.apicurio.registry.rest.v3.beans.CreateBranch;
import io.apicurio.registry.rest.v3.beans.CreateGroup;
import io.apicurio.registry.rest.v3.beans.CreateRule;
import io.apicurio.registry.rest.v3.beans.CreateVersion;
import io.apicurio.registry.rest.v3.beans.EditableArtifactMetaData;
import io.apicurio.registry.rest.v3.beans.EditableBranchMetaData;
import io.apicurio.registry.rest.v3.beans.EditableGroupMetaData;
import io.apicurio.registry.rest.v3.beans.EditableVersionMetaData;
import io.apicurio.registry.rest.v3.beans.GroupMetaData;
import io.apicurio.registry.rest.v3.beans.GroupSearchResults;
import io.apicurio.registry.rest.v3.beans.GroupSortBy;
import io.apicurio.registry.rest.v3.beans.HandleReferencesType;
import io.apicurio.registry.rest.v3.beans.IfArtifactExists;
import io.apicurio.registry.rest.v3.beans.NewComment;
import io.apicurio.registry.rest.v3.beans.ReplaceBranchVersions;
import io.apicurio.registry.rest.v3.beans.Rule;
import io.apicurio.registry.rest.v3.beans.SortOrder;
import io.apicurio.registry.rest.v3.beans.VersionMetaData;
import io.apicurio.registry.rest.v3.beans.VersionSearchResults;
import io.apicurio.registry.rest.v3.beans.VersionSortBy;
import io.apicurio.registry.types.ReferenceType;
import io.apicurio.registry.types.RuleType;
import jakarta.validation.constraints.NotNull;
import jakarta.validation.constraints.Pattern;
import jakarta.ws.rs.Consumes;
import jakarta.ws.rs.DELETE;
import jakarta.ws.rs.GET;
import jakarta.ws.rs.POST;
import jakarta.ws.rs.PUT;
import jakarta.ws.rs.Path;
import jakarta.ws.rs.PathParam;
import jakarta.ws.rs.Produces;
import jakarta.ws.rs.QueryParam;
import jakarta.ws.rs.core.Response;
import java.math.BigInteger;
import java.util.List;

/**
 * A JAX-RS interface. An implementation of this interface must be provided.
 */
@Path("/apis/registry/v3/groups")
public interface GroupsResource {
  /**
   * <p>
   * Retrieves the metadata for a single version of the artifact. The version
   * metadata is a subset of the artifact metadata and only includes the metadata
   * that is specific to the version (for example, this doesn't include
   * <code>modifiedOn</code>).
   * </p>
   * <p>
   * This operation can fail for the following reasons:
   * </p>
   * <ul>
   * <li>No artifact with this <code>artifactId</code> exists (HTTP error
   * <code>404</code>)</li>
   * <li>No version with this <code>version</code> exists (HTTP error
   * <code>404</code>)</li>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/{groupId}/artifacts/{artifactId}/versions/{versionExpression}")
  @GET
  @Produces("application/json")
  VersionMetaData getArtifactVersionMetaData(@PathParam("groupId") @Pattern(regexp = "^.{1,512}$") String groupId,
      @PathParam("artifactId") @Pattern(regexp = "^.{1,512}$") String artifactId,
      @PathParam("versionExpression") String versionExpression);

  /**
   * <p>
   * Updates the user-editable portion of the artifact version's metadata. Only
   * some of the metadata fields are editable by the user. For example,
   * <code>description</code> is editable, but <code>createdOn</code> is not.
   * </p>
   * <p>
   * This operation can fail for the following reasons:
   * </p>
   * <ul>
   * <li>No artifact with this <code>artifactId</code> exists (HTTP error
   * <code>404</code>)</li>
   * <li>No version with this <code>version</code> exists (HTTP error
   * <code>404</code>)</li>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/{groupId}/artifacts/{artifactId}/versions/{versionExpression}")
  @PUT
  @Consumes("application/json")
  void updateArtifactVersionMetaData(@PathParam("groupId") @Pattern(regexp = "^.{1,512}$") String groupId,
      @PathParam("artifactId") @Pattern(regexp = "^.{1,512}$") String artifactId,
      @PathParam("versionExpression") String versionExpression, @NotNull EditableVersionMetaData data);

  /**
   * <p>
   * Deletes a single version of the artifact. Parameters <code>groupId</code>,
   * <code>artifactId</code> and the unique <code>version</code> are needed. If
   * this is the only version of the artifact, this operation is the same as
   * deleting the entire artifact.
   * </p>
   * <p>
   * This feature is disabled by default and it's discouraged for normal usage. To
   * enable it, set the <code>registry.rest.artifact.deletion.enabled</code>
   * property to true. This operation can fail for the following reasons:
   * </p>
   * <ul>
   * <li>No artifact with this <code>artifactId</code> exists (HTTP error
   * <code>404</code>)</li>
   * <li>No version with this <code>version</code> exists (HTTP error
   * <code>404</code>)</li>
   * <li>Feature is disabled (HTTP error <code>405</code>)</li>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/{groupId}/artifacts/{artifactId}/versions/{versionExpression}")
  @DELETE
  void deleteArtifactVersion(@PathParam("groupId") @Pattern(regexp = "^.{1,512}$") String groupId,
      @PathParam("artifactId") @Pattern(regexp = "^.{1,512}$") String artifactId,
      @PathParam("versionExpression") String versionExpression);

  /**
   * <p>
   * Returns a list of all rules configured for the artifact. The set of rules
   * determines how the content of an artifact can evolve over time. If no rules
   * are configured for an artifact, the set of globally configured rules are
   * used. If no global rules are defined, there are no restrictions on content
   * evolution.
   * </p>
   * <p>
   * This operation can fail for the following reasons:
   * </p>
   * <ul>
   * <li>No artifact with this <code>artifactId</code> exists (HTTP error
   * <code>404</code>)</li>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/{groupId}/artifacts/{artifactId}/rules")
  @GET
  @Produces("application/json")
  List<RuleType> listArtifactRules(@PathParam("groupId") @Pattern(regexp = "^.{1,512}$") String groupId,
      @PathParam("artifactId") @Pattern(regexp = "^.{1,512}$") String artifactId);

  /**
   * <p>
   * Adds a rule to the list of rules that get applied to the artifact when adding
   * new versions. All configured rules must pass to successfully add a new
   * artifact version.
   * </p>
   * <p>
   * This operation can fail for the following reasons:
   * </p>
   * <ul>
   * <li>No artifact with this <code>artifactId</code> exists (HTTP error
   * <code>404</code>)</li>
   * <li>Rule (named in the request body) is unknown (HTTP error
   * <code>400</code>)</li>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/{groupId}/artifacts/{artifactId}/rules")
  @POST
  @Consumes("application/json")
  void createArtifactRule(@PathParam("groupId") @Pattern(regexp = "^.{1,512}$") String groupId,
      @PathParam("artifactId") @Pattern(regexp = "^.{1,512}$") String artifactId, @NotNull CreateRule data);

  /**
   * <p>
   * Deletes all of the rules configured for the artifact. After this is done, the
   * global rules apply to the artifact again.
   * </p>
   * <p>
   * This operation can fail for the following reasons:
   * </p>
   * <ul>
   * <li>No artifact with this <code>artifactId</code> exists (HTTP error
   * <code>404</code>)</li>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/{groupId}/artifacts/{artifactId}/rules")
  @DELETE
  void deleteArtifactRules(@PathParam("groupId") @Pattern(regexp = "^.{1,512}$") String groupId,
      @PathParam("artifactId") @Pattern(regexp = "^.{1,512}$") String artifactId);

  /**
   * <p>
   * Returns information about a single rule configured for an artifact. This is
   * useful when you want to know what the current configuration settings are for
   * a specific rule.
   * </p>
   * <p>
   * This operation can fail for the following reasons:
   * </p>
   * <ul>
   * <li>No artifact with this <code>artifactId</code> exists (HTTP error
   * <code>404</code>)</li>
   * <li>No rule with this name/type is configured for this artifact (HTTP error
   * <code>404</code>)</li>
   * <li>Invalid rule type (HTTP error <code>400</code>)</li>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/{groupId}/artifacts/{artifactId}/rules/{ruleType}")
  @GET
  @Produces("application/json")
  Rule getArtifactRuleConfig(@PathParam("groupId") @Pattern(regexp = "^.{1,512}$") String groupId,
      @PathParam("artifactId") @Pattern(regexp = "^.{1,512}$") String artifactId,
      @PathParam("ruleType") RuleType ruleType);

  /**
   * <p>
   * Updates the configuration of a single rule for the artifact. The
   * configuration data is specific to each rule type, so the configuration of the
   * <code>COMPATIBILITY</code> rule is in a different format from the
   * configuration of the <code>VALIDITY</code> rule.
   * </p>
   * <p>
   * This operation can fail for the following reasons:
   * </p>
   * <ul>
   * <li>No artifact with this <code>artifactId</code> exists (HTTP error
   * <code>404</code>)</li>
   * <li>No rule with this name/type is configured for this artifact (HTTP error
   * <code>404</code>)</li>
   * <li>Invalid rule type (HTTP error <code>400</code>)</li>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/{groupId}/artifacts/{artifactId}/rules/{ruleType}")
  @PUT
  @Produces("application/json")
  @Consumes("application/json")
  Rule updateArtifactRuleConfig(@PathParam("groupId") @Pattern(regexp = "^.{1,512}$") String groupId,
      @PathParam("artifactId") @Pattern(regexp = "^.{1,512}$") String artifactId,
      @PathParam("ruleType") RuleType ruleType, @NotNull Rule data);

  /**
   * <p>
   * Deletes a rule from the artifact. This results in the rule no longer applying
   * for this artifact. If this is the only rule configured for the artifact, this
   * is the same as deleting <strong>all</strong> rules, and the globally
   * configured rules now apply to this artifact.
   * </p>
   * <p>
   * This operation can fail for the following reasons:
   * </p>
   * <ul>
   * <li>No artifact with this <code>artifactId</code> exists (HTTP error
   * <code>404</code>)</li>
   * <li>No rule with this name/type is configured for this artifact (HTTP error
   * <code>404</code>)</li>
   * <li>Invalid rule type (HTTP error <code>400</code>)</li>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/{groupId}/artifacts/{artifactId}/rules/{ruleType}")
  @DELETE
  void deleteArtifactRule(@PathParam("groupId") @Pattern(regexp = "^.{1,512}$") String groupId,
      @PathParam("artifactId") @Pattern(regexp = "^.{1,512}$") String artifactId,
      @PathParam("ruleType") RuleType ruleType);

  /**
   * <p>
   * Retrieves all references for a single version of an artifact. Both the
   * <code>artifactId</code> and the unique <code>version</code> number must be
   * provided. Using the <code>refType</code> query parameter, it is possible to
   * retrieve an array of either the inbound or outbound references.
   * </p>
   * <p>
   * This operation can fail for the following reasons:
   * </p>
   * <ul>
   * <li>No artifact with this <code>artifactId</code> exists (HTTP error
   * <code>404</code>)</li>
   * <li>No version with this <code>version</code> exists (HTTP error
   * <code>404</code>)</li>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/{groupId}/artifacts/{artifactId}/versions/{versionExpression}/references")
  @GET
  @Produces("application/json")
  List<ArtifactReference> getArtifactVersionReferences(
      @PathParam("groupId") @Pattern(regexp = "^.{1,512}$") String groupId,
      @PathParam("artifactId") @Pattern(regexp = "^.{1,512}$") String artifactId,
      @PathParam("versionExpression") String versionExpression, @QueryParam("refType") ReferenceType refType);

  /**
   * <p>
   * Returns a list of all artifacts in the group. This list is paged.
   * </p>
   * 
   */
  @Path("/{groupId}/artifacts")
  @GET
  @Produces("application/json")
  ArtifactSearchResults listArtifactsInGroup(@PathParam("groupId") @Pattern(regexp = "^.{1,512}$") String groupId,
      @QueryParam("limit") BigInteger limit, @QueryParam("offset") BigInteger offset,
      @QueryParam("order") SortOrder order, @QueryParam("orderby") ArtifactSortBy orderby);

  /**
   * <p>
   * Creates a new artifact. The body of the request should be a
   * <code>CreateArtifact</code> object, which includes the metadata of the new
   * artifact and, optionally, the metadata and content of the first version.
   * </p>
   * <p>
   * If the artifact type is not provided, the registry attempts to figure out
   * what kind of artifact is being added from the following supported list:
   * </p>
   * <ul>
   * <li>Avro (<code>AVRO</code>)</li>
   * <li>Protobuf (<code>PROTOBUF</code>)</li>
   * <li>JSON Schema (<code>JSON</code>)</li>
   * <li>Kafka Connect (<code>KCONNECT</code>)</li>
   * <li>OpenAPI (<code>OPENAPI</code>)</li>
   * <li>AsyncAPI (<code>ASYNCAPI</code>)</li>
   * <li>GraphQL (<code>GRAPHQL</code>)</li>
   * <li>Web Services Description Language (<code>WSDL</code>)</li>
   * <li>XML Schema (<code>XSD</code>)</li>
   * </ul>
   * <p>
   * An artifact will be created using the unique artifact ID that can optionally
   * be provided in the request body. If not provided in the request, the server
   * will generate a unique ID for the artifact. It is typically recommended that
   * callers provide the ID, because it is typically a meaningful identifier, and
   * as such for most use cases should be supplied by the caller.
   * </p>
   * <p>
   * If an artifact with the provided artifact ID already exists, the default
   * behavior is for the server to reject the content with a 409 error. However,
   * the caller can supply the <code>ifExists</code> query parameter to alter this
   * default behavior. The <code>ifExists</code> query parameter can have one of
   * the following values:
   * </p>
   * <ul>
   * <li><code>FAIL</code> (<em>default</em>) - server rejects the content with a
   * 409 error</li>
   * <li><code>UPDATE</code> - server updates the existing artifact and returns
   * the new metadata</li>
   * <li><code>RETURN</code> - server does not create or add content to the
   * server, but instead returns the metadata for the existing artifact</li>
   * <li><code>RETURN_OR_UPDATE</code> - server returns an existing
   * <strong>version</strong> that matches the provided content if such a version
   * exists, otherwise a new version is created</li>
   * </ul>
   * <p>
   * This operation may fail for one of the following reasons:
   * </p>
   * <ul>
   * <li>An invalid <code>ArtifactType</code> was indicated (HTTP error
   * <code>400</code>)</li>
   * <li>No <code>ArtifactType</code> was indicated and the server could not
   * determine one from the content (HTTP error <code>400</code>)</li>
   * <li>Provided content (request body) was empty (HTTP error
   * <code>400</code>)</li>
   * <li>An artifact with the provided ID already exists (HTTP error
   * <code>409</code>)</li>
   * <li>The content violates one of the configured global rules (HTTP error
   * <code>409</code>)</li>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * <p>
   * Note that if the <code>dryRun</code> query parameter is set to
   * <code>true</code>, then this operation will not actually make any changes.
   * Instead it will succeed or fail based on whether it <strong>would have
   * worked</strong>. Use this option to, for example, check if an artifact is
   * valid or if a new version passes configured compatibility checks.
   * </p>
   * 
   */
  @Path("/{groupId}/artifacts")
  @POST
  @Produces("application/json")
  @Consumes("application/json")
  CreateArtifactResponse createArtifact(@PathParam("groupId") @Pattern(regexp = "^.{1,512}$") String groupId,
      @QueryParam("ifExists") IfArtifactExists ifExists, @QueryParam("canonical") Boolean canonical,
      @QueryParam("dryRun") Boolean dryRun, @NotNull CreateArtifact data);

  /**
   * <p>
   * Deletes all of the artifacts that exist in a given group.
   * </p>
   * 
   */
  @Path("/{groupId}/artifacts")
  @DELETE
  void deleteArtifactsInGroup(@PathParam("groupId") @Pattern(regexp = "^.{1,512}$") String groupId);

  /**
   * <p>
   * Returns a list of all groups. This list is paged.
   * </p>
   * 
   */
  @GET
  @Produces("application/json")
  GroupSearchResults listGroups(@QueryParam("limit") BigInteger limit, @QueryParam("offset") BigInteger offset,
      @QueryParam("order") SortOrder order, @QueryParam("orderby") GroupSortBy orderby);

  /**
   * <p>
   * Creates a new group.
   * </p>
   * <p>
   * This operation can fail for the following reasons:
   * </p>
   * <ul>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * <li>The group already exist (HTTP error <code>409</code>)</li>
   * </ul>
   * 
   */
  @POST
  @Produces("application/json")
  @Consumes("application/json")
  GroupMetaData createGroup(@NotNull CreateGroup data);

  /**
   * <p>
   * Retrieves all comments for a version of an artifact. Both the
   * <code>artifactId</code> and the unique <code>version</code> number must be
   * provided.
   * </p>
   * <p>
   * This operation can fail for the following reasons:
   * </p>
   * <ul>
   * <li>No artifact with this <code>artifactId</code> exists (HTTP error
   * <code>404</code>)</li>
   * <li>No version with this <code>version</code> exists (HTTP error
   * <code>404</code>)</li>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/{groupId}/artifacts/{artifactId}/versions/{versionExpression}/comments")
  @GET
  @Produces("application/json")
  List<Comment> getArtifactVersionComments(@PathParam("groupId") @Pattern(regexp = "^.{1,512}$") String groupId,
      @PathParam("artifactId") @Pattern(regexp = "^.{1,512}$") String artifactId,
      @PathParam("versionExpression") String versionExpression);

  /**
   * <p>
   * Adds a new comment to the artifact version. Both the <code>artifactId</code>
   * and the unique <code>version</code> number must be provided.
   * </p>
   * <p>
   * This operation can fail for the following reasons:
   * </p>
   * <ul>
   * <li>No artifact with this <code>artifactId</code> exists (HTTP error
   * <code>404</code>)</li>
   * <li>No version with this <code>version</code> exists (HTTP error
   * <code>404</code>)</li>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/{groupId}/artifacts/{artifactId}/versions/{versionExpression}/comments")
  @POST
  @Produces("application/json")
  @Consumes("application/json")
  Comment addArtifactVersionComment(@PathParam("groupId") @Pattern(regexp = "^.{1,512}$") String groupId,
      @PathParam("artifactId") @Pattern(regexp = "^.{1,512}$") String artifactId,
      @PathParam("versionExpression") String versionExpression, @NotNull NewComment data);

  /**
   * <p>
   * Updates the value of a single comment in an artifact version. Only the owner
   * of the comment can modify it. The <code>artifactId</code>, unique
   * <code>version</code> number, and <code>commentId</code> must be provided.
   * </p>
   * <p>
   * This operation can fail for the following reasons:
   * </p>
   * <ul>
   * <li>No artifact with this <code>artifactId</code> exists (HTTP error
   * <code>404</code>)</li>
   * <li>No version with this <code>version</code> exists (HTTP error
   * <code>404</code>)</li>
   * <li>No comment with this <code>commentId</code> exists (HTTP error
   * <code>404</code>)</li>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/{groupId}/artifacts/{artifactId}/versions/{versionExpression}/comments/{commentId}")
  @PUT
  @Consumes("application/json")
  void updateArtifactVersionComment(@PathParam("groupId") @Pattern(regexp = "^.{1,512}$") String groupId,
      @PathParam("artifactId") @Pattern(regexp = "^.{1,512}$") String artifactId,
      @PathParam("versionExpression") String versionExpression, @PathParam("commentId") String commentId,
      @NotNull NewComment data);

  /**
   * <p>
   * Deletes a single comment in an artifact version. Only the owner of the
   * comment can delete it. The <code>artifactId</code>, unique
   * <code>version</code> number, and <code>commentId</code> must be provided.
   * </p>
   * <p>
   * This operation can fail for the following reasons:
   * </p>
   * <ul>
   * <li>No artifact with this <code>artifactId</code> exists (HTTP error
   * <code>404</code>)</li>
   * <li>No version with this <code>version</code> exists (HTTP error
   * <code>404</code>)</li>
   * <li>No comment with this <code>commentId</code> exists (HTTP error
   * <code>404</code>)</li>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/{groupId}/artifacts/{artifactId}/versions/{versionExpression}/comments/{commentId}")
  @DELETE
  void deleteArtifactVersionComment(@PathParam("groupId") @Pattern(regexp = "^.{1,512}$") String groupId,
      @PathParam("artifactId") @Pattern(regexp = "^.{1,512}$") String artifactId,
      @PathParam("versionExpression") String versionExpression, @PathParam("commentId") String commentId);

  /**
   * <p>
   * Returns a list of all branches in the artifact. Each branch is a list of
   * version identifiers, ordered from the latest (tip of the branch) to the
   * oldest.
   * </p>
   * <p>
   * This operation can fail for the following reasons:
   * </p>
   * <ul>
   * <li>No artifact with this <code>groupId</code> and <code>artifactId</code>
   * exists (HTTP error <code>404</code>)</li>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/{groupId}/artifacts/{artifactId}/branches")
  @GET
  @Produces("application/json")
  BranchSearchResults listBranches(@PathParam("groupId") @Pattern(regexp = "^.{1,512}$") String groupId,
      @PathParam("artifactId") @Pattern(regexp = "^.{1,512}$") String artifactId,
      @QueryParam("offset") BigInteger offset, @QueryParam("limit") BigInteger limit);

  /**
   * <p>
   * Creates a new branch for the artifact. A new branch consists of metadata and
   * a list of versions.
   * </p>
   * <p>
   * This operation can fail for the following reasons:
   * </p>
   * <ul>
   * <li>No artifact with this <code>groupId</code> and <code>artifactId</code>
   * exists (HTTP error <code>404</code>)</li>
   * <li>A branch with the given <code>branchId</code> already exists (HTTP error
   * <code>409</code>)</li>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/{groupId}/artifacts/{artifactId}/branches")
  @POST
  @Produces("application/json")
  @Consumes("application/json")
  BranchMetaData createBranch(@PathParam("groupId") @Pattern(regexp = "^.{1,512}$") String groupId,
      @PathParam("artifactId") @Pattern(regexp = "^.{1,512}$") String artifactId, @NotNull CreateBranch data);

  /**
   * <p>
   * Returns a list of all versions of the artifact. The result set is paged.
   * </p>
   * <p>
   * This operation can fail for the following reasons:
   * </p>
   * <ul>
   * <li>No artifact with this <code>artifactId</code> exists (HTTP error
   * <code>404</code>)</li>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/{groupId}/artifacts/{artifactId}/versions")
  @GET
  @Produces("application/json")
  VersionSearchResults listArtifactVersions(@PathParam("groupId") @Pattern(regexp = "^.{1,512}$") String groupId,
      @PathParam("artifactId") @Pattern(regexp = "^.{1,512}$") String artifactId,
      @QueryParam("offset") BigInteger offset, @QueryParam("limit") BigInteger limit,
      @QueryParam("order") SortOrder order, @QueryParam("orderby") VersionSortBy orderby);

  /**
   * <p>
   * Creates a new version of the artifact by uploading new content. The
   * configured rules for the artifact are applied, and if they all pass, the new
   * content is added as the most recent version of the artifact. If any of the
   * rules fail, an error is returned.
   * </p>
   * <p>
   * The body of the request can be the raw content of the new artifact version,
   * or the raw content and a set of references pointing to other artifacts, and
   * the type of that content should match the artifact's type (for example if the
   * artifact type is <code>AVRO</code> then the content of the request should be
   * an Apache Avro document).
   * </p>
   * <p>
   * This operation can fail for the following reasons:
   * </p>
   * <ul>
   * <li>Provided content (request body) was empty (HTTP error
   * <code>400</code>)</li>
   * <li>No artifact with this <code>artifactId</code> exists (HTTP error
   * <code>404</code>)</li>
   * <li>The new content violates one of the rules configured for the artifact
   * (HTTP error <code>409</code>)</li>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/{groupId}/artifacts/{artifactId}/versions")
  @POST
  @Produces("application/json")
  @Consumes("application/json")
  VersionMetaData createArtifactVersion(@PathParam("groupId") @Pattern(regexp = "^.{1,512}$") String groupId,
      @PathParam("artifactId") @Pattern(regexp = "^.{1,512}$") String artifactId, @QueryParam("dryRun") Boolean dryRun,
      @NotNull CreateVersion data);

  /**
   * <p>
   * Returns the metaData of a branch.
   * </p>
   * <p>
   * This operation can fail for the following reasons:
   * </p>
   * <ul>
   * <li>No artifact with this <code>groupId</code> and <code>artifactId</code>
   * exists (HTTP error <code>404</code>)</li>
   * <li>No branch with this <code>branchId</code> exists (HTTP error
   * <code>404</code>)</li>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/{groupId}/artifacts/{artifactId}/branches/{branchId}")
  @GET
  @Produces("application/json")
  BranchMetaData getBranchMetaData(@PathParam("groupId") @Pattern(regexp = "^.{1,512}$") String groupId,
      @PathParam("artifactId") @Pattern(regexp = "^.{1,512}$") String artifactId,
      @PathParam("branchId") @Pattern(regexp = "^[a-zA-Z0-9._\\-+]{1,256}$") String branchId);

  /**
   * <p>
   * Updates the metadata of a branch.
   * </p>
   * <p>
   * This operation can fail for the following reasons:
   * </p>
   * <ul>
   * <li>No artifact with this <code>groupId</code> and <code>artifactId</code>
   * exists (HTTP error <code>404</code>)</li>
   * <li>No branch with this <code>branchId</code> exists (HTTP error
   * <code>404</code>)</li>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/{groupId}/artifacts/{artifactId}/branches/{branchId}")
  @PUT
  @Consumes("application/json")
  void updateBranchMetaData(@PathParam("groupId") @Pattern(regexp = "^.{1,512}$") String groupId,
      @PathParam("artifactId") @Pattern(regexp = "^.{1,512}$") String artifactId,
      @PathParam("branchId") @Pattern(regexp = "^[a-zA-Z0-9._\\-+]{1,256}$") String branchId,
      @NotNull EditableBranchMetaData data);

  /**
   * <p>
   * Deletes a single branch in the artifact.
   * </p>
   * <p>
   * This operation can fail for the following reasons:
   * </p>
   * <ul>
   * <li>No artifact with this <code>groupId</code> and <code>artifactId</code>
   * exists (HTTP error <code>404</code>)</li>
   * <li>No branch with this <code>branchId</code> exists (HTTP error
   * <code>404</code>)</li>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/{groupId}/artifacts/{artifactId}/branches/{branchId}")
  @DELETE
  void deleteBranch(@PathParam("groupId") @Pattern(regexp = "^.{1,512}$") String groupId,
      @PathParam("artifactId") @Pattern(regexp = "^.{1,512}$") String artifactId,
      @PathParam("branchId") @Pattern(regexp = "^[a-zA-Z0-9._\\-+]{1,256}$") String branchId);

  /**
   * <p>
   * Returns a group using the specified id.
   * </p>
   * <p>
   * This operation can fail for the following reasons:
   * </p>
   * <ul>
   * <li>No group exists with the specified ID (HTTP error <code>404</code>)</li>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/{groupId}")
  @GET
  @Produces("application/json")
  GroupMetaData getGroupById(@PathParam("groupId") @Pattern(regexp = "^.{1,512}$") String groupId);

  /**
   * <p>
   * Updates the metadata of a group using the specified id.
   * </p>
   * <p>
   * This operation can fail for the following reasons:
   * </p>
   * <ul>
   * <li>No group exists with the specified ID (HTTP error <code>404</code>)</li>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/{groupId}")
  @PUT
  @Consumes("application/json")
  void updateGroupById(@PathParam("groupId") @Pattern(regexp = "^.{1,512}$") String groupId,
      @NotNull EditableGroupMetaData data);

  /**
   * <p>
   * Deletes a group by identifier. This operation also deletes all artifacts
   * within the group, so should be used very carefully.
   * </p>
   * <p>
   * This operation can fail for the following reasons:
   * </p>
   * <ul>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * <li>The group does not exist (HTTP error <code>404</code>)</li>
   * </ul>
   * 
   */
  @Path("/{groupId}")
  @DELETE
  void deleteGroupById(@PathParam("groupId") @Pattern(regexp = "^.{1,512}$") String groupId);

  /**
   * <p>
   * Gets the metadata for an artifact in the registry, based on the latest
   * version. If the latest version of the artifact is marked as
   * <code>DISABLED</code>, the next available non-disabled version will be used.
   * The returned metadata includes both generated (read-only) and editable
   * metadata (such as name and description).
   * </p>
   * <p>
   * This operation can fail for the following reasons:
   * </p>
   * <ul>
   * <li>No artifact with this <code>artifactId</code> exists or all versions are
   * <code>DISABLED</code> (HTTP error <code>404</code>)</li>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/{groupId}/artifacts/{artifactId}")
  @GET
  @Produces("application/json")
  ArtifactMetaData getArtifactMetaData(@PathParam("groupId") @Pattern(regexp = "^.{1,512}$") String groupId,
      @PathParam("artifactId") @Pattern(regexp = "^.{1,512}$") String artifactId);

  /**
   * <p>
   * Updates the editable parts of the artifact's metadata. Not all metadata
   * fields can be updated. Note that only the properties included will be
   * updated. You can update only the name by including only the <code>name</code>
   * property in the payload of the request. Properties that are allowed but not
   * present will result in the artifact's metadata not being changed.
   * </p>
   * <p>
   * This operation can fail for the following reasons:
   * </p>
   * <ul>
   * <li>No artifact with the <code>artifactId</code> exists (HTTP error
   * <code>404</code>)</li>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/{groupId}/artifacts/{artifactId}")
  @PUT
  @Consumes("application/json")
  void updateArtifactMetaData(@PathParam("groupId") @Pattern(regexp = "^.{1,512}$") String groupId,
      @PathParam("artifactId") @Pattern(regexp = "^.{1,512}$") String artifactId,
      @NotNull EditableArtifactMetaData data);

  /**
   * <p>
   * Deletes an artifact completely, resulting in all versions of the artifact
   * also being deleted. This may fail for one of the following reasons:
   * </p>
   * <ul>
   * <li>No artifact with the <code>artifactId</code> exists (HTTP error
   * <code>404</code>)</li>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/{groupId}/artifacts/{artifactId}")
  @DELETE
  void deleteArtifact(@PathParam("groupId") @Pattern(regexp = "^.{1,512}$") String groupId,
      @PathParam("artifactId") @Pattern(regexp = "^.{1,512}$") String artifactId);

  /**
   * <p>
   * Retrieves a single version of the artifact content. Both the
   * <code>artifactId</code> and the unique <code>version</code> number must be
   * provided. The <code>Content-Type</code> of the response depends on the
   * artifact type. In most cases, this is <code>application/json</code>, but for
   * some types it may be different (for example, <code>PROTOBUF</code>).
   * </p>
   * <p>
   * This operation can fail for the following reasons:
   * </p>
   * <ul>
   * <li>No artifact with this <code>artifactId</code> exists (HTTP error
   * <code>404</code>)</li>
   * <li>No version with this <code>version</code> exists (HTTP error
   * <code>404</code>)</li>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/{groupId}/artifacts/{artifactId}/versions/{versionExpression}/content")
  @GET
  @Produces("*/*")
  Response getArtifactVersionContent(@PathParam("groupId") @Pattern(regexp = "^.{1,512}$") String groupId,
      @PathParam("artifactId") @Pattern(regexp = "^.{1,512}$") String artifactId,
      @PathParam("versionExpression") String versionExpression,
      @QueryParam("references") HandleReferencesType references);

  /**
   * <p>
   * Get a list of all versions in the branch. Returns a list of version
   * identifiers in the branch, ordered from the latest (tip of the branch) to the
   * oldest.
   * </p>
   * <p>
   * This operation can fail for the following reasons:
   * </p>
   * <ul>
   * <li>No artifact with this <code>groupId</code> and <code>artifactId</code>
   * exists (HTTP error <code>404</code>)</li>
   * <li>No branch with this <code>branchId</code> exists (HTTP error
   * <code>404</code>)</li>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/{groupId}/artifacts/{artifactId}/branches/{branchId}/versions")
  @GET
  @Produces("application/json")
  VersionSearchResults listBranchVersions(@PathParam("groupId") @Pattern(regexp = "^.{1,512}$") String groupId,
      @PathParam("artifactId") @Pattern(regexp = "^.{1,512}$") String artifactId,
      @PathParam("branchId") @Pattern(regexp = "^[a-zA-Z0-9._\\-+]{1,256}$") String branchId,
      @QueryParam("offset") BigInteger offset, @QueryParam("limit") BigInteger limit);

  /**
   * <p>
   * Add a new version to an artifact branch. Branch is created if it does not
   * exist. Returns a list of version identifiers in the artifact branch, ordered
   * from the latest (tip of the branch) to the oldest. This operation can fail
   * for the following reasons:
   * </p>
   * <ul>
   * <li>No artifact with this <code>groupId</code> and <code>artifactId</code>
   * exists (HTTP error <code>404</code>)</li>
   * <li>No branch with this <code>branchId</code> exists (HTTP error
   * <code>404</code>)</li>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/{groupId}/artifacts/{artifactId}/branches/{branchId}/versions")
  @PUT
  @Consumes("application/json")
  void replaceBranchVersions(@PathParam("groupId") @Pattern(regexp = "^.{1,512}$") String groupId,
      @PathParam("artifactId") @Pattern(regexp = "^.{1,512}$") String artifactId,
      @PathParam("branchId") @Pattern(regexp = "^[a-zA-Z0-9._\\-+]{1,256}$") String branchId,
      @NotNull ReplaceBranchVersions data);

  /**
   * <p>
   * Add a new version to an artifact branch. Returns a list of version
   * identifiers in the branch, ordered from the latest (tip of the branch) to the
   * oldest.
   * </p>
   * <p>
   * This operation can fail for the following reasons:
   * </p>
   * <ul>
   * <li>No artifact with this <code>groupId</code> and <code>artifactId</code>
   * exists (HTTP error <code>404</code>)</li>
   * <li>No branch with this <code>branchId</code> exists (HTTP error
   * <code>404</code>)</li>
   * <li>Branch already contains the given version. Artifact branches are
   * append-only, cycles and history rewrites, except by replacing the entire
   * branch using the replaceBranchVersions operation, are not supported. (HTTP
   * error <code>409</code>)</li>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/{groupId}/artifacts/{artifactId}/branches/{branchId}/versions")
  @POST
  @Consumes("application/json")
  void addVersionToBranch(@PathParam("groupId") @Pattern(regexp = "^.{1,512}$") String groupId,
      @PathParam("artifactId") @Pattern(regexp = "^.{1,512}$") String artifactId,
      @PathParam("branchId") @Pattern(regexp = "^[a-zA-Z0-9._\\-+]{1,256}$") String branchId,
      @NotNull AddVersionToBranch data);
}
