package io.apicurio.registry.rest.v3;

import io.apicurio.registry.rest.v3.beans.ArtifactSearchResults;
import io.apicurio.registry.rest.v3.beans.ArtifactSortBy;
import io.apicurio.registry.rest.v3.beans.GroupSearchResults;
import io.apicurio.registry.rest.v3.beans.GroupSortBy;
import io.apicurio.registry.rest.v3.beans.SortOrder;
import io.apicurio.registry.rest.v3.beans.VersionSearchResults;
import io.apicurio.registry.rest.v3.beans.VersionSortBy;
import jakarta.validation.constraints.NotNull;
import jakarta.validation.constraints.Pattern;
import jakarta.ws.rs.Consumes;
import jakarta.ws.rs.DefaultValue;
import jakarta.ws.rs.GET;
import jakarta.ws.rs.POST;
import jakarta.ws.rs.Path;
import jakarta.ws.rs.Produces;
import jakarta.ws.rs.QueryParam;
import java.io.InputStream;
import java.math.BigInteger;
import java.util.List;

/**
 * A JAX-RS interface. An implementation of this interface must be provided.
 */
@Path("/apis/registry/v3/search")
public interface SearchResource {
  /**
   * <p>
   * Returns a paginated list of all artifacts that match the provided filter
   * criteria.
   * </p>
   * <p>
   * This operation can fail for the following reasons:
   * </p>
   * <ul>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/artifacts")
  @GET
  @Produces("application/json")
  ArtifactSearchResults searchArtifacts(@QueryParam("name") String name,
      @QueryParam("offset") @DefaultValue("0") BigInteger offset,
      @QueryParam("limit") @DefaultValue("20") BigInteger limit, @QueryParam("order") SortOrder order,
      @QueryParam("orderby") ArtifactSortBy orderby, @QueryParam("labels") List<String> labels,
      @QueryParam("description") String description, @QueryParam("groupId") String groupId,
      @QueryParam("globalId") Long globalId, @QueryParam("contentId") Long contentId,
      @QueryParam("artifactId") String artifactId);

  /**
   * <p>
   * Returns a paginated list of all artifacts with at least one version that
   * matches the posted content.
   * </p>
   * <p>
   * This operation can fail for the following reasons:
   * </p>
   * <ul>
   * <li>Provided content (request body) was empty (HTTP error
   * <code>400</code>)</li>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/artifacts")
  @POST
  @Produces("application/json")
  @Consumes("*/*")
  ArtifactSearchResults searchArtifactsByContent(@QueryParam("canonical") Boolean canonical,
      @QueryParam("artifactType") String artifactType, @QueryParam("groupId") String groupId,
      @QueryParam("offset") @DefaultValue("0") BigInteger offset,
      @QueryParam("limit") @DefaultValue("20") BigInteger limit, @QueryParam("order") SortOrder order,
      @QueryParam("orderby") ArtifactSortBy orderby, @NotNull InputStream data);

  /**
   * <p>
   * Returns a paginated list of all groups that match the provided filter
   * criteria.
   * </p>
   * <p>
   * This operation can fail for the following reasons:
   * </p>
   * <ul>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/groups")
  @GET
  @Produces("application/json")
  GroupSearchResults searchGroups(@QueryParam("offset") @DefaultValue("0") BigInteger offset,
      @QueryParam("limit") @DefaultValue("20") BigInteger limit, @QueryParam("order") SortOrder order,
      @QueryParam("orderby") GroupSortBy orderby, @QueryParam("labels") List<String> labels,
      @QueryParam("description") String description, @QueryParam("groupId") String groupId);

  /**
   * <p>
   * Returns a paginated list of all versions that match the provided filter
   * criteria.
   * </p>
   * <p>
   * This operation can fail for the following reasons:
   * </p>
   * <ul>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/versions")
  @GET
  @Produces("application/json")
  VersionSearchResults searchVersions(
      @QueryParam("version") @Pattern(regexp = "^[a-zA-Z0-9._\\-+]{1,256}$") String version,
      @QueryParam("offset") @DefaultValue("0") BigInteger offset,
      @QueryParam("limit") @DefaultValue("20") BigInteger limit, @QueryParam("order") SortOrder order,
      @QueryParam("orderby") VersionSortBy orderby, @QueryParam("labels") List<String> labels,
      @QueryParam("description") String description,
      @QueryParam("groupId") @Pattern(regexp = "^.{1,512}$") String groupId, @QueryParam("globalId") Long globalId,
      @QueryParam("contentId") Long contentId,
      @QueryParam("artifactId") @Pattern(regexp = "^.{1,512}$") String artifactId, @QueryParam("name") String name);

  /**
   * <p>
   * Returns a paginated list of all versions that match the posted content.
   * </p>
   * <p>
   * This operation can fail for the following reasons:
   * </p>
   * <ul>
   * <li>Provided content (request body) was empty (HTTP error
   * <code>400</code>)</li>
   * <li>A server error occurred (HTTP error <code>500</code>)</li>
   * </ul>
   * 
   */
  @Path("/versions")
  @POST
  @Produces("application/json")
  @Consumes("*/*")
  VersionSearchResults searchVersionsByContent(@QueryParam("canonical") Boolean canonical,
      @QueryParam("artifactType") String artifactType, @QueryParam("offset") @DefaultValue("0") BigInteger offset,
      @QueryParam("limit") @DefaultValue("20") BigInteger limit, @QueryParam("order") SortOrder order,
      @QueryParam("orderby") VersionSortBy orderby,
      @QueryParam("groupId") @Pattern(regexp = "^.{1,512}$") String groupId,
      @QueryParam("artifactId") @Pattern(regexp = "^.{1,512}$") String artifactId, @NotNull InputStream data);
}
