/*
 * Copyright 2014 JBoss Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.apiman.gateway.engine.ispn;

import io.apiman.gateway.engine.async.AsyncResultImpl;
import io.apiman.gateway.engine.async.IAsyncResultHandler;
import io.apiman.gateway.engine.components.IRateLimiterComponent;
import io.apiman.gateway.engine.components.rate.RateLimitResponse;
import io.apiman.gateway.engine.rates.RateBucketPeriod;
import io.apiman.gateway.engine.rates.RateLimiterBucket;

import java.util.Collections;
import java.util.Map;

/**
 * Rate limiter component backed by an Infinispan cache.  This allows rate limiting
 * to be done across nodes in a cluster of gateways.
 *
 * @author eric.wittmann@redhat.com
 */
public class InfinispanRateLimiterComponent extends AbstractInfinispanComponent implements IRateLimiterComponent {

    private static final String DEFAULT_CACHE_CONTAINER = "java:jboss/infinispan/container/apiman-gateway"; //$NON-NLS-1$
    private static final String DEFAULT_CACHE = "rate-limiter"; //$NON-NLS-1$

    private Object mutex = new Object();

    /**
     * Constructor.
     */
    public InfinispanRateLimiterComponent() {
        this(Collections.EMPTY_MAP);
    }

    /**
     * Constructor.
     * @param config the config
     */
    public InfinispanRateLimiterComponent(Map<String, String> config) {
        super(config, DEFAULT_CACHE_CONTAINER, DEFAULT_CACHE);
    }

    /**
     * @see io.apiman.gateway.engine.components.IRateLimiterComponent#accept(java.lang.String, io.apiman.gateway.engine.rates.RateBucketPeriod, long, long, io.apiman.gateway.engine.async.IAsyncResultHandler)
     */
    @Override
    public void accept(final String bucketId, final RateBucketPeriod period, final long limit,
            final long increment, final IAsyncResultHandler<RateLimitResponse> handler) {
        RateLimiterBucket bucket = null;
        synchronized (mutex) {
            bucket = (RateLimiterBucket) getCache().get(bucketId);
            if (bucket == null) {
                bucket = new RateLimiterBucket();
                getCache().put(bucketId, bucket);
            }
            bucket.resetIfNecessary(period);

            RateLimitResponse response = new RateLimitResponse();
            if (bucket.getCount() > limit) {
                response.setAccepted(false);
            } else {
                response.setAccepted(bucket.getCount() < limit);
                bucket.setCount(bucket.getCount() + increment);
                bucket.setLast(System.currentTimeMillis());
            }
            int reset = (int) (bucket.getResetMillis(period) / 1000L);
            response.setReset(reset);
            response.setRemaining(limit - bucket.getCount());
            handler.handle(AsyncResultImpl.<RateLimitResponse>create(response));
            getCache().put(bucketId, bucket);
        }
    }
}
