package io.apitestbase.core.teststep;

import com.ibm.mq.*;
import com.ibm.mq.constants.CMQC;
import com.ibm.mq.headers.MQMD;
import com.ibm.mq.headers.*;
import io.apitestbase.models.endpoint.Endpoint;
import io.apitestbase.models.endpoint.MqConnectionMode;
import io.apitestbase.models.endpoint.MqEndpointProperties;
import io.apitestbase.models.teststep.*;
import io.apitestbase.models.teststep.apirequest.MqEnqueueOrPublishFromFileRequest;
import io.apitestbase.models.teststep.apirequest.MqEnqueueOrPublishFromTextRequest;
import io.apitestbase.models.teststep.apirequest.MqRequest;
import io.apitestbase.models.teststep.apiresponse.ApiResponse;
import io.apitestbase.models.teststep.apiresponse.MqCheckQueueDepthResponse;
import io.apitestbase.models.teststep.apiresponse.MqDequeueResponse;
import io.apitestbase.utils.GeneralUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.ByteArrayInputStream;
import java.io.DataInputStream;
import java.io.IOException;
import java.util.GregorianCalendar;
import java.util.Hashtable;

public class MQTeststepActionRunner extends TeststepActionRunner {
    private static final Logger LOGGER = LoggerFactory.getLogger(MQTeststepActionRunner.class);

    @Override
    public TeststepActionRunResult run() throws Exception {
        Teststep teststep = getTeststep();
        String action = teststep.getAction();
        if (teststep.getAction() == null) {
            throw new Exception("Action not specified.");
        }
        MqTeststepProperties teststepProperties = (MqTeststepProperties) teststep.getOtherProperties();
        if (teststepProperties.getDestinationType() == null) {
            throw new Exception("Destination type not specified.");
        }

        TeststepActionRunResult basicTeststepRun = new TeststepActionRunResult();

        ApiResponse response = null;
        Endpoint endpoint = teststep.getEndpoint();
        MqEndpointProperties endpointProperties = (MqEndpointProperties) endpoint.getOtherProperties();
        MQQueueManager queueManager = null;
        try {
            //  connect to queue manager
            if (endpointProperties.getConnectionMode() == MqConnectionMode.BINDINGS) {
                queueManager = new MQQueueManager(endpointProperties.getQueueManagerName());
            } else {
                Hashtable qmConnProperties = new Hashtable();
                qmConnProperties.put(CMQC.HOST_NAME_PROPERTY,  endpoint.getHost());
                qmConnProperties.put(CMQC.PORT_PROPERTY, endpoint.getPort());
                qmConnProperties.put(CMQC.CHANNEL_PROPERTY, endpointProperties.getSvrConnChannelName());
                queueManager = new MQQueueManager(endpointProperties.getQueueManagerName(), qmConnProperties);
            }

            if (MqDestinationType.QUEUE == teststepProperties.getDestinationType()) {
                response = doQueueAction(queueManager, teststepProperties.getQueueName(), action,
                        (MqRequest) teststep.getApiRequest());
            } else if (MqDestinationType.TOPIC == teststepProperties.getDestinationType()) {
                doTopicAction(queueManager, teststepProperties.getTopicString(), action,
                        (MqRequest) teststep.getApiRequest());
            }
        } finally {
            if (queueManager != null) {
                queueManager.disconnect();
            }
        }

        basicTeststepRun.setResponse(response);

        return basicTeststepRun;
    }

    private ApiResponse doQueueAction(MQQueueManager queueManager, String queueName, String action,
                                      MqRequest apiRequest) throws Exception {
        ApiResponse response = null;
        MQQueue queue = null;
        int openOptions = CMQC.MQOO_FAIL_IF_QUIESCING + CMQC.MQOO_INPUT_SHARED;
        try {
            //  open queue
            if (Teststep.ACTION_CHECK_DEPTH.equals(action)) {
                openOptions += CMQC.MQOO_INQUIRE;
            } else if (Teststep.ACTION_ENQUEUE.equals(action)) {
                openOptions += CMQC.MQOO_OUTPUT;
            }
            try {
                queue = queueManager.accessQueue(queueName, openOptions, null, null, null);
            } catch (MQException mqEx) {
                if (mqEx.getCompCode() == CMQC.MQCC_FAILED && mqEx.getReason() == CMQC.MQRC_UNKNOWN_OBJECT_NAME) {
                    throw new Exception("Queue \"" + queueName + "\" not found.");
                } else {
                    throw mqEx;
                }
            }

            //  do the action
            if (Teststep.ACTION_CLEAR.equals(action)) {
                clearQueue(queue);
            } else if (Teststep.ACTION_CHECK_DEPTH.equals(action)) {
                response = new MqCheckQueueDepthResponse();
                ((MqCheckQueueDepthResponse) response).setQueueDepth(queue.getCurrentDepth());
            } else if (Teststep.ACTION_DEQUEUE.equals(action)) {
                response = dequeue(queue);
            } else if (Teststep.ACTION_ENQUEUE.equals(action)) {
                enqueue(queue, apiRequest);
            } else {
                throw new Exception("Unrecognized action " + action + ".");
            }
        } finally {
            if (queue != null) {
                queue.close();
            }
        }

        return response;
    }

    private void doTopicAction(MQQueueManager queueManager, String topicString, String action, MqRequest apiRequest)
            throws Exception {
        if ("".equals(StringUtils.trimToEmpty(topicString))) {
            throw new Exception("Topic string not specified.");
        }

        MQTopic publisher = null;
        try {
            //  open topic for publish
            publisher = queueManager.accessTopic(topicString,null, CMQC.MQTOPIC_OPEN_AS_PUBLICATION,
                    CMQC.MQOO_OUTPUT);

            if (Teststep.ACTION_PUBLISH.equals(action)) {
                MQMessage message = buildMessage(apiRequest);
                publisher.put(message);
            } else {
                throw new Exception("Unrecognized action " + action + ".");
            }
        } finally {
            if (publisher != null) {
                publisher.close();
            }
        }
    }

    private MQMessage buildMessage(MqRequest apiRequest) throws Exception {
        MQMessage message;
        if (apiRequest instanceof MqEnqueueOrPublishFromTextRequest) {
            MqEnqueueOrPublishFromTextRequest request = (MqEnqueueOrPublishFromTextRequest) apiRequest;
            message = buildMessageFromText(request.getBody(), request.getRfh2Header());
        } else {
            MqEnqueueOrPublishFromFileRequest request = (MqEnqueueOrPublishFromFileRequest) apiRequest;
            message = buildMessageFromFile(request.getFileContent());
        }
        return message;
    }

    private void enqueue(MQQueue queue, MqRequest apiRequest) throws Exception {
        MQMessage message = buildMessage(apiRequest);
        MQPutMessageOptions pmo = new MQPutMessageOptions();
        queue.put(message, pmo);
    }

    private MQMessage buildMessageFromText(String body, MqRfh2Header rfh2Header)
            throws IOException, MQDataException {
        if (body == null) {
            throw new IllegalArgumentException("Message body can not be null.");
        }

        MQMessage message = new MQMessage();

        //  create MQMD properties on the message object (MQMD is not written into message, but is used by MQ PUT)
        MQMD mqmd = new MQMD();
        message.putDateTime = new GregorianCalendar();
        mqmd.setEncoding(CMQC.MQENC_REVERSED);

        if (rfh2Header == null) {
            mqmd.copyTo(message);
        } else {   //  add RFH2 header if included
            mqmd.setFormat(CMQC.MQFMT_RF_HEADER_2);
            mqmd.setCodedCharSetId(CMQC.MQCCSI_DEFAULT);
            mqmd.setPersistence(CMQC.MQPER_PERSISTENT);
            mqmd.copyTo(message);

            //  populate RFH2 header
            MQRFH2 mqrfh2 = new MQRFH2();
            mqrfh2.setFolderStrings(rfh2Header.getFolderStrings());
            mqrfh2.write(message);
        }

        //  populate message body
        message.writeString(body);

        return message;
    }

    private MQMessage buildMessageFromFile(byte[] bytes) throws MQDataException, IOException {
        if (bytes == null) {
            throw new IllegalArgumentException("File can not be null.");
        }

        MQMessage message = new MQMessage();
        MQMD mqmdHeader;
        try {
            mqmdHeader = new MQMD(new DataInputStream(new ByteArrayInputStream(bytes)),
                    CMQC.MQENC_REVERSED, CMQC.MQCCSI_DEFAULT);
        } catch (Exception e) {
            LOGGER.info("Not able to construct MQMD out of the bytes. Exception details: ", e);
            mqmdHeader = null;
        }
        if (mqmdHeader != null && CMQC.MQMD_STRUC_ID.equals(mqmdHeader.getStrucId()) &&
                (CMQC.MQMD_VERSION_1 == mqmdHeader.getVersion() || CMQC.MQMD_VERSION_2 == mqmdHeader.getVersion())) {
            LOGGER.info("MQMD constructed. Writing other bytes as application data.");
            message.putDateTime = new GregorianCalendar();
            mqmdHeader.copyTo(message);
            message.persistence = CMQC.MQPER_PERSISTENT;
            message.write(bytes, MQMD.SIZE2, bytes.length - MQMD.SIZE2);
        } else {
            LOGGER.info("No valid MQMD. Writing all bytes as application data.");
            message.write(bytes);
        }
        return message;
    }

    private MqDequeueResponse dequeue(MQQueue queue) throws MQException, IOException, MQDataException {
        MqDequeueResponse result = null;
        MQGetMessageOptions getOptions = new MQGetMessageOptions();
        //  The MQGMO_PROPERTIES_FORCE_MQRFH2 is to enforce message properties to be returned in the MQRFH2 headers.
        //  This is so that user can see message properties with names reserved by MQRFH2 folders (like <mqps> or <usr>) as MQRFH2 folders.
        getOptions.options = CMQC.MQGMO_NO_WAIT + CMQC.MQGMO_FAIL_IF_QUIESCING + CMQC.MQGMO_PROPERTIES_FORCE_MQRFH2;
        MQMessage message = new MQMessage();
        try {
            queue.get(message, getOptions);

            //  create the response object only when there is message returned from the queue
            //  when there is no message on the queue, keep the response object as null (i.e. there is no response)
            result = new MqDequeueResponse();

            //  parse the MQMessage to API Test Base model
            MqRfh2Header mqrfh2Header = null;
            MQHeaderIterator it = new MQHeaderIterator(message);
            while (it.hasNext()) {
                MQHeader header = it.nextHeader();
                if (header instanceof MQRFH2) {
                    mqrfh2Header = new MqRfh2Header();
                    MQRFH2 mqrfh2 = (MQRFH2) header;
                    String[] folderStrings = mqrfh2.getFolderStrings();
                    for (int i = 0; i < folderStrings.length; i++) {
                        MqRfh2Folder mqrfh2Folder = new MqRfh2Folder();
                        mqrfh2Folder.setString(folderStrings[i]);
                        GeneralUtils.validateMQRFH2FolderStringAndSetFolderName(mqrfh2Folder);
                        mqrfh2Header.getFolders().add(mqrfh2Folder);
                    }
                }
            }
            result.setMqrfh2Header(mqrfh2Header);
            result.setBodyAsText(it.getBodyAsText());
        } catch(MQException mqEx) {
            if (mqEx.getCompCode() == CMQC.MQCC_FAILED && mqEx.getReason() == CMQC.MQRC_NO_MSG_AVAILABLE) {
                //  No more message available on the queue
            } else {
                throw mqEx;
            }
        }
        return result;
    }

    private void clearQueue(MQQueue queue) throws MQException {
        MQGetMessageOptions getOptions = new MQGetMessageOptions();
        getOptions.options = CMQC.MQGMO_NO_WAIT + CMQC.MQGMO_FAIL_IF_QUIESCING;
        while (true) {
            //  read message from queue
            MQMessage message = new MQMessage();
            try {
                queue.get(message, getOptions);
            } catch(MQException mqEx) {
                if (mqEx.getCompCode() == CMQC.MQCC_FAILED && mqEx.getReason() == CMQC.MQRC_NO_MSG_AVAILABLE) {
                    //  No more message available on the queue
                    break;
                } else {
                    throw mqEx;
                }
            }
        }
    }
}