/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.appium.java_client.chromium;

import io.appium.java_client.AppiumClientConfig;
import io.appium.java_client.AppiumDriver;
import io.appium.java_client.remote.AutomationName;
import io.appium.java_client.service.local.AppiumDriverLocalService;
import io.appium.java_client.service.local.AppiumServiceBuilder;
import org.openqa.selenium.Capabilities;
import org.openqa.selenium.remote.HttpCommandExecutor;
import org.openqa.selenium.remote.http.ClientConfig;
import org.openqa.selenium.remote.http.HttpClient;

import java.net.URL;

/**
 * ChromiumDriver is an officially supported Appium driver created to automate Mobile browsers
 * and web views based on the Chromium engine. The driver uses W3CWebDriver protocol and is built
 * on top of chromium driver server.
 * <br>
 *
 * <p>Read <a href='https://github.com/appium/appium-chromium-driver'>appium-chromium-driver</a>
 * for more details on how to configure and use it.</p>
 */
public class ChromiumDriver extends AppiumDriver {
    private static final String AUTOMATION_NAME = AutomationName.CHROMIUM;

    public ChromiumDriver(HttpCommandExecutor executor, Capabilities capabilities) {
        super(executor, ensureAutomationName(capabilities, AUTOMATION_NAME));
    }

    public ChromiumDriver(URL remoteAddress, Capabilities capabilities) {
        super(remoteAddress, ensureAutomationName(capabilities, AUTOMATION_NAME));
    }

    public ChromiumDriver(URL remoteAddress, HttpClient.Factory httpClientFactory, Capabilities capabilities) {
        super(remoteAddress, httpClientFactory, ensureAutomationName(capabilities, AUTOMATION_NAME));
    }

    public ChromiumDriver(AppiumDriverLocalService service, Capabilities capabilities) {
        super(service, ensureAutomationName(capabilities, AUTOMATION_NAME));
    }

    public ChromiumDriver(AppiumDriverLocalService service, HttpClient.Factory httpClientFactory,
                          Capabilities capabilities) {
        super(service, httpClientFactory, ensureAutomationName(capabilities, AUTOMATION_NAME));
    }

    public ChromiumDriver(AppiumServiceBuilder builder, Capabilities capabilities) {
        super(builder, ensureAutomationName(capabilities, AUTOMATION_NAME));
    }

    public ChromiumDriver(AppiumServiceBuilder builder, HttpClient.Factory httpClientFactory,
                          Capabilities capabilities) {
        super(builder, httpClientFactory, ensureAutomationName(capabilities, AUTOMATION_NAME));
    }

    public ChromiumDriver(HttpClient.Factory httpClientFactory, Capabilities capabilities) {
        super(httpClientFactory, ensureAutomationName(capabilities, AUTOMATION_NAME));
    }

    /**
     * This is a special constructor used to connect to a running driver instance.
     * It does not do any necessary verifications, but rather assumes the given
     * driver session is already running at `remoteSessionAddress`.
     * The maintenance of driver state(s) is the caller's responsibility.
     * !!! This API is supposed to be used for **debugging purposes only**.
     *
     * @param remoteSessionAddress The address of the **running** session including the session identifier.
     * @param platformName The name of the target platform.
     */
    public ChromiumDriver(URL remoteSessionAddress, String platformName) {
        super(remoteSessionAddress, platformName, AUTOMATION_NAME);
    }

    /**
     * Creates a new instance based on the given ClientConfig and {@code capabilities}.
     * The HTTP client is default client generated by {@link HttpCommandExecutor#getDefaultClientFactory}.
     * For example:
     *
     * <pre>
     *
     * ClientConfig clientConfig = ClientConfig.defaultConfig()
     *     .baseUri(URI.create("WebDriver URL"))
     *     .readTimeout(Duration.ofMinutes(5));
     * ChromiumOptions options = new ChromiumOptions();
     * ChromiumDriver driver = new ChromiumDriver(clientConfig, options);
     *
     * </pre>
     *
     * @param clientConfig take a look at {@link ClientConfig}
     * @param capabilities take a look at {@link Capabilities}
     *
     */
    public ChromiumDriver(ClientConfig clientConfig, Capabilities capabilities) {
        super(AppiumClientConfig.fromClientConfig(clientConfig), ensureAutomationName(capabilities, AUTOMATION_NAME));
    }

    /**
     * Creates a new instance based on the given ClientConfig and {@code capabilities}.
     * The HTTP client is default client generated by {@link HttpCommandExecutor#getDefaultClientFactory}.
     * For example:
     *
     * <pre>
     *
     * AppiumClientConfig appiumClientConfig = AppiumClientConfig.defaultConfig()
     *     .directConnect(true)
     *     .baseUri(URI.create("WebDriver URL"))
     *     .readTimeout(Duration.ofMinutes(5));
     * ChromiumOptions options = new ChromiumOptions();
     * ChromiumDriver driver = new ChromiumDriver(options, appiumClientConfig);
     *
     * </pre>
     *
     * @param appiumClientConfig take a look at {@link AppiumClientConfig}
     * @param capabilities take a look at {@link Capabilities}
     *
     */
    public ChromiumDriver(AppiumClientConfig appiumClientConfig, Capabilities capabilities) {
        super(appiumClientConfig, ensureAutomationName(capabilities, AUTOMATION_NAME));
    }

    public ChromiumDriver(Capabilities capabilities) {
        super(ensureAutomationName(capabilities, AUTOMATION_NAME));
    }
}
