/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.appium.java_client.remote;

import io.appium.java_client.ExecutesMethod;
import io.appium.java_client.MobileCommand;
import org.openqa.selenium.DeviceRotation;
import org.openqa.selenium.ScreenOrientation;
import org.openqa.selenium.WebDriver;
import org.openqa.selenium.remote.Response;

import java.util.Map;

public interface SupportsRotation extends WebDriver, ExecutesMethod {
    /**
     * Get device rotation.
     *
     * @return The rotation value.
     */
    default DeviceRotation rotation() {
        Response response = execute(MobileCommand.GET_SCREEN_ROTATION);
        //noinspection unchecked
        return new DeviceRotation((Map<String, Number>) response.getValue());
    }

    default void rotate(DeviceRotation rotation) {
        execute(MobileCommand.SET_SCREEN_ROTATION, rotation.parameters());
    }

    default void rotate(ScreenOrientation orientation) {
        execute(MobileCommand.SET_SCREEN_ORIENTATION,
                Map.of("orientation", orientation.value().toUpperCase()));
    }

    /**
     * Get device orientation.
     *
     * @return The orientation value.
     */
    default ScreenOrientation getOrientation() {
        Response response = execute(MobileCommand.GET_SCREEN_ORIENTATION);
        String orientation = String.valueOf(response.getValue());
        return ScreenOrientation.valueOf(orientation.toUpperCase());
    }
}
