package arrow.core.extensions.`try`.monadThrow

import arrow.core.ForTry
import arrow.core.Try
import arrow.core.Try.Companion
import arrow.core.extensions.TryMonadThrow
import arrow.typeclasses.MonadContinuation
import arrow.typeclasses.MonadErrorContinuation
import kotlin.Suppress
import kotlin.Throwable
import kotlin.jvm.JvmName

/**
 * Entry point for monad bindings which enables for comprehensions. The underlying implementation is based on
 * coroutines. A coroutine is initiated and suspended inside [MonadErrorContinuation] yielding to [Monad.flatMap].
 * Once all the flatMap binds are completed, the underlying monad is returned from the act of executing the coroutine.
 *
 * This one operates over [MonadError] instances that can support [Throwable] in their error type automatically
 * lifting errors as failed computations in their monadic context and not letting exceptions thrown as the regular
 * monad binding does.
 *
 * ### Example
 *
 * Oftentimes we find ourselves in situations where we need to sequence some computations that could potentially fail.
 * [bindingCatch] allows us to safely compute those by automatically catching any exceptions thrown during the process.
 *
 * ```kotlin:ank:playground
 * import arrow.core.*
 * import arrow.core.extensions.`try`.monadThrow.*
 * import arrow.core.*
 *
 *
 * import arrow.Kind
 * import arrow.typeclasses.MonadThrow
 *
 * typealias Impacted = Boolean
 *
 * object Nuke
 * object Target
 * class MissedByMeters(private val meters: Int) : Throwable("Missed by $meters meters")
 *
 * fun <F> MonadThrow<F>.arm(): Kind<F, Nuke> = just(Nuke)
 * fun <F> MonadThrow<F>.aim(): Kind<F, Target> = just(Target)
 * fun <F> MonadThrow<F>.launchImpure(target: Target, nuke: Nuke): Impacted {
 *   throw MissedByMeters(5)
 * }
 *
 * fun main(args: Array<String>) {
 *    //sampleStart
 *    fun <F> MonadThrow<F>.attack(): Kind<F, Impacted> =
 *      bindingCatch {
 *        val nuke = arm().bind()
 *        val target = aim().bind()
 *        val impact = launchImpure(target, nuke) // this throws!
 *        impact
 *      }
 *
 *    val result = Try.monadThrow().attack()
 *    //sampleEnd
 *    println(result)
 * }
 * ```
 *
 */
@JvmName("bindingCatch")
@Suppress(
        "UNCHECKED_CAST",
        "USELESS_CAST",
        "EXTENSION_SHADOWED_BY_MEMBER",
        "UNUSED_PARAMETER"
)
fun <B> bindingCatch(arg0: suspend MonadErrorContinuation<ForTry, *>.() -> B): Try<B> = arrow.core.Try
   .monadThrow()
   .bindingCatch<B>(arg0) as arrow.core.Try<B>

@JvmName("binding")
@Suppress(
        "UNCHECKED_CAST",
        "USELESS_CAST",
        "EXTENSION_SHADOWED_BY_MEMBER",
        "UNUSED_PARAMETER"
)
fun <B> binding(arg0: suspend MonadContinuation<ForTry, *>.() -> B): Try<B> = arrow.core.Try
   .monadThrow()
   .binding<B>(arg0) as arrow.core.Try<B>

@JvmName("raiseNonFatal")
@Suppress(
        "UNCHECKED_CAST",
        "USELESS_CAST",
        "EXTENSION_SHADOWED_BY_MEMBER",
        "UNUSED_PARAMETER"
)
fun <A> Throwable.raiseNonFatal(): Try<A> = arrow.core.Try.monadThrow().run {
  this@raiseNonFatal.raiseNonFatal<A>() as arrow.core.Try<A>
}

/**
 * ank_macro_hierarchy(arrow.typeclasses.MonadThrow)
 *
 * A MonadError with the error type fixed to Throwable. It provides [bindingCatch] for automatically catching throwable
 * errors in the context of a binding, short-circuiting the complete computation and returning the error raised to the
 * same computational context (through [raiseError]).
 *
 * ```kotlin:ank:playground
 * import arrow.core.*
 * import arrow.core.extensions.`try`.monadThrow.*
 * import arrow.core.*
 *
 *
 *
 * fun main(args: Array<String>) {
 *   val result =
 *   //sampleStart
 *   Try.monadThrow()
 *   //sampleEnd
 *   println(result)
 * }
 * ```
 *
 * ### Example
 *
 * Oftentimes we find ourselves in situations where we need to sequence some computations that could potentially fail.
 * [bindingCatch] allows us to safely compute those by automatically catching any exceptions thrown during the process.
 *
 * ```kotlin:ank:playground
 * import arrow.core.*
 * import arrow.core.extensions.`try`.monadThrow.*
 * import arrow.core.*
 *
 *
 * import arrow.Kind
 * import arrow.typeclasses.MonadThrow
 *
 * typealias Impacted = Boolean
 *
 * object Nuke
 * object Target
 * class MissedByMeters(private val meters: Int) : Throwable("Missed by $meters meters")
 *
 * fun <F> MonadThrow<F>.arm(): Kind<F, Nuke> = just(Nuke)
 * fun <F> MonadThrow<F>.aim(): Kind<F, Target> = just(Target)
 * fun <F> MonadThrow<F>.launchImpure(target: Target, nuke: Nuke): Impacted {
 *   throw MissedByMeters(5)
 * }
 *
 * fun main(args: Array<String>) {
 *    //sampleStart
 *    fun <F> MonadThrow<F>.attack(): Kind<F, Impacted> =
 *      bindingCatch {
 *        val nuke = arm().bind()
 *        val target = aim().bind()
 *        val impact = launchImpure(target, nuke) // this throws!
 *        impact
 *      }
 *
 *    val result = Try.monadThrow().attack()
 *    //sampleEnd
 *    println(result)
 * }
 * ```
 */
fun Companion.monadThrow(): TryMonadThrow = object : arrow.core.extensions.TryMonadThrow {  }