package arrow.effects.extensions.io.async

import arrow.Kind
import arrow.core.Either
import arrow.effects.ForIO
import arrow.effects.IO
import arrow.effects.IO.Companion
import arrow.effects.extensions.IOAsync
import kotlin.Function0
import kotlin.Function1
import kotlin.Suppress
import kotlin.Throwable
import kotlin.Unit
import kotlin.coroutines.CoroutineContext
import kotlin.jvm.JvmName

/**
 * [async] variant that can suspend side effects in the provided registration function.
 *
 * The passed in function is injected with a side-effectful callback for signaling the final result of an asynchronous process.
 *
 * ```kotlin:ank:playground
 * import arrow.effects.*
 * import arrow.effects.extensions.io.async.*
 * import arrow.core.*
 *
 *
 * import arrow.effects.*
 * import arrow.effects.typeclasses.Async
 *
 * fun main(args: Array<String>) {
 *   //sampleStart
 *   fun <F> Async<F>.makeCompleteAndGetPromiseInAsync() =
 *     asyncF<String> { cb: (Either<Throwable, String>) -> Unit ->
 *       Promise.uncancelable<F, String>(this).flatMap { promise ->
 *         promise.complete("Hello World!").flatMap {
 *           promise.get().map { str -> cb(Right(str)) }
 *         }
 *       }
 *     }
 *
 *   val result = IO.async().makeCompleteAndGetPromiseInAsync()
 *  //sampleEnd
 *  println(result)
 * }
 * ```
 *
 * @see async for a simpler, non suspending version.
 */
@JvmName("asyncF")
@Suppress(
        "UNCHECKED_CAST",
        "USELESS_CAST",
        "EXTENSION_SHADOWED_BY_MEMBER",
        "UNUSED_PARAMETER"
)
fun <A> asyncF(arg0: Function1<Function1<Either<Throwable, A>, Unit>, Kind<ForIO, Unit>>): IO<A> = arrow.effects.IO
   .async()
   .asyncF<A>(arg0) as arrow.effects.IO<A>

/**
 * Continue the evaluation on provided [CoroutineContext]
 *
 * @param ctx [CoroutineContext] to run evaluation on
 *
 * ```kotlin:ank:playground
 * import arrow.effects.*
 * import arrow.effects.extensions.io.async.*
 * import arrow.core.*
 *
 *
 * import kotlinx.coroutines.Dispatchers
 *
 * fun main(args: Array<String>) {
 *   //sampleStart
 *   fun <F> Async<F>.runOnDefaultDispatcher(): Kind<F, String> =
 *     just(Unit).continueOn(Dispatchers.Default).flatMap {
 *       delay({ Thread.currentThread().name })
 *     }
 *
 *   val result = IO.async().runOnDefaultDispatcher()
 *   //sampleEnd
 *   println(result)
 * }
 * ```
 */
@JvmName("continueOn")
@Suppress(
        "UNCHECKED_CAST",
        "USELESS_CAST",
        "EXTENSION_SHADOWED_BY_MEMBER",
        "UNUSED_PARAMETER"
)
fun <A> Kind<ForIO, A>.continueOn(arg1: CoroutineContext): IO<A> = arrow.effects.IO.async().run {
  this@continueOn.continueOn<A>(arg1) as arrow.effects.IO<A>
}

/**
 * Delay a computation on provided [CoroutineContext].
 *
 * @param ctx [CoroutineContext] to run evaluation on.
 *
 * ```kotlin:ank:playground
 * import arrow.effects.*
 * import arrow.effects.extensions.io.async.*
 * import arrow.core.*
 *
 *
 * import kotlinx.coroutines.Dispatchers
 *
 * fun main(args: Array<String>) {
 *   //sampleStart
 *   fun <F> Async<F>.invokeOnDefaultDispatcher(): Kind<F, String> =
 *     delay(Dispatchers.Default, { Thread.currentThread().name })
 *
 *   val result = IO.async().invokeOnDefaultDispatcher()
 *   //sampleEnd
 *   println(result)
 * }
 * ```
 */
@JvmName("delay")
@Suppress(
        "UNCHECKED_CAST",
        "USELESS_CAST",
        "EXTENSION_SHADOWED_BY_MEMBER",
        "UNUSED_PARAMETER"
)
fun <A> delay(arg0: CoroutineContext, arg1: Function0<A>): IO<A> = arrow.effects.IO
   .async()
   .delay<A>(arg0, arg1) as arrow.effects.IO<A>

@JvmName("invoke")
@Suppress(
        "UNCHECKED_CAST",
        "USELESS_CAST",
        "EXTENSION_SHADOWED_BY_MEMBER",
        "UNUSED_PARAMETER"
)
fun <A> invoke(arg0: CoroutineContext, arg1: Function0<A>): IO<A> = arrow.effects.IO
   .async()
   .invoke<A>(arg0, arg1) as arrow.effects.IO<A>

/**
 * Delay a computation on provided [CoroutineContext].
 *
 * @param ctx [CoroutineContext] to run evaluation on.
 *
 * ```kotlin:ank:playground
 * import arrow.effects.*
 * import arrow.effects.extensions.io.async.*
 * import arrow.core.*
 *
 *
 * import kotlinx.coroutines.Dispatchers
 *
 * fun main(args: Array<String>) {
 *   //sampleStart
 *   fun <F> Async<F>.invokeOnDefaultDispatcher(): Kind<F, String> =
 *     defer(Dispatchers.Default, { delay { Thread.currentThread().name } })
 *
 *   val result = IO.async().invokeOnDefaultDispatcher().fix().unsafeRunSync()
 *   //sampleEnd
 *   println(result)
 * }
 * ```
 */
@JvmName("defer")
@Suppress(
        "UNCHECKED_CAST",
        "USELESS_CAST",
        "EXTENSION_SHADOWED_BY_MEMBER",
        "UNUSED_PARAMETER"
)
fun <A> defer(arg0: CoroutineContext, arg1: Function0<Kind<ForIO, A>>): IO<A> = arrow.effects.IO
   .async()
   .defer<A>(arg0, arg1) as arrow.effects.IO<A>

/**
 * Shift evaluation to provided [CoroutineContext].
 *
 * @receiver [CoroutineContext] to run evaluation on.
 *
 * ```kotlin:ank:playground
 * import arrow.effects.*
 * import arrow.effects.extensions.io.async.*
 * import arrow.core.*
 *
 *
 * import kotlinx.coroutines.Dispatchers
 *
 * fun main(args: Array<String>) {
 *   //sampleStart
 *   IO.async().run {
 *     val result = Dispatchers.Default.shift().map {
 *       Thread.currentThread().name
 *     }
 *
 *     println(result)
 *   }
 *   //sampleEnd
 * }
 * ```
 */
@JvmName("shift")
@Suppress(
        "UNCHECKED_CAST",
        "USELESS_CAST",
        "EXTENSION_SHADOWED_BY_MEMBER",
        "UNUSED_PARAMETER"
)
fun CoroutineContext.shift(): IO<Unit> = arrow.effects.IO.async().run {
  this@shift.shift() as arrow.effects.IO<kotlin.Unit>
}

/**
 * Task that never finishes evaluating.
 *
 * ```kotlin:ank:playground
 * import arrow.effects.*
 * import arrow.effects.extensions.io.async.*
 * import arrow.core.*
 *
 *
 *
 * fun main(args: Array<String>) {
 *   //sampleStart
 *   val i = IO.async().never<Int>()
 *
 *   println(i)
 *   //sampleEnd
 * }
 * ```
 */
@JvmName("never")
@Suppress(
        "UNCHECKED_CAST",
        "USELESS_CAST",
        "EXTENSION_SHADOWED_BY_MEMBER",
        "UNUSED_PARAMETER"
)
fun <A> never(): IO<A> = arrow.effects.IO
   .async()
   .never<A>() as arrow.effects.IO<A>

/**
 * ank_macro_hierarchy(arrow.effects.typeclasses.Async)
 *
 * [Async] models how a data type runs an asynchronous computation that may fail.
 * Defined by the [Proc] signature, which is the consumption of a callback.
 */
fun Companion.async(): IOAsync = object : arrow.effects.extensions.IOAsync {  }