package arrow.effects.extensions.io.bracket

import arrow.Kind
import arrow.effects.ForIO
import arrow.effects.IO
import arrow.effects.IO.Companion
import arrow.effects.extensions.IOBracket
import arrow.effects.typeclasses.ExitCase
import kotlin.Function1
import kotlin.Function2
import kotlin.Suppress
import kotlin.Throwable
import kotlin.Unit
import kotlin.jvm.JvmName

/**
 * A way to safely acquire a resource and release in the face of errors and cancellation.
 * It uses [ExitCase] to distinguish between different exit cases when releasing the acquired resource.
 *
 * @param use is the action to consume the resource and produce an [F] with the result.
 * Once the resulting [F] terminates, either successfully, error or cancelled.
 *
 * @param release the allocated resource after the resulting [F] of [use] is terminates.
 *
 * ```kotlin:ank:playground
 * import arrow.effects.*
 * import arrow.effects.extensions.io.bracket.*
 * import arrow.core.*
 *
 *
 * import arrow.effects.extensions.io.monadDefer.defer
 * import arrow.effects.extensions.io.monadDefer.delay
 *
 * class File(url: String) {
 *   fun open(): File = this
 *   fun close(): Unit {}
 *   override fun toString(): String = "This file contains some interesting content!"
 * }
 *
 * fun openFile(uri: String): Kind<F, File> = delay({ File(uri).open() })
 * fun closeFile(file: File): Kind<F, Unit> = delay({ file.close() })
 * fun fileToString(file: File): Kind<F, String> = delay({ file.toString() })
 *
 * fun main(args: Array<String>) {
 *   //sampleStart
 *   val release: (File, ExitCase<Throwable>) -> Kind<F, Unit> = { file, exitCase ->
 *       when (exitCase) {
 *         is ExitCase.Completed -> { / * do something * / }
 *         is ExitCase.Canceled -> { / * do something * / }
 *         is ExitCase.Error -> { / * do something * / }
 *       }
 *       closeFile(file)
 *   }
 *
 *   val use: (File) -> Kind<F, String> = { file: File -> fileToString(file) }
 *
 *   val safeComputation = openFile("data.json").bracketCase(release, use)
 *   //sampleEnd
 *   println(safeComputation)
 * }
 *  ```
 */
@JvmName("bracketCase")
@Suppress(
        "UNCHECKED_CAST",
        "USELESS_CAST",
        "EXTENSION_SHADOWED_BY_MEMBER",
        "UNUSED_PARAMETER"
)
fun <A, B> Kind<ForIO, A>.bracketCase(arg1: Function2<A, ExitCase<Throwable>, Kind<ForIO, Unit>>, arg2: Function1<A, Kind<ForIO, B>>): IO<B> = arrow.effects.IO.bracket().run {
  this@bracketCase.bracketCase<A, B>(arg1, arg2) as arrow.effects.IO<B>
}

/**
 * Meant for specifying tasks with safe resource acquisition and release in the face of errors and interruption.
 * It would be the the equivalent of `try/catch/finally` statements in mainstream imperative languages for resource
 * acquisition and release.
 *
 * @param release is the action that's supposed to release the allocated resource after `use` is done, irregardless
 * of its exit condition.
 *
 * ```kotlin:ank:playground
 * import arrow.effects.*
 * import arrow.effects.extensions.io.bracket.*
 * import arrow.core.*
 *
 *
 * import arrow.effects.extensions.io.monadDefer.defer
 * import arrow.effects.extensions.io.monadDefer.delay
 *
 * class File(url: String) {
 *   fun open(): File = this
 *   fun close(): Unit {}
 *   override fun toString(): String = "This file contains some interesting content!"
 * }
 *
 * fun openFile(uri: String): Kind<F, File> = delay({ File(uri).open() })
 * fun closeFile(file: File): Kind<F, Unit> = delay({ file.close() })
 * fun fileToString(file: File): Kind<F, String> = delay({ file.toString() })
 *
 * fun main(args: Array<String>) {
 *   //sampleStart
 *   val safeComputation = openFile("data.json").bracket({ file: File -> closeFile(file) }, { file -> fileToString(file) })
 *   //sampleEnd
 *   println(safeComputation)
 * }
 * ```
 */
@JvmName("bracket")
@Suppress(
        "UNCHECKED_CAST",
        "USELESS_CAST",
        "EXTENSION_SHADOWED_BY_MEMBER",
        "UNUSED_PARAMETER"
)
fun <A, B> Kind<ForIO, A>.bracket(arg1: Function1<A, Kind<ForIO, Unit>>, arg2: Function1<A, Kind<ForIO, B>>): IO<B> = arrow.effects.IO.bracket().run {
  this@bracket.bracket<A, B>(arg1, arg2) as arrow.effects.IO<B>
}

/**
 * Meant for ensuring a given task continues execution even when interrupted.
 */
@JvmName("uncancelable")
@Suppress(
        "UNCHECKED_CAST",
        "USELESS_CAST",
        "EXTENSION_SHADOWED_BY_MEMBER",
        "UNUSED_PARAMETER"
)
fun <A> Kind<ForIO, A>.uncancelable(): IO<A> = arrow.effects.IO.bracket().run {
  this@uncancelable.uncancelable<A>() as arrow.effects.IO<A>
}

/**
 * Executes the given `finalizer` when the source is finished, either in success or in error, or if canceled.
 *
 * As best practice, it's not a good idea to release resources via `guaranteeCase` in polymorphic code.
 * Prefer [bracket] for the acquisition and release of resources.
 *
 * @see [guaranteeCase] for the version that can discriminate between termination conditions
 *
 * @see [bracket] for the more general operation
 */
@JvmName("guarantee")
@Suppress(
        "UNCHECKED_CAST",
        "USELESS_CAST",
        "EXTENSION_SHADOWED_BY_MEMBER",
        "UNUSED_PARAMETER"
)
fun <A> Kind<ForIO, A>.guarantee(arg1: Kind<ForIO, Unit>): IO<A> = arrow.effects.IO.bracket().run {
  this@guarantee.guarantee<A>(arg1) as arrow.effects.IO<A>
}

/**
 * Executes the given `finalizer` when the source is finished, either in success or in error, or if canceled, allowing
 * for differentiating between exit conditions. That's thanks to the [ExitCase] argument of the finalizer.
 *
 * As best practice, it's not a good idea to release resources via `guaranteeCase` in polymorphic code.
 * Prefer [bracketCase] for the acquisition and release of resources.
 *
 * @see [guarantee] for the simpler version
 *
 * @see [bracketCase] for the more general operation
 *
 */
@JvmName("guaranteeCase")
@Suppress(
        "UNCHECKED_CAST",
        "USELESS_CAST",
        "EXTENSION_SHADOWED_BY_MEMBER",
        "UNUSED_PARAMETER"
)
fun <A> Kind<ForIO, A>.guaranteeCase(arg1: Function1<ExitCase<Throwable>, Kind<ForIO, Unit>>): IO<A> = arrow.effects.IO.bracket().run {
  this@guaranteeCase.guaranteeCase<A>(arg1) as arrow.effects.IO<A>
}

/**
 * ank_macro_hierarchy(arrow.effects.typeclasses.Bracket)
 *
 * Extension of MonadError exposing the [bracket] operation, a generalized abstracted pattern of safe resource
 * acquisition and release in the face of errors or interruption.
 *
 * @define The functions receiver here (Kind<F, A>) would stand for the "acquireParam", and stands for an action that
 * "acquires" some expensive resource, that needs to be used and then discarded.
 *
 * @define use is the action that uses the newly allocated resource and that will provide the final result.
 */
fun Companion.bracket(): IOBracket = object : arrow.effects.extensions.IOBracket {  }