package arrow.data.extensions.sequencek.functor

import arrow.Kind
import arrow.core.Tuple2
import arrow.data.ForSequenceK
import arrow.data.SequenceK
import arrow.data.SequenceK.Companion
import arrow.data.extensions.SequenceKFunctor
import kotlin.Function1
import kotlin.Suppress
import kotlin.Unit
import kotlin.jvm.JvmName

/**
 * Transform the [F] wrapped value [A] into [B] preserving the [F] structure
 * Kind<F, A> -> Kind<F, B>
 *
 * ```kotlin:ank:playground
 * import arrow.data.*
 * import arrow.data.extensions.sequencek.functor.*
 * import arrow.core.*
 *
 *
 * import arrow.data.extensions.sequencek.applicative.just
 *
 * fun main(args: Array<String>) {
 *   val result =
 *   //sampleStart
 *   "Hello".just().map({ "$it World" })
 *   //sampleEnd
 *   println(result)
 * }
 * ```
 */
@JvmName("map")
@Suppress(
        "UNCHECKED_CAST",
        "USELESS_CAST",
        "EXTENSION_SHADOWED_BY_MEMBER",
        "UNUSED_PARAMETER"
)
fun <A, B> Kind<ForSequenceK, A>.map(arg1: Function1<A, B>): SequenceK<B> = arrow.data.SequenceK.functor().run {
  this@map.map<A, B>(arg1) as arrow.data.SequenceK<B>
}

@JvmName("imap")
@Suppress(
        "UNCHECKED_CAST",
        "USELESS_CAST",
        "EXTENSION_SHADOWED_BY_MEMBER",
        "UNUSED_PARAMETER"
)
fun <A, B> Kind<ForSequenceK, A>.imap(arg1: Function1<A, B>, arg2: Function1<B, A>): SequenceK<B> = arrow.data.SequenceK.functor().run {
  this@imap.imap<A, B>(arg1, arg2) as arrow.data.SequenceK<B>
}

/**
 * Lifts a function `A -> B` to the [F] structure returning a polymorphic function
 * that can be applied over all [F] values in the shape of Kind<F, A>
 *
 * `A -> B -> Kind<F, A> -> Kind<F, B>`
 *
 * ```kotlin:ank:playground
 * import arrow.data.*
 * import arrow.data.extensions.sequencek.functor.*
 * import arrow.core.*
 *
 *
 * import arrow.data.extensions.sequencek.applicative.just
 *
 * fun main(args: Array<String>) {
 *   val result =
 *   //sampleStart
 *   lift({ s: CharSequence -> "$s World" })("Hello".just())
 *   //sampleEnd
 *   println(result)
 * }
 * ```
 */
@JvmName("lift")
@Suppress(
        "UNCHECKED_CAST",
        "USELESS_CAST",
        "EXTENSION_SHADOWED_BY_MEMBER",
        "UNUSED_PARAMETER"
)
fun <A, B> lift(arg0: Function1<A, B>): Function1<Kind<ForSequenceK, A>, Kind<ForSequenceK, B>> = arrow.data.SequenceK
   .functor()
   .lift<A, B>(arg0) as kotlin.Function1<arrow.Kind<arrow.data.ForSequenceK, A>, arrow.Kind<arrow.data.ForSequenceK, B>>

/**
 * Discards the [A] value inside [F] signaling this container may be pointing to a noop
 * or an effect whose return value is deliberately ignored. The singleton value [Unit] serves as signal.
 *
 * Kind<F, A> -> Kind<F, Unit>
 *
 * ```kotlin:ank:playground
 * import arrow.data.*
 * import arrow.data.extensions.sequencek.functor.*
 * import arrow.core.*
 *
 *
 * import arrow.data.extensions.sequencek.applicative.just
 *
 * fun main(args: Array<String>) {
 *   val result =
 *   //sampleStart
 *   "Hello World".just().unit()
 *   //sampleEnd
 *   println(result)
 * }
 * ```
 */
@JvmName("unit")
@Suppress(
        "UNCHECKED_CAST",
        "USELESS_CAST",
        "EXTENSION_SHADOWED_BY_MEMBER",
        "UNUSED_PARAMETER"
)
fun <A> Kind<ForSequenceK, A>.unit(): SequenceK<Unit> = arrow.data.SequenceK.functor().run {
  this@unit.unit<A>() as arrow.data.SequenceK<kotlin.Unit>
}

/**
 * Applies [f] to an [A] inside [F] and returns the [F] structure with a tuple of the [A] value and the
 * computed [B] value as result of applying [f]
 *
 * Kind<F, A> -> Kind<F, Tuple2<A, B>>
 *
 * ```kotlin:ank:playground
 * import arrow.data.*
 * import arrow.data.extensions.sequencek.functor.*
 * import arrow.core.*
 *
 *
 * import arrow.data.extensions.sequencek.applicative.just
 *
 * fun main(args: Array<String>) {
 *   val result =
 *   //sampleStart
 *   "Hello".just().fproduct({ "$it World" })
 *   //sampleEnd
 *   println(result)
 * }
 * ```
 */
@JvmName("fproduct")
@Suppress(
        "UNCHECKED_CAST",
        "USELESS_CAST",
        "EXTENSION_SHADOWED_BY_MEMBER",
        "UNUSED_PARAMETER"
)
fun <A, B> Kind<ForSequenceK, A>.fproduct(arg1: Function1<A, B>): SequenceK<Tuple2<A, B>> = arrow.data.SequenceK.functor().run {
  this@fproduct.fproduct<A, B>(arg1) as arrow.data.SequenceK<arrow.core.Tuple2<A, B>>
}

/**
 * Replaces [A] inside [F] with [B] resulting in a Kind<F, B>
 *
 * Kind<F, A> -> Kind<F, B>
 *
 * ```kotlin:ank:playground
 * import arrow.data.*
 * import arrow.data.extensions.sequencek.functor.*
 * import arrow.core.*
 *
 *
 * import arrow.data.extensions.sequencek.applicative.just
 *
 * fun main(args: Array<String>) {
 *   val result =
 *   //sampleStart
 *   "Hello World".just().`as`("...")
 *   //sampleEnd
 *   println(result)
 * }
 * ```
 */
@JvmName("as")
@Suppress(
        "UNCHECKED_CAST",
        "USELESS_CAST",
        "EXTENSION_SHADOWED_BY_MEMBER",
        "UNUSED_PARAMETER"
)
fun <A, B> Kind<ForSequenceK, A>.`as`(arg1: B): SequenceK<B> = arrow.data.SequenceK.functor().run {
  this@`as`.`as`<A, B>(arg1) as arrow.data.SequenceK<B>
}

/**
 * Pairs [B] with [A] returning a Kind<F, Tuple2<B, A>>
 *
 * Kind<F, A> -> Kind<F, Tuple2<B, A>>
 *
 * ```kotlin:ank:playground
 * import arrow.data.*
 * import arrow.data.extensions.sequencek.functor.*
 * import arrow.core.*
 *
 *
 * import arrow.data.extensions.sequencek.applicative.just
 *
 * fun main(args: Array<String>) {
 *   val result =
 *   //sampleStart
 *   "Hello".just().tupleLeft("World")
 *   //sampleEnd
 *   println(result)
 * }
 * ```
 */
@JvmName("tupleLeft")
@Suppress(
        "UNCHECKED_CAST",
        "USELESS_CAST",
        "EXTENSION_SHADOWED_BY_MEMBER",
        "UNUSED_PARAMETER"
)
fun <A, B> Kind<ForSequenceK, A>.tupleLeft(arg1: B): SequenceK<Tuple2<B, A>> = arrow.data.SequenceK.functor().run {
  this@tupleLeft.tupleLeft<A, B>(arg1) as arrow.data.SequenceK<arrow.core.Tuple2<B, A>>
}

/**
 * Pairs [A] with [B] returning a Kind<F, Tuple2<A, B>>
 *
 * Kind<F, A> -> Kind<F, Tuple2<A, B>>
 *
 * ```kotlin:ank:playground
 * import arrow.data.*
 * import arrow.data.extensions.sequencek.functor.*
 * import arrow.core.*
 *
 *
 * import arrow.data.extensions.sequencek.applicative.just
 *
 * fun main(args: Array<String>) {
 *   val result =
 *   //sampleStart
 *   "Hello".just().tupleRight("World")
 *   //sampleEnd
 *   println(result)
 * }
 * ```
 */
@JvmName("tupleRight")
@Suppress(
        "UNCHECKED_CAST",
        "USELESS_CAST",
        "EXTENSION_SHADOWED_BY_MEMBER",
        "UNUSED_PARAMETER"
)
fun <A, B> Kind<ForSequenceK, A>.tupleRight(arg1: B): SequenceK<Tuple2<A, B>> = arrow.data.SequenceK.functor().run {
  this@tupleRight.tupleRight<A, B>(arg1) as arrow.data.SequenceK<arrow.core.Tuple2<A, B>>
}

/**
 * Given [A] is a sub type of [B], re-type this value from Kind<F, A> to Kind<F, B>
 *
 * Kind<F, A> -> Kind<F, B>
 *
 * ```kotlin:ank:playground
 * import arrow.data.*
 * import arrow.data.extensions.sequencek.functor.*
 * import arrow.core.*
 *
 *
 * import arrow.data.extensions.sequencek.applicative.just
 * import arrow.Kind
 *
 * fun main(args: Array<String>) {
 *   val result: Kind<*, CharSequence> =
 *   //sampleStart
 *   "Hello".just().map({ "$it World" }).widen()
 *   //sampleEnd
 *   println(result)
 * }
 * ```
 */
@JvmName("widen")
@Suppress(
        "UNCHECKED_CAST",
        "USELESS_CAST",
        "EXTENSION_SHADOWED_BY_MEMBER",
        "UNUSED_PARAMETER"
)
fun <B, A : B> Kind<ForSequenceK, A>.widen(): SequenceK<B> = arrow.data.SequenceK.functor().run {
  this@widen.widen<B, A>() as arrow.data.SequenceK<B>
}

/**
 * ank_macro_hierarchy(arrow.typeclasses.Functor)
 *
 * The [Functor] type class abstracts the ability to [map] over the computational context of a type constructor.
 * Examples of type constructors that can implement instances of the Functor type class include [SequenceK],
 * [arrow.core.Option], [arrow.data.NonEmptyList], [List] and many other data types that include a [map] function with the shape
 * `fun <F, A, B> Kind<F, A>.map(f: (A) -> B): Kind<F, B>` where `F` refers to any type constructor whose contents can be transformed.
 *
 * ```kotlin:ank:playground
 * import arrow.data.*
 * import arrow.data.extensions.sequencek.functor.*
 * import arrow.core.*
 *
 *
 *
 * fun main(args: Array<String>) {
 *   val result =
 *   //sampleStart
 *   SequenceK.functor()
 *   //sampleEnd
 *   println(result)
 * }
 * ```
 *
 * ### Example
 *
 * Oftentimes we find ourselves in situations where we need to transform the contents of some data type.
 * [map] allows us to safely compute over values under the assumption that they'll be there returning the
 * transformation encapsulated in the same context.
 *
 * Consider both [arrow.core.Option] and [arrow.core.Try]:
 *
 * `Option<A>` allows us to model absence and has two possible states, `Some(a: A)` if the value is not absent and `None` to represent an empty case.
 * In a similar fashion `Try<A>` may have two possible cases `Success(a: A)` for computations that succeed and `Failure(e: Throwable)` if they fail
 * with an exception.
 *
 * Both [arrow.core.Try] and [arrow.core.Option] are examples of data types that can be computed over transforming their inner results.
 *
 * ```kotlin:ank:playground
 * import arrow.*
 * import arrow.core.*
 * import arrow.data.*
 *
 * fun main(args: Array<String>) {
 *   val result =
 *   //sampleStart
 *   Try { "1".toInt() }.map { it * 2 }
 *   //sampleEnd
 *   println(result)
 * }
 * ```
 *
 * ```kotlin:ank:playground
 * import arrow.*
 * import arrow.core.*
 * import arrow.data.*
 *
 * fun main(args: Array<String>) {
 *   val result =
 *   //sampleStart
 *   Option(1).map { it * 2 }
 *   //sampleEnd
 *   println(result)
 * }
 * ```
 *
 */
fun Companion.functor(): SequenceKFunctor = object : arrow.data.extensions.SequenceKFunctor {  }