/*
 * Copyright 2023 asyncer.io projects
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.asyncer.r2dbc.mysql.message.server;

import io.asyncer.r2dbc.mysql.MySqlColumnMetadata;
import io.asyncer.r2dbc.mysql.internal.util.NettyBufferUtils;
import io.asyncer.r2dbc.mysql.message.FieldValue;
import io.netty.util.ReferenceCounted;

import static io.asyncer.r2dbc.mysql.internal.util.AssertUtils.requireNonNull;

/**
 * A message includes data fields which is a row of result.
 */
public final class RowMessage implements ReferenceCounted, ServerMessage {

    static final short NULL_VALUE = 0xFB;

    private static final byte BIT_MASK_INIT = 1 << 2;

    private final FieldReader reader;

    RowMessage(FieldReader reader) {
        this.reader = requireNonNull(reader, "reader must not be null");
    }

    /**
     * Decode this message to an array of {@link FieldValue}.
     *
     * @param isBinary if decode with binary protocol.
     * @param context  information context array.
     * @return the {@link FieldValue} array.
     */
    public FieldValue[] decode(boolean isBinary, MySqlColumnMetadata[] context) {
        return isBinary ? binary(context) : text(context.length);
    }

    private FieldValue[] text(int size) {
        FieldValue[] fields = new FieldValue[size];

        try {
            for (int i = 0; i < size; ++i) {
                if (NULL_VALUE == reader.getUnsignedByte()) {
                    reader.skipOneByte();
                    fields[i] = FieldValue.nullField();
                } else {
                    fields[i] = reader.readVarIntSizedField();
                }
            }

            return fields;
        } catch (Throwable e) {
            NettyBufferUtils.releaseAll(fields, size);
            throw e;
        }
    }

    private FieldValue[] binary(MySqlColumnMetadata[] context) {
        reader.skipOneByte(); // constant 0x00

        int size = context.length;
        // MySQL will make sure columns less than 4096, no need check overflow.
        byte[] nullBitmap = reader.readSizeFixedBytes((size + 9) >> 3);
        int bitmapIndex = 0;
        byte bitMask = BIT_MASK_INIT;
        FieldValue[] fields = new FieldValue[size];

        try {
            for (int i = 0; i < size; ++i) {
                if ((nullBitmap[bitmapIndex] & bitMask) != 0) {
                    fields[i] = FieldValue.nullField();
                } else {
                    int bytes = context[i].getType().getBinarySize();
                    if (bytes > 0) {
                        fields[i] = reader.readSizeFixedField(bytes);
                    } else {
                        fields[i] = reader.readVarIntSizedField();
                    }
                }

                bitMask <<= 1;

                // Do NOT use `bitMask == 0` only.
                if ((bitMask & 0xFF) == 0) {
                    // An approach to circular left shift 1-bit.
                    bitMask = 1;
                    // Current byte has been completed by read.
                    ++bitmapIndex;
                }
            }

            return fields;
        } catch (Throwable e) {
            NettyBufferUtils.releaseAll(fields, size);
            throw e;
        }
    }

    @Override
    public int refCnt() {
        return reader.refCnt();
    }

    @Override
    public RowMessage retain() {
        reader.retain();
        return this;
    }

    @Override
    public RowMessage retain(int increment) {
        reader.retain(increment);
        return this;
    }

    @Override
    public RowMessage touch() {
        reader.touch();
        return this;
    }

    @Override
    public RowMessage touch(Object o) {
        reader.touch(o);
        return this;
    }

    @Override
    public boolean release() {
        return reader.release();
    }

    @Override
    public boolean release(int decrement) {
        return reader.release(decrement);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof RowMessage)) {
            return false;
        }
        RowMessage that = (RowMessage) o;
        return reader.equals(that.reader);
    }

    @Override
    public int hashCode() {
        return reader.hashCode();
    }

    @Override
    public String toString() {
        return "RowMessage(encoded)";
    }
}
