/*
 * Decompiled with CFR 0.152.
 */
package io.atlasmap.core;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import io.atlasmap.api.AtlasContextFactory;
import io.atlasmap.api.AtlasException;
import io.atlasmap.v2.ADMDigest;
import io.atlasmap.v2.AtlasMapping;
import io.atlasmap.v2.DataSourceKey;
import io.atlasmap.v2.DataSourceMetadata;
import io.atlasmap.v2.Json;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Path;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.zip.GZIPInputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ADMArchiveHandler {
    private static final Logger LOG = LoggerFactory.getLogger(ADMArchiveHandler.class);
    private static final String MAPPING_DEFINITION_FILTER = "atlasmapping";
    private static final String MAPPING_DEFINITION_TEMPLATE = "atlasmapping-UI.%s.json";
    private static final String GZIPPED_ADM_DIGEST_FILTER = "adm-catalog-files";
    private static final String GZIPPED_ADM_DIGEST_TEMPLATE = "adm-catalog-files-%s.gz";
    private byte[] buffer = new byte[2048];
    private byte[] gzippedAdmDigestBytes = null;
    private byte[] mappingDefinitionBytes = null;
    private ObjectMapper jsonMapper;
    private ObjectMapper jsonMapperForDigest;
    private AtlasMapping mappingDefinition = null;
    private String mappingDefinitionId = "0";
    private Map<DataSourceKey, DataSourceMetadata> dataSourceMetadata;
    private boolean ignoreLibrary = false;
    private Path persistDirectory;
    private Path libraryDirectory;

    public ADMArchiveHandler() {
        this(ADMArchiveHandler.class.getClassLoader());
    }

    public ADMArchiveHandler(ClassLoader loader) {
        this.jsonMapper = Json.withClassLoader((ClassLoader)loader);
        this.jsonMapperForDigest = this.jsonMapper.copy();
        this.jsonMapperForDigest.configure(DeserializationFeature.UNWRAP_ROOT_VALUE, false);
    }

    public void load(Path path) throws AtlasException {
        this.clear();
        File file = path.toFile();
        if (!file.exists() || !file.isFile() && !file.isDirectory()) {
            throw new AtlasException(String.format("'%s' doesn't exist or is not a regular file/directory", path.toString()));
        }
        if (file.isDirectory()) {
            this.loadExploded(file);
        } else if (file.getName().toLowerCase().endsWith(".adm")) {
            this.loadADMFile(file);
        } else {
            try (FileInputStream fin = new FileInputStream(file);){
                this.mappingDefinitionBytes = this.readIntoByteArray(fin);
            }
            catch (Exception e) {
                throw new AtlasException(String.format("Invalid mapping definition file: '%s'", path.toString()), (Throwable)e);
            }
        }
    }

    public void load(InputStream in) throws AtlasException {
        this.load(AtlasContextFactory.Format.ADM, in);
    }

    public void load(AtlasContextFactory.Format format, InputStream in) throws AtlasException {
        if (format == AtlasContextFactory.Format.ADM) {
            this.loadADMStream(in);
        } else {
            try {
                this.mappingDefinitionBytes = this.readIntoByteArray(in);
            }
            catch (Exception e) {
                throw new AtlasException("Invalid mapping definition from stream", (Throwable)e);
            }
        }
    }

    public void export(OutputStream out) throws AtlasException {
        LOG.debug("Creating ADM archive file for ID:'{}'", (Object)this.mappingDefinitionId);
        try (ZipOutputStream zipOut = new ZipOutputStream(out);){
            ZipEntry catEntry = null;
            if (this.getMappingDefinitionBytes() != null) {
                String mappingFileName = this.getMappingDefinitionFileName();
                LOG.debug("  Creating mapping definition file '{}'", (Object)mappingFileName);
                catEntry = new ZipEntry(mappingFileName);
                zipOut.putNextEntry(catEntry);
                zipOut.write(this.getMappingDefinitionBytes(), 0, this.getMappingDefinitionBytes().length);
                zipOut.closeEntry();
            }
            if (this.getGzippedADMDigestBytes() != null) {
                LOG.debug("  Creating gzipped ADM digest file '{}'", (Object)this.getGzippedADMDigestFileName());
                catEntry = new ZipEntry(this.getGzippedADMDigestFileName());
                zipOut.putNextEntry(catEntry);
                zipOut.write(this.getGzippedADMDigestBytes(), 0, this.getGzippedADMDigestBytes().length);
                zipOut.closeEntry();
                zipOut.putNextEntry(new ZipEntry("lib/"));
                zipOut.closeEntry();
            }
            if (!this.isIgnoreLibrary() && this.libraryDirectory != null && this.libraryDirectory.toFile().isDirectory()) {
                for (File jarFile : this.libraryDirectory.toFile().listFiles()) {
                    LOG.debug("  Creating jar file entry '{}'", (Object)("lib/" + jarFile.getName()));
                    ZipEntry libEntry = new ZipEntry("lib/" + jarFile.getName());
                    zipOut.putNextEntry(libEntry);
                    this.redirectStream(new FileInputStream(jarFile), zipOut);
                    zipOut.closeEntry();
                }
            }
        }
        catch (Exception e) {
            throw new AtlasException("Error exporting ADM archive file", (Throwable)e);
        }
    }

    public void persist() throws AtlasException {
        if (this.persistDirectory == null) {
            throw new AtlasException("Persist Directory must be set");
        }
        Path mdPath = this.persistDirectory.resolve(this.getMappingDefinitionFileName());
        if (this.getMappingDefinitionBytes() != null) {
            try {
                this.mappingDefinition = (AtlasMapping)this.jsonMapper.readValue(this.getMappingDefinitionBytes(), AtlasMapping.class);
            }
            catch (Exception e) {
                LOG.warn("Invalid serialized mapping definition content detected, discarding");
                this.mappingDefinitionBytes = null;
                this.mappingDefinition = null;
            }
        }
        if (this.mappingDefinition != null) {
            try {
                this.jsonMapper.writeValue(mdPath.toFile(), (Object)this.mappingDefinition);
            }
            catch (Exception e) {
                LOG.warn("Failed to persist mapping definition", (Throwable)e);
            }
        }
        if (this.getGzippedADMDigestBytes() != null) {
            Path digestPath = this.persistDirectory.resolve(this.getGzippedADMDigestFileName());
            try (FileOutputStream out = new FileOutputStream(digestPath.toFile());){
                out.write(this.getGzippedADMDigestBytes());
            }
            catch (Exception e) {
                LOG.warn("Failed to persist gzipped ADM digest file");
            }
        }
    }

    public AtlasMapping getMappingDefinition() {
        block3: {
            if (this.mappingDefinition == null && this.mappingDefinitionBytes != null) {
                try {
                    this.mappingDefinition = (AtlasMapping)this.jsonMapper.readValue(this.mappingDefinitionBytes, AtlasMapping.class);
                }
                catch (Exception e) {
                    LOG.warn("Invalid serialized mapping definition content detected, discarding");
                    this.mappingDefinitionBytes = null;
                    if (!LOG.isDebugEnabled()) break block3;
                    LOG.warn("", (Throwable)e);
                }
            }
        }
        return this.mappingDefinition;
    }

    public void setMappingDefinition(AtlasMapping mapping) {
        this.mappingDefinitionBytes = null;
        this.mappingDefinition = mapping;
    }

    public void setMappingDefinitionBytes(InputStream is) throws AtlasException {
        try {
            this.mappingDefinition = null;
            this.mappingDefinitionBytes = this.readIntoByteArray(is);
            if (LOG.isDebugEnabled()) {
                LOG.debug(this.jsonMapper.writeValueAsString((Object)this.getMappingDefinition()));
            }
        }
        catch (Exception e) {
            throw new AtlasException((Throwable)e);
        }
    }

    public byte[] getMappingDefinitionBytes() throws AtlasException {
        try {
            if (this.mappingDefinitionBytes == null && this.mappingDefinition != null) {
                this.mappingDefinitionBytes = this.jsonMapper.writeValueAsBytes((Object)this.mappingDefinition);
            }
            return this.mappingDefinitionBytes;
        }
        catch (Exception e) {
            throw new AtlasException((Throwable)e);
        }
    }

    public void setGzippedADMDigest(InputStream is) throws AtlasException {
        try {
            this.gzippedAdmDigestBytes = this.readIntoByteArray(is);
        }
        catch (Exception e) {
            throw new AtlasException((Throwable)e);
        }
    }

    public byte[] getGzippedADMDigestBytes() {
        return this.gzippedAdmDigestBytes;
    }

    public DataSourceMetadata getDataSourceMetadata(boolean isSource, String documentId) throws AtlasException {
        return this.getDataSourceMetadata(new DataSourceKey(isSource, documentId));
    }

    public DataSourceMetadata getDataSourceMetadata(DataSourceKey key) throws AtlasException {
        if (this.getDataSourceMetadataMap() == null) {
            return null;
        }
        return this.getDataSourceMetadataMap().get(key);
    }

    public Map<DataSourceKey, DataSourceMetadata> getDataSourceMetadataMap() throws AtlasException {
        if (this.dataSourceMetadata == null) {
            if (this.gzippedAdmDigestBytes == null) {
                return null;
            }
            try (GZIPInputStream in = new GZIPInputStream(new ByteArrayInputStream(this.gzippedAdmDigestBytes));){
                ADMDigest digest = (ADMDigest)this.jsonMapperForDigest.readValue((InputStream)in, ADMDigest.class);
                this.dataSourceMetadata = new HashMap<DataSourceKey, DataSourceMetadata>();
                for (int i = 0; i < digest.getExportMeta().length; ++i) {
                    DataSourceMetadata meta = digest.getExportMeta()[i];
                    String spec = digest.getExportBlockData()[i].getValue();
                    if (meta.getId() == null) {
                        meta.setId(meta.getName());
                    }
                    meta.setSpecification(spec != null ? spec.getBytes() : null);
                    this.dataSourceMetadata.put(new DataSourceKey(meta.getIsSource(), meta.getId()), meta);
                }
            }
            catch (Exception e) {
                throw new AtlasException((Throwable)e);
            }
        }
        return Collections.unmodifiableMap(this.dataSourceMetadata);
    }

    public AtlasMapping cloneMappingDefinition() throws AtlasException {
        AtlasMapping atlasMapping = this.getMappingDefinition();
        if (atlasMapping == null) {
            return null;
        }
        try {
            byte[] bytes = this.jsonMapper.writeValueAsBytes((Object)atlasMapping);
            return (AtlasMapping)this.jsonMapper.readValue(bytes, AtlasMapping.class);
        }
        catch (Exception e) {
            throw new AtlasException((Throwable)e);
        }
    }

    public void clear() {
        this.mappingDefinitionBytes = null;
        this.mappingDefinition = null;
        this.gzippedAdmDigestBytes = null;
        this.dataSourceMetadata = null;
    }

    public void setIgnoreLibrary(boolean ignoreLib) {
        this.ignoreLibrary = ignoreLib;
    }

    public boolean isIgnoreLibrary() {
        return this.ignoreLibrary;
    }

    public void setPersistDirectory(Path dir) throws AtlasException {
        this.ensureDirectory(dir);
        this.persistDirectory = dir;
    }

    public void setLibraryDirectory(Path dir) throws AtlasException {
        this.ensureDirectory(dir);
        this.libraryDirectory = dir;
    }

    public void setMappingDefinitionId(String id) {
        this.mappingDefinitionId = id;
    }

    public String getGzippedADMDigestFileName() {
        return String.format(GZIPPED_ADM_DIGEST_TEMPLATE, this.mappingDefinitionId);
    }

    public String getMappingDefinitionFileName() {
        return String.format(MAPPING_DEFINITION_TEMPLATE, this.mappingDefinitionId);
    }

    private void loadExploded(File dir) throws AtlasException {
        File digestFile;
        this.setPersistDirectory(dir.toPath());
        this.mappingDefinitionId = dir.getName();
        File mappingDefinitionFile = dir.toPath().resolve(this.getMappingDefinitionFileName()).toFile();
        if (mappingDefinitionFile.exists() && mappingDefinitionFile.isFile()) {
            try (FileInputStream mappingdefis = new FileInputStream(mappingDefinitionFile);){
                this.mappingDefinitionBytes = this.readIntoByteArray(mappingdefis);
            }
            catch (Exception e) {
                throw new AtlasException("Failed to read mapping definition file", (Throwable)e);
            }
        }
        if ((digestFile = dir.toPath().resolve(this.getGzippedADMDigestFileName()).toFile()).exists() && digestFile.isFile()) {
            try (FileInputStream digestis = new FileInputStream(digestFile);){
                this.gzippedAdmDigestBytes = this.readIntoByteArray(digestis);
            }
            catch (Exception e) {
                throw new AtlasException("Failed to read digest file", (Throwable)e);
            }
        }
    }

    private void loadADMFile(File file) throws AtlasException {
        try {
            this.loadADMStream(new FileInputStream(file));
        }
        catch (AtlasException ae) {
            throw ae;
        }
        catch (Exception e) {
            throw new AtlasException((Throwable)e);
        }
    }

    private void loadADMStream(InputStream in) throws AtlasException {
        ZipInputStream zipIn = null;
        try {
            ZipEntry catEntry;
            zipIn = new ZipInputStream(in);
            boolean mappingDefinitionFound = false;
            while ((catEntry = zipIn.getNextEntry()) != null) {
                String catEntryName = catEntry.getName();
                LOG.debug("  Extracting ADM file entry '{}'", (Object)catEntryName);
                if (catEntryName.contains(GZIPPED_ADM_DIGEST_FILTER)) {
                    this.gzippedAdmDigestBytes = this.readIntoByteArray(zipIn);
                    continue;
                }
                if (!this.isIgnoreLibrary() && catEntryName.contains(".jar")) {
                    if (this.libraryDirectory == null) {
                        throw new AtlasException("Library directory is not specified");
                    }
                    int separatorPos = catEntryName.replaceAll("\\\\", "/").lastIndexOf("/");
                    String name = separatorPos == -1 ? catEntryName : catEntryName.substring(separatorPos + 1);
                    Path libPath = this.libraryDirectory.resolve(name);
                    try (FileOutputStream fos = new FileOutputStream(libPath.toFile());){
                        this.redirectStream(zipIn, fos);
                    }
                    catch (Exception e) {
                        LOG.warn(String.format("Failed to save a jar file '%s', ignoring...", name), (Throwable)e);
                    }
                    continue;
                }
                if (catEntryName.contains(MAPPING_DEFINITION_FILTER)) {
                    if (mappingDefinitionFound) {
                        throw new AtlasException("Multiple mapping definition files are found in a same .adm archive");
                    }
                    this.mappingDefinitionBytes = this.readIntoByteArray(zipIn);
                    mappingDefinitionFound = true;
                    continue;
                }
                LOG.debug("Ignoring file '{}' in .adm archive", (Object)catEntryName);
            }
        }
        catch (Exception e) {
            throw new AtlasException((Throwable)e);
        }
        finally {
            try {
                zipIn.close();
            }
            catch (Exception exception) {}
        }
    }

    private void redirectStream(InputStream in, OutputStream out) throws Exception {
        int len = 0;
        while ((len = in.read(this.buffer)) > 0) {
            out.write(this.buffer, 0, len);
        }
    }

    private byte[] readIntoByteArray(InputStream in) throws Exception {
        try (ByteArrayOutputStream baos = new ByteArrayOutputStream();){
            this.redirectStream(in, baos);
            byte[] byArray = baos.toByteArray();
            return byArray;
        }
    }

    private boolean ensureDirectory(Path dir) throws AtlasException {
        if (dir == null) {
            throw new AtlasException(String.format("Directory must not be Null", new Object[0]));
        }
        File dirf = dir.toFile();
        if (dirf.exists() && !dirf.isDirectory()) {
            throw new AtlasException(String.format("File '%s' is not a directory", dirf.getAbsolutePath()));
        }
        if (!dirf.exists()) {
            dirf.mkdirs();
        }
        return true;
    }
}

