/*
 * Decompiled with CFR 0.152.
 */
package io.avaje.jsonb.core;

import io.avaje.json.JsonAdapter;
import io.avaje.json.JsonDataException;
import io.avaje.json.JsonReader;
import io.avaje.json.JsonWriter;
import io.avaje.jsonb.AdapterFactory;
import io.avaje.jsonb.Jsonb;
import io.avaje.jsonb.core.Util;
import java.lang.reflect.Type;
import java.net.Inet4Address;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.UnknownHostException;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;

final class BasicTypeAdapters {
    static final AdapterFactory FACTORY = (type, jsonb) -> {
        if (type == Byte.TYPE) {
            return new ByteAdapter();
        }
        if (type == Character.TYPE) {
            return new CharacterAdapter();
        }
        if (type == Short.TYPE) {
            return new ShortAdapter();
        }
        if (type == Byte.class) {
            return new ByteAdapter().nullSafe();
        }
        if (type == Character.class) {
            return new CharacterAdapter().nullSafe();
        }
        if (type == Short.class) {
            return new ShortAdapter().nullSafe();
        }
        if (type == UUID.class) {
            return new UuidAdapter().nullSafe();
        }
        if (type == URL.class) {
            return new UrlAdapter().nullSafe();
        }
        if (type == URI.class) {
            return new UriAdapter().nullSafe();
        }
        if (type == InetAddress.class) {
            return new InetAddressAdapter().nullSafe();
        }
        if (type == Inet4Address.class) {
            return new InetAddressAdapter().nullSafe();
        }
        if (type == Inet6Address.class) {
            return new InetAddressAdapter().nullSafe();
        }
        if (type == StackTraceElement.class) {
            return new StackTraceElementAdapter().nullSafe();
        }
        if (type == Object.class) {
            return new ObjectJsonAdapter(jsonb).nullSafe();
        }
        if (type == Throwable.class) {
            return new ThrowableAdapter(jsonb).nullSafe();
        }
        Class<?> rawType = Util.rawType(type);
        if (rawType.isEnum()) {
            return new EnumJsonAdapter(rawType).nullSafe();
        }
        return null;
    };

    BasicTypeAdapters() {
    }

    static int rangeCheckNextInt(JsonReader reader, String typeMessage, int min, int max) {
        int value = reader.readInt();
        if (value >= min && value <= max) {
            return value;
        }
        throw new JsonDataException(String.format("Expected %s but was %s at path %s", typeMessage, value, reader.location()));
    }

    static final class ByteAdapter
    implements JsonAdapter<Byte> {
        ByteAdapter() {
        }

        public Byte fromJson(JsonReader reader) {
            return (byte)BasicTypeAdapters.rangeCheckNextInt(reader, "a byte", -128, 255);
        }

        public void toJson(JsonWriter writer, Byte value) {
            writer.value((long)(value.intValue() & 0xFF));
        }

        public String toString() {
            return "JsonAdapter(Byte)";
        }
    }

    static final class CharacterAdapter
    implements JsonAdapter<Character> {
        CharacterAdapter() {
        }

        public Character fromJson(JsonReader reader) {
            String value = reader.readString();
            if (value.length() > 1) {
                throw new JsonDataException(String.format("Expected %s but was %s at path %s", "a char", "\"" + value + "\"", reader.location()));
            }
            return Character.valueOf(value.charAt(0));
        }

        public void toJson(JsonWriter writer, Character value) {
            writer.value(value.toString());
        }

        public String toString() {
            return "JsonAdapter(Character)";
        }
    }

    static final class ShortAdapter
    implements JsonAdapter<Short> {
        ShortAdapter() {
        }

        public Short fromJson(JsonReader reader) {
            return (short)BasicTypeAdapters.rangeCheckNextInt(reader, "a short", Short.MIN_VALUE, Short.MAX_VALUE);
        }

        public void toJson(JsonWriter writer, Short value) {
            writer.value((long)value.intValue());
        }

        public String toString() {
            return "JsonAdapter(Short)";
        }
    }

    private static final class UuidAdapter
    implements JsonAdapter<UUID> {
        private UuidAdapter() {
        }

        public UUID fromJson(JsonReader reader) {
            return UUID.fromString(reader.readString());
        }

        public void toJson(JsonWriter writer, UUID value) {
            writer.value(value.toString());
        }

        public String toString() {
            return "JsonAdapter(UUID)";
        }
    }

    private static final class UrlAdapter
    implements JsonAdapter<URL> {
        private UrlAdapter() {
        }

        public URL fromJson(JsonReader reader) {
            try {
                return new URI(reader.readString()).toURL();
            }
            catch (MalformedURLException | URISyntaxException e) {
                throw new JsonDataException((Throwable)e);
            }
        }

        public void toJson(JsonWriter writer, URL value) {
            writer.value(value.toString());
        }

        public String toString() {
            return "JsonAdapter(URL)";
        }
    }

    private static final class UriAdapter
    implements JsonAdapter<URI> {
        private UriAdapter() {
        }

        public URI fromJson(JsonReader reader) {
            return URI.create(reader.readString());
        }

        public void toJson(JsonWriter writer, URI value) {
            writer.value(value.toString());
        }

        public String toString() {
            return "JsonAdapter(URI)";
        }
    }

    private static final class InetAddressAdapter
    implements JsonAdapter<InetAddress> {
        private InetAddressAdapter() {
        }

        public InetAddress fromJson(JsonReader reader) {
            try {
                return InetAddress.getByName(reader.readString());
            }
            catch (UnknownHostException e) {
                throw new JsonDataException((Throwable)e);
            }
        }

        public void toJson(JsonWriter writer, InetAddress value) {
            writer.value(value.getHostAddress());
        }

        public String toString() {
            return "JsonAdapter(InetAddress)";
        }
    }

    private static final class StackTraceElementAdapter
    implements JsonAdapter<StackTraceElement> {
        private StackTraceElementAdapter() {
        }

        public StackTraceElement fromJson(JsonReader reader) {
            throw new UnsupportedOperationException();
        }

        public void toJson(JsonWriter writer, StackTraceElement value) {
            writer.value(value.toString());
        }

        public String toString() {
            return "JsonAdapter(StackTraceElement)";
        }
    }

    static final class ObjectJsonAdapter
    implements JsonAdapter<Object> {
        private final Jsonb jsonb;
        private final JsonAdapter<List> listAdapter;
        private final JsonAdapter<Map> mapAdapter;
        private final JsonAdapter<String> stringAdapter;
        private final JsonAdapter<Double> doubleAdapter;
        private final JsonAdapter<Boolean> booleanAdapter;

        ObjectJsonAdapter(Jsonb jsonb) {
            this.jsonb = jsonb;
            this.listAdapter = jsonb.adapter(List.class);
            this.mapAdapter = jsonb.adapter(Map.class);
            this.stringAdapter = jsonb.adapter(String.class);
            this.doubleAdapter = jsonb.adapter(Double.class);
            this.booleanAdapter = jsonb.adapter(Boolean.class);
        }

        public Object fromJson(JsonReader reader) {
            switch (reader.currentToken()) {
                case BEGIN_ARRAY: {
                    return this.listAdapter.fromJson(reader);
                }
                case BEGIN_OBJECT: {
                    return this.mapAdapter.fromJson(reader);
                }
                case STRING: {
                    return this.stringAdapter.fromJson(reader);
                }
                case NUMBER: {
                    Double d = (Double)this.doubleAdapter.fromJson(reader);
                    if (d % 1.0 == 0.0) {
                        return d.longValue();
                    }
                    return d;
                }
                case BOOLEAN: {
                    return this.booleanAdapter.fromJson(reader);
                }
                case NULL: {
                    return null;
                }
            }
            throw new IllegalStateException("Expected a value but was " + String.valueOf(reader.currentToken()) + " at path " + reader.location());
        }

        public void toJson(JsonWriter writer, Object value) {
            Class<?> valueClass = value.getClass();
            if (valueClass == Object.class) {
                writer.beginObject();
                writer.endObject();
            } else if (value instanceof Optional) {
                Optional op = (Optional)value;
                op.ifPresentOrElse(v -> this.toJson(writer, v), () -> ((JsonWriter)writer).nullValue());
            } else {
                this.jsonb.adapter(this.toJsonType(valueClass)).toJson(writer, value);
            }
        }

        private Type toJsonType(Class<?> valueClass) {
            if (Map.class.isAssignableFrom(valueClass)) {
                return Map.class;
            }
            return Collection.class.isAssignableFrom(valueClass) ? Collection.class : valueClass;
        }

        public String toString() {
            return "JsonAdapter(Object)";
        }
    }

    private static final class ThrowableAdapter
    implements JsonAdapter<Throwable> {
        private static final int MAX_STACK = 5;
        private final JsonAdapter<StackTraceElement> stackTraceElementAdapter;

        private ThrowableAdapter(Jsonb jsonb) {
            this.stackTraceElementAdapter = jsonb.adapter(StackTraceElement.class);
        }

        public Throwable fromJson(JsonReader reader) {
            throw new UnsupportedOperationException();
        }

        public void toJson(JsonWriter writer, Throwable value) {
            Throwable[] suppressed;
            Throwable cause;
            writer.beginObject();
            writer.name("type");
            writer.value(value.getClass().toString());
            writer.name("message");
            writer.value(value.getMessage());
            StackTraceElement[] stackTrace = value.getStackTrace();
            if (stackTrace != null && stackTrace.length > 0) {
                int end = Math.min(5, stackTrace.length);
                List<StackTraceElement> stackTraceElements = Arrays.asList(stackTrace).subList(0, end);
                writer.name("stackTrace");
                writer.beginArray();
                for (StackTraceElement element : stackTraceElements) {
                    this.stackTraceElementAdapter.toJson(writer, (Object)element);
                }
                writer.endArray();
            }
            if ((cause = value.getCause()) != null) {
                writer.name("cause");
                this.toJson(writer, cause);
            }
            if ((suppressed = value.getSuppressed()) != null && suppressed.length > 0) {
                writer.name("suppressed");
                writer.beginArray();
                for (Throwable sup : suppressed) {
                    this.toJson(writer, sup);
                }
                writer.endArray();
            }
            writer.endObject();
        }

        public String toString() {
            return "JsonAdapter(URI)";
        }
    }

    static class EnumJsonAdapter<T extends Enum<T>>
    implements JsonAdapter<T> {
        protected final Class<T> enumType;

        EnumJsonAdapter(Class<T> enumType) {
            this.enumType = enumType;
        }

        public T fromJson(JsonReader reader) {
            String value = reader.readString();
            return Enum.valueOf(this.enumType, value);
        }

        public void toJson(JsonWriter writer, T value) {
            if (value != null) {
                writer.value(((Enum)value).name());
            } else {
                writer.value((String)null);
            }
        }

        protected final void throwException(Object value, String location) {
            throw new JsonDataException("Unable to determine enum value " + String.valueOf(this.enumType) + " value for " + String.valueOf(value) + " at " + location);
        }

        public String toString() {
            return "JsonAdapter(" + this.enumType.getName() + ")";
        }
    }

    static final class StringAdapter
    implements JsonAdapter<String> {
        StringAdapter() {
        }

        public String fromJson(JsonReader reader) {
            return reader.readString();
        }

        public void toJson(JsonWriter writer, String value) {
            writer.value(value);
        }

        public String toString() {
            return "JsonAdapter(String)";
        }
    }

    static final class LongAdapter
    implements JsonAdapter<Long> {
        LongAdapter() {
        }

        public Long fromJson(JsonReader reader) {
            return reader.readLong();
        }

        public void toJson(JsonWriter writer, Long value) {
            writer.value(value);
        }

        public String toString() {
            return "JsonAdapter(Long)";
        }
    }

    static final class IntegerAdapter
    implements JsonAdapter<Integer> {
        IntegerAdapter() {
        }

        public Integer fromJson(JsonReader reader) {
            return reader.readInt();
        }

        public void toJson(JsonWriter writer, Integer value) {
            writer.value(value);
        }

        public String toString() {
            return "JsonAdapter(Integer)";
        }
    }

    static final class FloatAdapter
    implements JsonAdapter<Float> {
        FloatAdapter() {
        }

        public Float fromJson(JsonReader reader) {
            float value = (float)reader.readDouble();
            if (Float.isInfinite(value)) {
                throw new JsonDataException("JSON forbids NaN and infinities: " + value + " at path " + reader.location());
            }
            return Float.valueOf(value);
        }

        public void toJson(JsonWriter writer, Float value) {
            Objects.requireNonNull(value);
            writer.value((double)value.floatValue());
        }

        public String toString() {
            return "JsonAdapter(Float)";
        }
    }

    static final class DoubleAdapter
    implements JsonAdapter<Double> {
        DoubleAdapter() {
        }

        public Double fromJson(JsonReader reader) {
            return reader.readDouble();
        }

        public void toJson(JsonWriter writer, Double value) {
            writer.value(value);
        }

        public String toString() {
            return "JsonAdapter(Double)";
        }
    }

    static final class BooleanAdapter
    implements JsonAdapter<Boolean> {
        BooleanAdapter() {
        }

        public Boolean fromJson(JsonReader reader) {
            return reader.readBoolean();
        }

        public void toJson(JsonWriter writer, Boolean value) {
            writer.value(value);
        }

        public String toString() {
            return "JsonAdapter(Boolean)";
        }
    }
}

