/*
 * This file is part of the BigConnect project.
 *
 * Copyright (c) 2013-2020 MWARE SOLUTIONS SRL
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License version 3
 * as published by the Free Software Foundation with the addition of the
 * following permission added to Section 15 as permitted in Section 7(a):
 * FOR ANY PART OF THE COVERED WORK IN WHICH THE COPYRIGHT IS OWNED BY
 * MWARE SOLUTIONS SRL, MWARE SOLUTIONS SRL DISCLAIMS THE WARRANTY OF
 * NON INFRINGEMENT OF THIRD PARTY RIGHTS

 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Affero General Public License for more details.
 * You should have received a copy of the GNU Affero General Public License
 * along with this program; if not, see http://www.gnu.org/licenses or write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA, 02110-1301 USA, or download the license from the following URL:
 * https://www.gnu.org/licenses/agpl-3.0.txt
 *
 * The interactive user interfaces in modified source and object code versions
 * of this program must display Appropriate Legal Notices, as required under
 * Section 5 of the GNU Affero General Public License.
 *
 * You can be released from the requirements of the license by purchasing
 * a commercial license. Buying such a license is mandatory as soon as you
 * develop commercial activities involving the BigConnect software without
 * disclosing the source code of your own applications.
 *
 * These activities include: offering paid services to customers as an ASP,
 * embedding the product in a web application, shipping BigConnect with a
 * closed source product.
 */
package com.mware.ge.store.decoder;

import com.mware.ge.*;
import com.mware.ge.collection.Pair;
import com.mware.ge.store.*;
import com.mware.ge.store.util.KeyBase;
import com.mware.ge.store.util.StorableKeyHelper;

import java.util.*;

public class ExtendedDataDecoder {
    private StorableGraph graph;
    private FetchHints fetchHints;
    private Authorizations authorizations;

    public ExtendedDataDecoder(StorableGraph graph, FetchHints fetchHints, Authorizations authorizations) {
        this.graph = graph;
        this.fetchHints = fetchHints;
        this.authorizations = authorizations;
    }

    public ExtendedDataRow decode(List<Pair<StoreKey, StoreValue>> row) {
        if (row.size() == 0)
            return null;

        ExtendedDataRowId rowId = StorableKeyHelper.parseExtendedDataRowId(row.get(0).first().id());
        Set<Property> properties = new HashSet<>();
        row.sort(Comparator.comparingLong(m -> m.other().ts()));

        for (Pair<StoreKey, StoreValue> rowEntry : row) {
            String columnFamily = rowEntry.first().cf();
            if (StorableElement.CF_EXTENDED_DATA.equals(columnFamily)) {
                String[] columnQualifierParts = KeyBase.splitOnValueSeparator(rowEntry.first().cq());
                if (columnQualifierParts.length != 1 && columnQualifierParts.length != 2) {
                    throw new GeException("Invalid column qualifier for extended data row: " + rowId + " (expected 1 or 2 parts, found " + columnQualifierParts.length + ")");
                }
                String propertyName = columnQualifierParts[0];
                String propertyKey = columnQualifierParts.length > 1 ? columnQualifierParts[1] : null;
                com.mware.ge.values.storable.Value propertyValue = graph.getGeSerializer().bytesToObject(rowId, rowEntry.other().value());
                long timestamp = rowEntry.other().ts();
                Visibility visibility = rowEntry.first().visibility();
                if (!authorizations.canRead(visibility))
                    continue;

                StorableExtendedDataRow.StorableExtendedDataRowProperty prop = new StorableExtendedDataRow.StorableExtendedDataRowProperty(
                        propertyName,
                        propertyKey,
                        propertyValue,
                        fetchHints,
                        timestamp,
                        visibility
                );
                properties.add(prop);
            } else {
                throw new GeException("unhandled column family: " + columnFamily);
            }
        }

        if (properties.size() == 0)
            return null;
        else
            return new StorableExtendedDataRow(rowId, properties, fetchHints);
    }
}
