/*
 * Decompiled with CFR 0.152.
 */
package com.mware.core.model.search;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import com.mware.core.config.Configuration;
import com.mware.core.exception.BcAccessDeniedException;
import com.mware.core.exception.BcException;
import com.mware.core.model.clientapi.dto.ClientApiSearch;
import com.mware.core.model.clientapi.dto.ClientApiSearchListResponse;
import com.mware.core.model.clientapi.dto.VisibilityJson;
import com.mware.core.model.graph.GraphRepository;
import com.mware.core.model.graph.GraphUpdateContext;
import com.mware.core.model.properties.SearchSchema;
import com.mware.core.model.properties.types.PropertyMetadata;
import com.mware.core.model.role.AuthorizationRepository;
import com.mware.core.model.search.SearchRepository;
import com.mware.core.model.user.GraphAuthorizationRepository;
import com.mware.core.model.user.PrivilegeRepository;
import com.mware.core.model.user.UserRepository;
import com.mware.core.model.workQueue.Priority;
import com.mware.core.model.workspace.WorkspaceRepository;
import com.mware.core.security.BcVisibility;
import com.mware.core.user.SystemUser;
import com.mware.core.user.User;
import com.mware.core.util.ClientApiConverter;
import com.mware.core.util.StreamUtil;
import com.mware.ge.Authorizations;
import com.mware.ge.Direction;
import com.mware.ge.Graph;
import com.mware.ge.Vertex;
import com.mware.ge.Visibility;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;
import org.json.JSONObject;

@Singleton
public class GeSearchRepository
extends SearchRepository {
    public static final String VISIBILITY_STRING = "search";
    public static final BcVisibility VISIBILITY = new BcVisibility("search");
    private static final String GLOBAL_SAVED_SEARCHES_ROOT_VERTEX_ID = "__bc_globalSavedSearchesRoot";
    private final Graph graph;
    private final GraphRepository graphRepository;
    private final UserRepository userRepository;
    private final AuthorizationRepository authorizationRepository;
    private final PrivilegeRepository privilegeRepository;
    private final WorkspaceRepository workspaceRepository;

    @Inject
    public GeSearchRepository(Graph graph, GraphRepository graphRepository, UserRepository userRepository, Configuration configuration, GraphAuthorizationRepository graphAuthorizationRepository, AuthorizationRepository authorizationRepository, PrivilegeRepository privilegeRepository, WorkspaceRepository workspaceRepository) {
        super(configuration);
        this.graph = graph;
        this.graphRepository = graphRepository;
        this.userRepository = userRepository;
        this.authorizationRepository = authorizationRepository;
        this.privilegeRepository = privilegeRepository;
        this.workspaceRepository = workspaceRepository;
        graphAuthorizationRepository.addAuthorizationToGraph(VISIBILITY_STRING);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public String saveSearch(String id, String name, String url, JSONObject searchParameters, User user, boolean update) {
        Preconditions.checkNotNull((Object)user, (Object)"User is required");
        Authorizations authorizations = this.authorizationRepository.getGraphAuthorizations(user, VISIBILITY_STRING, "user");
        if (!update && this.savedSearchWithSameNameExists(name, this.getUserSavedSearches(user, authorizations))) {
            throw new BcException("An existing saved search with the same name already exists.");
        }
        if (this.graph.doesVertexExist(id, authorizations)) {
            if (this.isSearchGlobal(id, authorizations)) {
                if (!this.privilegeRepository.hasPrivilege(user, "SEARCH_SAVE_GLOBAL")) throw new BcAccessDeniedException("User does not have the privilege to change a global search", user, id);
                this.deleteSearch(id, user);
            } else if (!this.isSearchPrivateToUser(id, user, authorizations)) {
                throw new BcAccessDeniedException("User does not own this this search", user, id);
            }
        }
        try (GraphUpdateContext ctx = this.graphRepository.beginGraphUpdate(Priority.LOW, user, authorizations);){
            ctx.setPushOnQueue(false);
            Vertex searchVertex = (Vertex)this.saveSearchVertex(ctx, id, name, url, searchParameters, user).get();
            Vertex userVertex = this.graph.getVertex(user.getUserId(), authorizations);
            Preconditions.checkNotNull((Object)userVertex, (Object)("Could not find user vertex with id " + user.getUserId()));
            String edgeId = (String)userVertex.getId() + "_" + "__shss" + "_" + (String)searchVertex.getId();
            ctx.getOrCreateEdgeAndUpdate(edgeId, (String)userVertex.getId(), (String)searchVertex.getId(), "__shss", VISIBILITY.getVisibility(), elemCtx -> {});
            Object object = searchVertex.getId();
            return object;
        }
        catch (Exception ex) {
            throw new BcException("Could not save private search", ex);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public String saveGlobalSearch(String id, String name, String url, JSONObject searchParameters, User user, boolean update) {
        if (!(user instanceof SystemUser) && !this.privilegeRepository.hasPrivilege(user, "SEARCH_SAVE_GLOBAL")) {
            throw new BcAccessDeniedException("User does not have the privilege to save a global search", user, id);
        }
        Authorizations authorizations = this.authorizationRepository.getGraphAuthorizations(this.userRepository.getSystemUser(), VISIBILITY_STRING, "user");
        if (!update && this.savedSearchWithSameNameExists(name, this.getGlobalSavedSearches(authorizations))) {
            throw new BcException("An existing global saved search with the same name already exists.");
        }
        if (this.isSearchPrivateToUser(id, user, authorizations)) {
            this.deleteSearch(id, user);
        }
        try (GraphUpdateContext ctx = this.graphRepository.beginGraphUpdate(Priority.LOW, user, authorizations);){
            ctx.setPushOnQueue(false);
            Vertex searchVertex = (Vertex)this.saveSearchVertex(ctx, id, name, url, searchParameters, user).get();
            String edgeId = String.format("%s_%s_%s", GLOBAL_SAVED_SEARCHES_ROOT_VERTEX_ID, "__shss", searchVertex.getId());
            ctx.getOrCreateEdgeAndUpdate(edgeId, (String)this.getGlobalSavedSearchesRootVertex().getId(), (String)searchVertex.getId(), "__shss", VISIBILITY.getVisibility(), elemCtx -> {});
            Object object = searchVertex.getId();
            return object;
        }
        catch (Exception ex) {
            throw new BcException("Could not save global search", ex);
        }
    }

    private boolean savedSearchWithSameNameExists(String name, Iterable<ClientApiSearch> searches) {
        return StreamSupport.stream(searches.spliterator(), false).anyMatch(ss -> ss.name.equals(name));
    }

    private GraphUpdateContext.UpdateFuture<Vertex> saveSearchVertex(GraphUpdateContext ctx, String id, String name, String url, JSONObject searchParameters, User user) {
        Visibility visibility = VISIBILITY.getVisibility();
        return ctx.getOrCreateVertexAndUpdate(id, visibility, "__ss", elemCtx -> {
            PropertyMetadata metadata = new PropertyMetadata(user, new VisibilityJson(), visibility);
            if (elemCtx.isNewElement()) {
                elemCtx.updateBuiltInProperties(metadata);
            }
            SearchSchema.NAME.updateProperty(elemCtx, name != null ? name : "", metadata);
            SearchSchema.URL.updateProperty(elemCtx, url, metadata);
            SearchSchema.PARAMETERS.updateProperty(elemCtx, searchParameters, metadata);
        });
    }

    @Override
    public ClientApiSearchListResponse getSavedSearches(User user) {
        Preconditions.checkNotNull((Object)user, (Object)"User is required");
        Authorizations authorizations = this.authorizationRepository.getGraphAuthorizations(user, VISIBILITY_STRING, "user");
        ClientApiSearchListResponse result = new ClientApiSearchListResponse();
        Iterables.addAll(result.searches, this.getGlobalSavedSearches(authorizations));
        Iterables.addAll(result.searches, this.getUserSavedSearches(user, authorizations));
        return result;
    }

    private Iterable<ClientApiSearch> getUserSavedSearches(User user, Authorizations authorizations) {
        Vertex userVertex = this.graph.getVertex(user.getUserId(), authorizations);
        Preconditions.checkNotNull((Object)userVertex, (Object)("Could not find user vertex with id " + user.getUserId()));
        Iterable<Vertex> userSearchVertices = userVertex.getVertices(Direction.OUT, "__shss", authorizations);
        return StreamUtil.stream(userSearchVertices).map(searchVertex -> GeSearchRepository.toClientApiSearch(searchVertex, ClientApiSearch.Scope.User)).collect(Collectors.toList());
    }

    private Iterable<ClientApiSearch> getGlobalSavedSearches(Authorizations authorizations) {
        Vertex globalSavedSearchesRootVertex = this.getGlobalSavedSearchesRootVertex();
        Iterable<Vertex> globalSearchVertices = globalSavedSearchesRootVertex.getVertices(Direction.OUT, "__shss", authorizations);
        return StreamUtil.stream(globalSearchVertices).map(searchVertex -> GeSearchRepository.toClientApiSearch(searchVertex, ClientApiSearch.Scope.Global)).collect(Collectors.toList());
    }

    private Vertex getGlobalSavedSearchesRootVertex() {
        Authorizations authorizations = this.authorizationRepository.getGraphAuthorizations(this.userRepository.getSystemUser(), VISIBILITY_STRING);
        Vertex globalSavedSearchesRootVertex = this.graph.getVertex(GLOBAL_SAVED_SEARCHES_ROOT_VERTEX_ID, authorizations);
        if (globalSavedSearchesRootVertex == null) {
            globalSavedSearchesRootVertex = this.graph.prepareVertex(GLOBAL_SAVED_SEARCHES_ROOT_VERTEX_ID, new Visibility(VISIBILITY_STRING), "thing").save(authorizations);
            this.graph.flush();
        }
        return globalSavedSearchesRootVertex;
    }

    private static ClientApiSearch toClientApiSearch(Vertex searchVertex) {
        return GeSearchRepository.toClientApiSearch(searchVertex, null);
    }

    public static ClientApiSearch toClientApiSearch(Vertex searchVertex, ClientApiSearch.Scope scope) {
        ClientApiSearch result = new ClientApiSearch();
        result.id = searchVertex.getId();
        result.name = (String)SearchSchema.NAME.getPropertyValue(searchVertex);
        result.url = (String)SearchSchema.URL.getPropertyValue(searchVertex);
        result.scope = scope;
        result.parameters = ClientApiConverter.toClientApiValue((JSONObject)SearchSchema.PARAMETERS.getPropertyValue(searchVertex));
        return result;
    }

    @Override
    public ClientApiSearch getSavedSearch(String id, User user) {
        Authorizations authorizations = this.authorizationRepository.getGraphAuthorizations(user, VISIBILITY_STRING, "user");
        Vertex searchVertex = this.graph.getVertex(id, authorizations);
        if (searchVertex == null) {
            return null;
        }
        return GeSearchRepository.toClientApiSearch(searchVertex);
    }

    @Override
    public ClientApiSearch getSavedSearchOnWorkspace(String id, User user, String workspaceId) {
        boolean hasWorkspaceAccess;
        Authorizations authorizations = this.authorizationRepository.getGraphAuthorizations(user, VISIBILITY_STRING, "user");
        Vertex searchVertex = this.graph.getVertex(id, authorizations);
        if (searchVertex == null) {
            return null;
        }
        boolean isGlobalSearch = this.isSearchGlobal(id, authorizations);
        boolean bl = hasWorkspaceAccess = workspaceId != null && this.workspaceRepository.hasReadPermissions(workspaceId, user);
        if (isGlobalSearch) {
            return GeSearchRepository.toClientApiSearch(searchVertex);
        }
        if (!isGlobalSearch && !hasWorkspaceAccess) {
            return null;
        }
        String workspaceCreatorId = this.workspaceRepository.getCreatorUserId(workspaceId, user);
        if (this.isSearchPrivateToUser(id, this.userRepository.findById(workspaceCreatorId), authorizations)) {
            return GeSearchRepository.toClientApiSearch(searchVertex);
        }
        return null;
    }

    @Override
    public void deleteSearch(String id, User user) {
        Preconditions.checkNotNull((Object)user, (Object)"User is required");
        Authorizations authorizations = this.authorizationRepository.getGraphAuthorizations(user, VISIBILITY_STRING, "user");
        Vertex searchVertex = this.graph.getVertex(id, authorizations);
        Preconditions.checkNotNull((Object)searchVertex, (Object)("Could not find search with id " + id));
        if (this.isSearchGlobal(id, authorizations)) {
            if (!this.privilegeRepository.hasPrivilege(user, "SEARCH_SAVE_GLOBAL")) {
                throw new BcAccessDeniedException("User does not have the privilege to delete a global search", user, id);
            }
        } else if (!this.isSearchPrivateToUser(id, user, authorizations)) {
            throw new BcAccessDeniedException("User does not own this this search", user, id);
        }
        this.graph.deleteVertex(searchVertex, authorizations);
        this.graph.flush();
    }

    @VisibleForTesting
    boolean isSearchGlobal(String id, Authorizations authorizations) {
        if (!this.graph.doesVertexExist(id, authorizations)) {
            return false;
        }
        Iterable<String> vertexIds = this.getGlobalSavedSearchesRootVertex().getVertexIds(Direction.OUT, "__shss", authorizations);
        return StreamUtil.stream(vertexIds).anyMatch(vertexId -> vertexId.equals(id));
    }

    @VisibleForTesting
    boolean isSearchPrivateToUser(String id, User user, Authorizations authorizations) {
        if (user instanceof SystemUser) {
            return false;
        }
        Vertex userVertex = this.graph.getVertex(user.getUserId(), authorizations);
        Preconditions.checkNotNull((Object)userVertex, (Object)("Could not find user vertex with id " + user.getUserId()));
        Iterable<String> vertexIds = userVertex.getVertexIds(Direction.OUT, "__shss", authorizations);
        return StreamUtil.stream(vertexIds).anyMatch(vertexId -> vertexId.equals(id));
    }
}

