package io.boxo.data.models

import org.json.JSONObject

class MiniappSettings(
    val status: Status,
    val brandingEnabled: Boolean,
    val disableUrlWhitelisting: Boolean,
    val whiteListedUrls: List<String>,
    val webviewFrameSize: Int,
    val orientation: Int,
    val whiteLabel: Boolean,
    val actionButtonPosition: ActionButtonPosition,
    val actionButtonsTheme: ActionButtonTheme,
    val statusBarBackground: String,
    val colors: Colors,
    val isConsented: Boolean,
    val miniappInfo: MiniappInfo,
    val hostappInfo: HostappInfo,
    val script: String,
    val json: String
) {
    fun isFullscreen() = webviewFrameSize == 1
}

enum class ActionButtonPosition { FIXED, FLOATING }
enum class ActionButtonTheme { LIGHT, DARK }

enum class Status {
    VERIFIED,
    DISABLED,
    NOT_APPROVED,
    PENDING_VERIFICATION,
    NOT_INTEGRATION,
    MAINTENANCE
}

internal fun String.toMiniappSettings(): MiniappSettings {
    val json = JSONObject(this)
    return MiniappSettings(
        status = when (json.getInt("status")) {
            0 -> Status.VERIFIED
            1 -> Status.DISABLED
            2 -> Status.NOT_APPROVED
            3 -> Status.PENDING_VERIFICATION
            4 -> Status.NOT_INTEGRATION
            5 -> Status.MAINTENANCE
            else -> Status.DISABLED
        },
        brandingEnabled =
        runCatching { json.getBoolean("hide_branding") }.getOrDefault(false).not(),
        disableUrlWhitelisting =
        runCatching { json.getBoolean("disable_url_whitelisting") }.getOrDefault(false),
        whiteListedUrls = runCatching {
            val urlArray = json.getJSONArray("white_listed_urls")
            val urls = mutableListOf<String>()
            repeat(urlArray.length()) { urls.add(urlArray.getString(it)) }
            urls
        }.getOrDefault(emptyList()),
        orientation = runCatching { json.getInt("orientation") }.getOrDefault(0),
        webviewFrameSize = json.getInt("webview_frame_size"),
        statusBarBackground = runCatching { json.getString("status_bar_background") }.getOrDefault("#FFFFFF"),
        whiteLabel = runCatching { json.getBoolean("white_label") }.getOrDefault(false),
        actionButtonsTheme = when (json.optString("action_button_theme")) {
            "light" -> ActionButtonTheme.LIGHT
            else -> ActionButtonTheme.DARK
        },
        actionButtonPosition = when (json.optString("action_button_position")) {
            "floating" -> ActionButtonPosition.FLOATING
            else -> ActionButtonPosition.FIXED
        },
        colors = json.getJSONObject("colors").let { colors ->
            Colors(
                primaryColor = colors.optString("primary_color") ?: "",
                secondaryColor = colors.optString("secondary_color") ?: "",
                tertiaryColor = colors.optString("tertiary_color") ?: ""
            )
        },
        miniappInfo = json.getJSONObject("miniapp").let { miniapp ->
            MiniappInfo(
                name = miniapp.optString("name") ?: "",
                description = miniapp.optString("description") ?: "",
                logo = miniapp.optString("logo") ?: "",
                url = miniapp.optString("url") ?: "",
                requiredFields = miniapp.getJSONArray("required_fields").let { array ->
                    mutableListOf<String>()
                        .apply { for (i in 0 until array.length()) add(array.getString(i)) }
                },
                isShopboxo = runCatching { miniapp.getBoolean("is_shopboxo") }.getOrDefault(false),
                maintenancePageUrl =
                runCatching { miniapp.getString("maintenance_page_url") }.getOrDefault("")
            )
        },
        hostappInfo = json.getJSONObject("hostapp").let { hostapp ->
            HostappInfo(
                name = hostapp.optString("name") ?: "",
                logo = hostapp.optString("logo") ?: "",
                consentText = hostapp.optString("consent_text") ?: "",
                isConsentEnabled = hostapp.optBoolean("is_consent_enabled", true),
                menuStyle = hostapp.optString("menu_style") ?: "default"
            )
        },
        isConsented = json.optBoolean("is_consented", false),
        script = json.optString("script") ?: "",
        json = this
    )
}