package io.boxo.data.network

import android.os.Build
import io.boxo.data.models.MiniappData
import io.boxo.data.models.MiniappSettings
import io.boxo.data.models.toMiniappSettings
import io.boxo.data.storage.IBoxoStorage
import io.boxo.log.exceptions.BoxoException
import io.boxo.log.exceptions.BoxoLoginException
import io.boxo.sdk.Boxo
import io.boxo.sdk.MiniappListCallback
import io.boxo.utils.extensions.await
import kotlinx.coroutines.GlobalScope
import kotlinx.coroutines.launch
import okhttp3.Call
import okhttp3.Callback
import okhttp3.HttpUrl.Companion.toHttpUrl
import okhttp3.HttpUrl.Companion.toHttpUrlOrNull
import okhttp3.MediaType.Companion.toMediaTypeOrNull
import okhttp3.OkHttpClient
import okhttp3.Request
import okhttp3.RequestBody.Companion.toRequestBody
import okhttp3.Response
import org.json.JSONArray
import org.json.JSONObject
import java.io.IOException
import java.security.MessageDigest

internal class NetworkService(
    private val okHttpClient: OkHttpClient,
    private val boxo: Boxo,
    private val storage: IBoxoStorage,
    private val sdkVersion: String
) : SentryService {
    companion object {
        private val stagingBoxo = "https://staging.boxo.io"
        private val prodBoxo = "https://dashboard.boxo.io"
        private val stagingShopboxo = "https://shop-dev.appboxo.com"
        private val prodShopboxo = "https://shop.appboxo.com"
    }

    private val host
        get() = if (boxo.config.isStaging) stagingBoxo else prodBoxo

    fun getMiniapps(callback: MiniappListCallback) {
        val request = Request.Builder()
            .url(
                "$host/api/v1/partner/miniapps/approved_miniapps/".toHttpUrl()
                    .newBuilder()
                    .addQueryParameter("client_id", boxo.config.clientId)
                    .addQueryParameter("sandbox_mode", boxo.config.isSandbox.toString())
                    .build()
            )
            .get()
            .build()
        okHttpClient.newCall(request).enqueue(object : Callback {
            override fun onFailure(call: Call, e: IOException) {
                GlobalScope.launch {
                    sendErrorEvent("", "get_miniapp_list", e)
                }
                callback.onFailure(e)
            }

            override fun onResponse(call: Call, response: Response) {
                val jsonString = response.body?.string() ?: "{}"
                if (response.code >= 400) {
                    runCatching { parseErrorBody(jsonString) }
                        .onFailure { callback.onFailure(Exception(it)) }
                        .onSuccess { callback.onFailure(BoxoException(it)) }
                } else {
                    runCatching {
                        val miniapps = mutableListOf<MiniappData>()
                        val array = JSONArray(jsonString)
                        for (i in 0 until array.length()) {
                            val json = array.getJSONObject(i)
                            miniapps.add(
                                MiniappData(
                                    appId = json.optString("app_id") ?: "",
                                    name = json.optString("name") ?: "",
                                    description = json.optString("long_description") ?: "",
                                    logo = json.optString("logo") ?: "",
                                    category = json.optString("category") ?: ""
                                )
                            )
                        }
                        miniapps
                    }
                        .onSuccess { callback.onSuccess(it) }
                        .onFailure { callback.onFailure(Exception(it)) }
                }
            }
        })
    }

    suspend fun getMiniappSettings(appId: String, token: String): MiniappSettings {
        val request = Request.Builder()
            .apply {
                if (token.isNotBlank()) addHeader("Authorization", token)
            }
            .url(
                "$host/api/v1/miniapps/${appId}/settings/".toHttpUrl()
                    .newBuilder()
                    .addQueryParameter("client_id", boxo.config.clientId)
                    .addQueryParameter("user_reference", boxo.config.userId)
                    .build()
            )
            .get()
            .build()
        val response = okHttpClient.newCall(request).await()
        val jsonString = response.body?.string() ?: "{}"
        if (response.code >= 400) {
            sendErrorEvent(appId, "get_miniapp_settings", BoxoException(jsonString))
            val message = parseErrorBody(jsonString)
            throw BoxoException(message)
        } else {
            return runCatching {
                jsonString.toMiniappSettings()
            }.onFailure {
                sendErrorEvent(appId, "get_miniapp_settings", BoxoException(jsonString))
            }.getOrThrow()
        }
    }

    suspend fun getUserConsent(appId: String): JSONObject {
        val request = Request.Builder()
            .url(
                "$host/api/v1/accounts/consent/get_consent/".toHttpUrl()
                    .newBuilder()
                    .addQueryParameter("client_id", boxo.config.clientId)
                    .addQueryParameter("app_id", appId)
                    .addQueryParameter("user_reference", boxo.config.userId)
                    .build()
            )
            .get()
            .build()
        val response = okHttpClient.newCall(request).await()
        val jsonString = response.body?.string() ?: "{}"
        if (response.code >= 400) {
            sendErrorEvent(appId, "get_user_consent", BoxoException(jsonString))
            val message = parseErrorBody(jsonString)
            throw BoxoException(message)
        } else {
            return runCatching {
                JSONObject(jsonString)
            }.onFailure {
                sendErrorEvent(appId, "get_user_consent", BoxoException(jsonString))
            }.getOrThrow()
        }
    }

    suspend fun revokeUserConsent(appId: String): JSONObject {
        val request = Request.Builder()
            .url(
                "$host/api/v1/accounts/consent/revoke/".toHttpUrl()
                    .newBuilder()
                    .addQueryParameter("client_id", boxo.config.clientId)
                    .addQueryParameter("app_id", appId)
                    .addQueryParameter("user_reference", boxo.config.userId)
                    .build()
            )
            .post("".toRequestBody("application/json; charset=utf-8".toMediaTypeOrNull()))
            .build()
        val response = okHttpClient.newCall(request).await()
        val jsonString = response.body?.string() ?: "{}"
        if (response.code >= 400) {
            sendErrorEvent(appId, "revoke_user_consent", BoxoException(jsonString))
            val message = parseErrorBody(jsonString)
            throw BoxoException(message)
        } else {
            return runCatching {
                JSONObject(jsonString)
            }.onFailure {
                sendErrorEvent(appId, "revoke_user_consent", BoxoException(jsonString))
            }.getOrThrow()
        }
    }

    suspend fun login(appId: String, authCode: String, isShopboxo: Boolean = false): String {
        val host = when {
            isShopboxo && boxo.config.isStaging -> stagingShopboxo
            isShopboxo -> prodShopboxo
            else -> this.host
        }
        val request = Request.Builder()
            .url("${host}/api/v1/authorize/")
            .post(
                JSONObject().apply {
                    put("client_id", boxo.config.clientId)
                    put("app_id", appId)
                    put("user_reference", boxo.config.userId)
                    put("auth_code", authCode)
                }.toString().toRequestBody("application/json; charset=utf-8".toMediaTypeOrNull())
            )
            .build()
        val response = okHttpClient.newCall(request).await()
        val jsonString = response.body?.string() ?: "{}"
        if (response.code >= 400) {
            val message = parseErrorBody(jsonString)
            sendErrorEvent(appId, "login", BoxoLoginException(message, jsonString))
            throw BoxoLoginException(message, jsonString)
        } else {
            return runCatching {
                jsonString
            }.onFailure {
                sendErrorEvent(appId, "login", BoxoException(jsonString))
            }.getOrThrow()
        }
    }

    suspend fun sendSessionStart(appId: String, startDate: Long) {
        sendSession("session_start", appId, JSONObject().apply {
            put("start_date", startDate)
            put("device_id", storage.guid)
        })
    }

    suspend fun sendSessionEnd(appId: String, startDate: Long, endDate: Long) {
        sendSession("session_end", appId, JSONObject().apply {
            put("start_date", startDate)
            put("end_date", endDate)
            put("device_id", storage.guid)
        })
    }

    private suspend fun sendSession(action: String, appId: String, payload: JSONObject) {
        val jsonObject = JSONObject()
        jsonObject.apply {
            put("app_id", appId)
            put("client_id", boxo.config.clientId)
            put("action", action)
            put("payload", payload)
            addPlatformDetails()
        }

        val request = Request.Builder()
            .post(
                jsonObject.toString()
                    .toRequestBody("application/json; charset=utf-8".toMediaTypeOrNull())
            )
            .url(
                "$host/api/v1/analytics/track/".toHttpUrlOrNull()!!
                    .newBuilder()
                    .addQueryParameter("hash", jsonObject.hash())
                    .build()
            )
            .build()
        val response = okHttpClient.newCall(request).await()
        val jsonString = response.body?.string() ?: "{}"
        if (response.code >= 400) {
            sendErrorEvent(appId, action, BoxoException(jsonString))
        }
    }

    override suspend fun sendErrorEvent(
        appId: String,
        message: String,
        e: Throwable
    ): Boolean {
        val jsonObject = JSONObject()
        jsonObject.apply {
            put("message", e.javaClass.name)
            put("exception", JSONArray().apply {
                put(JSONObject().apply {
                    put("type", message)
                    put("value", e.stackTraceToString())
                })
            })
            put("extra", JSONObject().apply {
                put("app_id", appId)
                put("client_id", boxo.config.clientId)
                addPlatformDetails()
            })
        }
        val request = Request.Builder()
            .post(
                jsonObject.toString()
                    .toRequestBody("application/json; charset=utf-8".toMediaTypeOrNull())
            )
            .header(
                "X-Sentry-Auth",
                "Sentry sentry_version=7, sentry_key=161b9a5f1addfd5992c2d4e9e9113fbf, sentry_client=androidSDK/$sdkVersion"
            )
            .url("https://sentry.io/api/4507649919614976/store/")
            .build()
        val response = okHttpClient.newCall(request).await()
        return response.code < 400
    }

    fun parseErrorBody(jsonString: String): String {
        val json = JSONObject(jsonString)
        var message = ""
        runCatching {
            json.keys().forEach { key ->
                if (key == "message" && json.optJSONArray(key) == null) {
                    message += (json.optString(key) ?: "") + "\n"
                } else {
                    val array = json.getJSONArray(key)
                    var str = ""
                    for (i in 0 until array.length())
                        str += array.getString(i) + " "
                    message += (if (key == "message") str.trim() else "$key: ${str.trim()}") + "\n"
                }
            }
        }
        return if (message.isBlank()) jsonString else message.trim()
    }

    suspend fun sendTransactionEvent(appId: String, payload: String): Boolean {
        val jsonObject = JSONObject()
        jsonObject.apply {
            put("app_id", appId)
            put("client_id", boxo.config.clientId)
            put("action", "transaction")
            put("payload", JSONObject(payload))
            addPlatformDetails()
        }

        val request = Request.Builder()
            .post(
                jsonObject.toString()
                    .toRequestBody("application/json; charset=utf-8".toMediaTypeOrNull())
            )
            .url(
                "${host}/api/v1/analytics/track/".toHttpUrlOrNull()!!
                    .newBuilder()
                    .addQueryParameter("hash", jsonObject.hash())
                    .build()
            )
            .build()
        val response = okHttpClient.newCall(request).await()
        val jsonString = response.body?.string() ?: "{}"
        if (response.code >= 400) {
            sendErrorEvent(appId, "js_transaction_track", BoxoException(jsonString))
        }
        return response.code < 400
    }

    private fun JSONObject.addPlatformDetails() {
        put("platform", "Android")
        put("version", "Android ${Build.VERSION.RELEASE}")
        put("model", "${Build.BRAND} ${Build.MODEL}")
        put("sdk_version", sdkVersion)
    }

    private fun JSONObject.hash(): String {
        return (salt() + toString()).sha256()
    }

    private fun String.sha256(): String {
        return MessageDigest
            .getInstance("SHA-256")
            .digest(toByteArray())
            .fold("") { str, it -> str + "%02x".format(it) }
    }

    private fun JSONObject.salt(): String {
        return (keys().asSequence().toList() +
                getJSONObject("payload").keys().asSequence().toList())
            .toMutableList()
            .apply { sort() }
            .joinToString(separator = "")
            .sha256()
    }
}