package io.boxo.events

import io.boxo.R
import io.boxo.data.storage.IBoxoPermissions
import io.boxo.data.storage.IBoxoStorage
import io.boxo.js.JSFunctions
import io.boxo.js.events.IAuthEvents
import io.boxo.sdk.Boxo
import io.boxo.ui.dialog.BoxoErrorDialog
import io.boxo.ui.dialog.BoxoAuthRequestDialog
import io.boxo.ui.dialog.LoadingDialog
import io.boxo.ui.main.BoxoActivity
import io.boxo.ui.main.BoxoViewModel
import io.boxo.ui.main.states.LoginState
import org.json.JSONObject

internal class AuthEvents(
    private val activity: BoxoActivity,
    private val viewModel: BoxoViewModel,
    private val jsFunctions: JSFunctions,
    private val storage: IBoxoStorage,
    private val permissionsStorage: IBoxoPermissions,
    private val evaluateJavascript: (String) -> Unit
) : IAuthEvents {

    private val loadingDialog by lazy { LoadingDialog(activity) }
    private var loginRequestId: String = ""
    private var loginInProgress = false

    override fun logout(requestId: String) {
        storage.clearToken(viewModel.appId)
        evaluateJavascript(jsFunctions.logout(requestId))
    }

    override fun login(requestId: String, message: String, requestedData: List<String>) {
        loginRequestId = requestId
        if (loginInProgress.not()) {
            loginInProgress = true
            if (viewModel.miniappSettings.hostappInfo.isConsentEnabled.not() ||
                permissionsStorage.hasUserDataPermission(viewModel.appId)
            ) {
                viewModel.miniapp.authListener?.onAuth(activity, viewModel.miniapp)
                loginInProgress = false
                loadingDialog.show(true)
            } else {
                BoxoAuthRequestDialog(viewModel.miniappSettings).apply {
                    setOnCancelListener {
                        loginInProgress = false
                        evaluateJavascript(jsFunctions.login(requestId, false))
                    }
                    setOnAllowClickListener {
                        viewModel.miniapp.authListener?.onAuth(
                            this@AuthEvents.activity,
                            viewModel.miniapp
                        )
                        dismiss()
                        loginInProgress = false
                        loadingDialog.show(true)
                    }
                }.show(activity.supportFragmentManager, null)
            }
        }
    }

    fun handleLoginState(loginState: LoginState) {
        loginInProgress = false
        when (loginState.state) {
            LoginState.State.LOADING -> {
                loadingDialog.show(true)
            }

            LoginState.State.ERROR -> {
                BoxoErrorDialog(
                    viewModel.miniappSettings,
                    if (Boxo.config.isSandbox) loginState.message else
                        activity.getString(R.string.boxo_login_unknown_error)
                ).apply {
                    setOnCancelListener {
                        evaluateJavascript(
                            jsFunctions.login(
                                loginRequestId,
                                false,
                                loginState.data
                            )
                        )
                    }
                }.show(activity.supportFragmentManager, null)
            }

            LoginState.State.SUCCESS -> {
                runCatching { JSONObject(loginState.data).getString("token") }
                    .onSuccess { storage.saveMiniappToken(viewModel.appId, it) }
                evaluateJavascript(jsFunctions.login(loginRequestId, true, loginState.data))
            }

            LoginState.State.IDLE -> {
                loadingDialog.show(false)
            }
        }
    }
}