package io.boxo.events

import android.annotation.SuppressLint
import android.content.Context
import android.location.LocationManager
import io.boxo.R
import io.boxo.data.storage.IBoxoPermissions
import io.boxo.js.JSFunctions
import io.boxo.js.events.ILocationEvents
import io.boxo.js.params.LocationParam
import io.boxo.ui.dialog.BoxoPermissionDialog
import io.boxo.ui.dialog.Permission
import io.boxo.ui.main.BoxoActivity
import io.boxo.ui.main.BoxoViewModel
import io.boxo.utils.ActivityResultHelper
import io.boxo.utils.Permissions
import io.boxo.utils.extensions.showDialog
import com.google.android.gms.location.LocationCallback
import com.google.android.gms.location.LocationRequest
import com.google.android.gms.location.LocationResult
import com.google.android.gms.location.LocationServices
import com.google.android.gms.location.Priority

internal class LocationEvents(
    private val activity: BoxoActivity,
    private val viewModel: BoxoViewModel,
    private val permissionsStorage: IBoxoPermissions,
    private val permissions: Permissions,
    private val jsFunctions: JSFunctions,
    private val forResult: ActivityResultHelper,
    private val evaluateJavascript: (String) -> Unit
) : ILocationEvents {

    private val locationManager by lazy { activity.getSystemService(Context.LOCATION_SERVICE) as LocationManager }
    private val isLocationEnabled: Boolean
        get() = locationManager.isProviderEnabled(LocationManager.GPS_PROVIDER)
    private val fusedLocationClient by lazy { LocationServices.getFusedLocationProviderClient(activity) }
    private var locationRequestId: String = ""

    override fun getGeolocation(requestId: String) {
        locationRequestId = requestId
        if (permissionsStorage.hasLocationPermission(viewModel.appId))
            requestSystemLocationPermission()
        else {
            BoxoPermissionDialog(viewModel.miniappSettings.miniappInfo, Permission.LOCATION)
                .setOnOkClickListener { requestSystemLocationPermission() }
                .setOnDismissClickListener { evaluateJavascript(jsFunctions.sendGeolocation(locationRequestId, null)) }
                .show(activity.supportFragmentManager, "permission_dialog")
        }
    }

    @SuppressLint("MissingPermission")
    private fun requestSystemLocationPermission() {
        permissionsStorage.allowLocationPermission(viewModel.appId)
        permissions.location(onAccess = {
            if (isLocationEnabled) {
                fusedLocationClient.lastLocation
                    .addOnCompleteListener {
                        if (it.result != null) {
                            evaluateJavascript(jsFunctions.sendGeolocation(locationRequestId, it.result))
                        } else {
                            fusedLocationClient.requestLocationUpdates(
                                LocationRequest.Builder(5 * 1000)
                                    .setPriority(Priority.PRIORITY_HIGH_ACCURACY).build(),
                                object : LocationCallback() {
                                    override fun onLocationResult(result: LocationResult) {
                                        evaluateJavascript(
                                            jsFunctions.sendGeolocation(
                                                locationRequestId,
                                                result.lastLocation
                                            )
                                        )
                                        fusedLocationClient.removeLocationUpdates(this)
                                    }
                                },
                                null
                            )
                        }
                    }
            } else {
                activity.showDialog(
                    message = activity.getString(R.string.boxo_system_location_disabled_text),
                    positive = activity.getString(R.string.boxo_open_location_settings_btn),
                    onNegativeClick = { evaluateJavascript(jsFunctions.sendGeolocation(locationRequestId, null)) },
                    onPositiveClick = { forResult.openLocationSettings { requestSystemLocationPermission() } }
                )
            }
        }, onDenied = {
            evaluateJavascript(jsFunctions.sendGeolocation(locationRequestId, null))
        }, onForeverDenied = {
            activity.showDialog(
                message = activity.getString(R.string.boxo_location_permission_denied_text),
                positive = activity.getString(R.string.boxo_open_settings_btn),
                onNegativeClick = { evaluateJavascript(jsFunctions.sendGeolocation(locationRequestId, null)) },
                onPositiveClick = { forResult.openSettings { requestSystemLocationPermission() } }
            )
        })
    }

    override fun chooseLocation(requestId: String) {
        forResult.openChooseLocation { lat, lng -> evaluateJavascript(jsFunctions.chooseLocation(requestId, lat, lng)) }
    }

    override fun openLocation(location: LocationParam) {
        forResult.openOpenLocation(location) {
            evaluateJavascript(
                jsFunctions.openLocationResult(
                    location.requestId,
                    it
                )
            )
        }
    }
}