package io.boxo.events

import io.boxo.data.models.Colors
import io.boxo.data.storage.IBoxoStorage
import io.boxo.js.JSFunctions
import io.boxo.js.events.IMiniappEvents
import io.boxo.sdk.Boxo
import io.boxo.sdk.Config
import io.boxo.sdk.Miniapp
import io.boxo.sdk.MiniappConfig
import io.boxo.ui.main.BoxoActivity
import io.boxo.ui.main.BoxoViewModel
import io.boxo.utils.toMap
import io.boxo.utils.toStringMap
import org.json.JSONObject

internal class MiniappEvents(
    private val activity: BoxoActivity,
    private val viewModel: BoxoViewModel,
    private val JS: JSFunctions,
    private val storage: IBoxoStorage,
    private val evaluateJavascript: (String) -> Unit,
    private val boxo: Boxo
) : IMiniappEvents {

    override fun getMiniappSettings(requestId: String) {
        evaluateJavascript(
            JS.getMiniAppSettings(
                requestId,
                viewModel.miniappSettings,
                viewModel.miniapp.config
            )
        )
    }

    override fun getInitialData(requestId: String) {
        evaluateJavascript(
            JS.getInitialData(
                requestId,
                storage.getMiniappToken(viewModel.appId) ?: "",
                viewModel
            )
        )
    }

    override fun closeMiniapp() = activity.closeMiniapp()

    private fun getMiniapp(appId: String): Miniapp {
        return boxo.getMiniapp(appId)
            .setCustomEventListener(viewModel.miniapp.customEventListener)
            .setAuthListener(viewModel.miniapp.authListener)
            .setPaymentEventListener(viewModel.miniapp.paymentEventListener)
            .setLifecycleListener(viewModel.miniapp.lifecycleListener)
            .setUrlChangeListener(viewModel.miniapp.urlChangeListener)
            .setCustomActionMenuItemClickListener(viewModel.miniapp.customActionMenuItemClickListener)
    }

    override fun openNewMiniapp(
        appId: String,
        json: JSONObject
    ) {
        val data = json.optJSONObject("data")?.toMap() ?: emptyMap()
        val urlSuffix: String = json.optString("url_suffix", "")
        val saveState: Boolean = json.optBoolean("save_state", true)
        val isSplashEnabled = json.optBoolean("is_splash_enabled", true)
        val extraParams = json.optJSONObject("extra_params")?.toStringMap() ?: emptyMap()
        val colors = json.optJSONObject("colors")?.let {
            Colors(
                it.getString("primary_color"),
                it.getString("secondary_color"),
                it.getString("tertiary_color")
            )
        }
        val theme = when (json.optString("theme")) {
            "dark" -> Config.Theme.DARK
            "light" -> Config.Theme.LIGHT
            else -> Config.Theme.SYSTEM
        }

        getMiniapp(appId)
            .setData(data)
            .setConfig(
                MiniappConfig.Builder().apply {
                    setTheme(theme)
                    enableSplash(isSplashEnabled)
                    saveState(saveState)
                    colors?.also {
                        setColors(
                            it.primaryColor,
                            it.secondaryColor,
                            it.tertiaryColor
                        )
                    }
                    viewModel.miniapp.config?.customActionMenuItemIcon?.also {
                        setCustomActionMenuItem(it)
                    }
                    setExtraUrlParams(extraParams)
                    setUrlSuffix(urlSuffix)
                }.build()
            )
            .open(activity)
    }

}