package io.boxo.events

import android.content.Context
import android.hardware.Sensor
import android.hardware.SensorEvent
import android.hardware.SensorEventListener
import android.hardware.SensorManager
import io.boxo.js.JSFunctions
import io.boxo.js.events.ISensorEvents
import io.boxo.ui.main.BoxoActivity
import io.boxo.utils.Compass

internal class SensorEvents(
    private val activity: BoxoActivity,
    private val jsFunctions: JSFunctions,
    private val evaluateJavascript: (String) -> Unit
) : ISensorEvents {

    private val sensorManager by lazy { activity.getSystemService(Context.SENSOR_SERVICE) as SensorManager }
    private val compass: Compass? by lazy { Compass.newInstance(activity) }

    private val accelerometerListener = object : SensorEventListener {
        override fun onAccuracyChanged(sensor: Sensor?, accuracy: Int) {}
        override fun onSensorChanged(event: SensorEvent) {
            val (x, y, z) = event.values
            evaluateJavascript(jsFunctions.onAccelerometerChange(x, y, z))
        }
    }

    private val gyroscopeListener = object : SensorEventListener {
        override fun onAccuracyChanged(sensor: Sensor?, accuracy: Int) {}
        override fun onSensorChanged(event: SensorEvent) {
            val (x, y, z) = event.values
            evaluateJavascript(jsFunctions.onGyroscopeChange(x, y, z))
        }
    }

    override fun startAccelerometer(requestId: String, interval: Int) {
        val sensor: Sensor? = sensorManager.getDefaultSensor(Sensor.TYPE_ACCELEROMETER)
        evaluateJavascript(jsFunctions.startAccelerometer(requestId, sensor != null))
        val intervalInMicroseconds = if (interval < 5) 5000 else interval * 1000
        sensor?.also {
            sensorManager.registerListener(
                accelerometerListener, sensor,
                intervalInMicroseconds
            )
        }
    }

    override fun stopAccelerometer(requestId: String) {
        evaluateJavascript(jsFunctions.stopAccelerometer(requestId))
        sensorManager.unregisterListener(accelerometerListener)
    }

    override fun startGyroscope(requestId: String, interval: Int) {
        val sensor: Sensor? = sensorManager.getDefaultSensor(Sensor.TYPE_GYROSCOPE)
        evaluateJavascript(jsFunctions.startGyroscope(requestId, sensor != null))
        val intervalInMicroseconds = if (interval < 5) 5000 else interval * 1000
        sensor?.also {
            sensorManager.registerListener(
                gyroscopeListener, sensor,
                intervalInMicroseconds
            )
        }
    }

    override fun stopGyroscope(requestId: String) {
        evaluateJavascript(jsFunctions.stopGyroscope(requestId))
        sensorManager.unregisterListener(gyroscopeListener)
    }

    override fun startCompass(requestId: String) {
        compass?.onNewAzimuth =
            { azimuth -> evaluateJavascript(jsFunctions.onCompassChange(azimuth)) }
        evaluateJavascript(jsFunctions.startCompass(requestId, compass?.start() != null))
    }

    override fun stopCompass(requestId: String) {
        compass?.stop()
        evaluateJavascript(jsFunctions.stopCompass(requestId))
    }

    fun close() {
        compass?.stop()
        sensorManager.unregisterListener(gyroscopeListener)
        sensorManager.unregisterListener(accelerometerListener)
    }
}