package io.boxo.events

import android.app.DownloadManager
import android.content.Context
import android.content.Context.EUICC_SERVICE
import android.content.Intent
import android.content.res.Configuration
import android.content.res.Resources
import android.graphics.Point
import android.location.LocationManager
import android.net.Uri
import android.os.Build
import android.os.Environment
import android.telephony.euicc.EuiccManager
import android.view.WindowManager
import androidx.appcompat.app.AppCompatActivity.DOWNLOAD_SERVICE
import androidx.core.content.FileProvider
import androidx.lifecycle.lifecycleScope
import io.boxo.BuildConfig
import io.boxo.js.JSFunctions
import io.boxo.js.events.ISystemEvents
import io.boxo.ui.dialog.LoadingDialog
import io.boxo.ui.main.BoxoActivity
import io.boxo.utils.Permissions
import io.boxo.utils.UiConst
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import okhttp3.OkHttpClient
import okhttp3.Request
import java.io.File
import java.io.FileOutputStream

internal class SystemEvents(
    private val activity: BoxoActivity,
    private val permissions: Permissions,
    private val jsFunctions: JSFunctions,
    private val evaluateJavascript: (String) -> Unit
) : ISystemEvents {

    private val locationManager by lazy { activity.getSystemService(Context.LOCATION_SERVICE) as LocationManager }
    private val isLocationEnabled: Boolean
        get() = locationManager.isProviderEnabled(LocationManager.GPS_PROVIDER)
    private val loadingDialog by lazy { LoadingDialog(activity) }

    override fun getSystemInfo(requestId: String) {
        val outPoint = Point()
        (activity.application.getSystemService(Context.WINDOW_SERVICE) as WindowManager).defaultDisplay.getRealSize(
            outPoint
        )
        val (screenWidth, screenHeight) = if (outPoint.y > outPoint.x) outPoint.x to outPoint.y else outPoint.y to outPoint.x

        val systemUiMode =
            when (activity.resources.configuration.uiMode and Configuration.UI_MODE_NIGHT_MASK) {
                Configuration.UI_MODE_NIGHT_YES -> "dark"
                else -> "light"
            }

        @Suppress("DEPRECATION")
        val language = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
            activity.resources.configuration.locales[0].language
        } else {
            activity.resources.configuration.locale.language
        }

        val isSupportESim = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.P) {
            runCatching { activity.getSystemService(EUICC_SERVICE) as? EuiccManager }
                .getOrNull()?.isEnabled ?: false
        } else false

        val systemInfo = mapOf<String, Any>(
            "brand" to Build.BRAND,
            "model" to Build.DEVICE,
            "pixelRatio" to Resources.getSystem().displayMetrics.density,
            "screenWidth" to screenWidth,
            "screenHeight" to screenHeight,
            "windowWidth" to Resources.getSystem().displayMetrics.widthPixels,
            "windowHeight" to Resources.getSystem().displayMetrics.heightPixels,
            "statusBarHeight" to UiConst.statusBarHeight,
            "system" to "Android ${Build.VERSION.RELEASE}",
            "platform" to "Android",
            "SDKVersion" to BuildConfig.VERSION_NAME,
            "cameraAuthorized" to permissions.hasCameraPermission(),
            "locationAuthorized" to permissions.hasLocationPermission(),
            "locationEnabled" to isLocationEnabled,
            "systemUiMode" to systemUiMode,
            "language" to language,
            "isSupportESim" to isSupportESim
        )
        evaluateJavascript(jsFunctions.getSystemInfo(requestId, systemInfo))
    }

    override fun downloadFile(url: String, filename: String) {
        runCatching {
            val uri = Uri.parse(url)
            val request = DownloadManager.Request(uri).apply {
                setDestinationInExternalPublicDir(Environment.DIRECTORY_DOWNLOADS, filename)
                setNotificationVisibility(DownloadManager.Request.VISIBILITY_VISIBLE_NOTIFY_COMPLETED)
            }
            val downloadManager = activity.getSystemService(DOWNLOAD_SERVICE) as DownloadManager
            downloadManager.enqueue(request)
        }
    }

    override fun share(text: String, url: String, filename: String) {
        activity.lifecycleScope.launch(Dispatchers.IO) {
            if (url.isNotBlank() && filename.isNotBlank()) {
                activity.lifecycleScope.launch(Dispatchers.Main) { loadingDialog.show() }
                val boxoFolder = File(activity.cacheDir, "boxo")
                if (boxoFolder.exists().not()) boxoFolder.mkdir()
                val file = File(boxoFolder, filename)
                val success = file.download(url)
                if (success) {
                    val uri = FileProvider.getUriForFile(
                        activity,
                        "${activity.packageName}.boxo.provider",
                        file
                    )
                    val shareIntent = Intent().apply {
                        action = Intent.ACTION_SEND
                        putExtra(Intent.EXTRA_STREAM, uri)
                        if (text.isNotBlank())
                            putExtra(Intent.EXTRA_TEXT, text)
                        type = file.getMimeTypeFromFile()
                        addFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION)
                    }
                    activity.startActivity(Intent.createChooser(shareIntent, "Share"))
                }
                activity.lifecycleScope.launch(Dispatchers.Main) { loadingDialog.dismiss() }
            } else {
                val shareIntent = Intent().apply {
                    action = Intent.ACTION_SEND
                    putExtra(Intent.EXTRA_TEXT, text)
                }
                activity.startActivity(Intent.createChooser(shareIntent, "Share"))
            }
        }
    }

    private fun File.download(url: String): Boolean {
        val client = OkHttpClient()

        val request = Request.Builder()
            .url(url)
            .build()

        return try {
            val response = client.newCall(request).execute()
            if (response.isSuccessful) {
                FileOutputStream(this).use { fos ->
                    fos.write(response.body?.bytes())
                }
                true
            } else {
                false
            }
        } catch (e: Exception) {
            false
        }
    }

    private fun File.getMimeTypeFromFile(): String {
        return when (extension.lowercase()) {
            "jpg", "jpeg" -> "image/jpeg"
            "png" -> "image/png"
            "gif" -> "image/gif"
            "pdf" -> "application/pdf"
            "txt" -> "text/plain"
            "mp3" -> "audio/mpeg"
            "mp4" -> "video/mp4"
            else -> "*/*"
        }
    }
}