package io.boxo.events

import android.content.res.ColorStateList
import android.graphics.Color
import android.view.View
import android.widget.FrameLayout
import android.widget.ProgressBar
import android.widget.TextView
import androidx.core.view.isVisible
import io.boxo.R
import io.boxo.data.models.ActionButtonPosition
import io.boxo.data.storage.IBoxoPermissions
import io.boxo.js.JSFunctions
import io.boxo.js.events.IUIEvents
import io.boxo.js.params.ActionButton
import io.boxo.js.params.ActionSheet
import io.boxo.js.params.Alert
import io.boxo.js.params.NavigationBar
import io.boxo.js.params.PullToRefresh
import io.boxo.js.params.TabBar
import io.boxo.ui.dialog.BoxoPermissionDialog
import io.boxo.ui.dialog.LoadingDialog
import io.boxo.ui.dialog.Permission
import io.boxo.ui.dialog.show
import io.boxo.ui.main.BoxoActivity
import io.boxo.ui.main.BoxoViewModel
import io.boxo.ui.view.ActionButtonsView
import io.boxo.ui.view.BoxoAppBar
import io.boxo.ui.view.BottomTabLayout
import io.boxo.ui.view.FloatingActionButtonsView
import io.boxo.ui.view.PullToRefreshLayout
import io.boxo.ui.view.PullToRefreshLayout.Companion.TYPE_EDGE_TOP
import io.boxo.utils.ActivityResultHelper
import io.boxo.utils.UiConst
import io.boxo.utils.extensions.bindView
import io.boxo.utils.extensions.setMarginTop

internal class UiEvents(
    private val activity: BoxoActivity,
    private val viewModel: BoxoViewModel,
    private val jsFunctions: JSFunctions,
    private val permissionsStorage: IBoxoPermissions,
    private val forResult: ActivityResultHelper,
    private val evaluateJavascript: (String) -> Unit
) : IUIEvents {

    private val loadingDialog by lazy { LoadingDialog(activity) }
    private val root by activity.bindView<View>(R.id.root)
    private val bottomNav by activity.bindView<BottomTabLayout>(R.id.bottom_nav)
    private val appBar by activity.bindView<BoxoAppBar>(R.id.app_bar)
    private val actionButtons by activity.bindView<ActionButtonsView>(R.id.action_buttons)
    private val floatingActionButtons by activity.bindView<FloatingActionButtonsView>(R.id.floating_action_buttons)

    private val pullLayout by activity.bindView<PullToRefreshLayout>(R.id.pull_layout)
    private val pullContent by activity.bindView<FrameLayout>(R.id.pull_layout_container)
    private val pullText by activity.bindView<TextView>(R.id.pull_refresh_text)
    private val pullProgress by activity.bindView<ProgressBar>(R.id.pull_refresh_progress)

    fun onCreate() {
        appBar.post {
            if (viewModel.isFullscreen)
                appBar.setMarginTop(UiConst.statusBarHeight)
        }
        bottomNav.tabClickListener = ::onClickTab
        pullLayout.isEnabled = false
    }

    fun ready() {
        appBar.ready()
    }

    fun initAppbar(contentView: View) {
        appBar.post {
            appBar.init(contentView, viewModel.isFullscreen, viewModel.defaultStatusBar)
            appBar.setNavigationOnClickListener { activity.onBackPress() }
        }
    }

    override fun setTabBar(tabBar: TabBar) {
        bottomNav.set(tabBar)
    }

    override fun setNavigationBar(navigationBar: NavigationBar) {
        appBar.set(navigationBar)
    }

    override fun setActionButtons(actionButton: ActionButton) {
        if (viewModel.miniappSettings.actionButtonPosition == ActionButtonPosition.FLOATING) {
            floatingActionButtons.set(actionButton)
            actionButtons.isVisible = false
        } else {
            actionButtons.set(actionButton)
            floatingActionButtons.isVisible = false
        }

    }

    override fun setPullToRefresh(data: PullToRefresh) {
        pullLayout.isEnabled = data.enabled
        pullLayout.stop()
        if (data.enabled) {
            if (data.backgroundColor.isNotBlank()) runCatching {
                pullContent.setBackgroundColor(Color.parseColor(data.backgroundColor))
            }
            if (data.textColor.isNotBlank()) runCatching {
                val color = Color.parseColor(data.textColor)
                pullText.setTextColor(color)
                pullProgress.indeterminateTintList = ColorStateList.valueOf(color)
            }
            pullLayout.setOnPullListener { _, fraction, _ ->
                pullProgress.isVisible = false
                pullText.text = if ((fraction ?: 0f).compareTo(1.0) != 0)
                    data.pullText.ifBlank { activity.getString(R.string.boxo_pull_refresh_pull) }
                else
                    data.releaseText.ifBlank { activity.getString(R.string.boxo_pull_refresh_release) }
            }
            pullLayout.setOnTriggerListener {
                when (it) {
                    TYPE_EDGE_TOP -> {
                        pullProgress.isVisible = true
                        pullText.text = data.loadingText.ifBlank {
                            activity.getString(R.string.boxo_pull_refresh_loading)
                        }
                        evaluateJavascript(jsFunctions.startPullToRefresh())
                    }
                }
            }
        }
    }

    override fun stopPullToRefresh() {
        pullLayout.stop()
    }

    override fun loadingIndicator(show: Boolean) {
        loadingDialog.show(show)
    }

    override fun showQrCodeReader() {
        fun openScanner() {
            forResult.openQrCodeReader { qrCode ->
                if (qrCode != null)
                    evaluateJavascript(jsFunctions.getFuncQRCodeReaderResult(qrCode))
                else
                    evaluateJavascript(jsFunctions.getFuncQRCodeReaderFailed())
            }
            permissionsStorage.allowCameraPermission(viewModel.appId)
        }
        if (permissionsStorage.hasCameraPermission(viewModel.appId))
            openScanner()
        else {
            BoxoPermissionDialog(viewModel.miniappSettings.miniappInfo, Permission.CAMERA)
                .setOnOkClickListener { openScanner() }
                .setOnDismissClickListener { evaluateJavascript(jsFunctions.getFuncQRCodeReaderFailed()) }
                .show(activity.supportFragmentManager, "permission_dialog")
        }
    }

    override fun showActionSheet(actionSheet: ActionSheet) {
        actionSheet.show(activity) { id ->
            evaluateJavascript(jsFunctions.actionSheetItemClick(id))
        }
    }

    override fun showAlert(alert: Alert) {
        alert.show(activity) { requestId, id ->
            evaluateJavascript(jsFunctions.alertButtonClick(requestId, id))
        }
    }

    override fun showGallery(images: List<String>) {
        forResult.openImageGallery(images) { evaluateJavascript(jsFunctions.showImagesResult(it)) }
    }

    override fun setBackgroundColor(color: String) {
        root.setBackgroundColor(Color.parseColor(color))
    }

    override fun setStatusBarColor(color: String) {
        appBar.set(NavigationBar(background = color))
    }

    private fun onClickTab(tabId: Int) {
        evaluateJavascript(jsFunctions.getFunctionForTab(tabId))
    }
}