package io.boxo.js

import android.location.Location
import io.boxo.data.models.MiniappSettings
import io.boxo.js.params.CustomEvent
import io.boxo.js.params.PaymentData
import io.boxo.sdk.Boxo
import io.boxo.sdk.MiniappConfig
import io.boxo.ui.main.BoxoViewModel
import org.json.JSONArray
import org.json.JSONObject

internal class JSFunctions(val boxo: Boxo) {

    private fun jsFunc(name: String, data: String): String = """ 
                window.dispatchEvent(new window.CustomEvent('AppBoxoWebAppEvent', {
                    bubbles: false,
                    cancelable: false,
                    detail: {
                        type: '$name',
                        data: $data
                    }
                }))
                """

    fun getMiniAppSettings(
        requestId: String,
        settings: MiniappSettings,
        miniappConfig: MiniappConfig?
    ): String {
        return jsFunc(
            name = "AppBoxoWebAppGetMiniappSettings",
            data = JSONObject().apply {
                put("request_id", requestId)
                put("miniapp_settings", JSONObject().apply {
                    put("colors", JSONObject().apply {
                        put("primary_color", miniappConfig?.colors?.primaryColor ?: settings.colors.primaryColor)
                        put("secondary_color", miniappConfig?.colors?.secondaryColor ?: settings.colors.secondaryColor)
                        put("tertiary_color", miniappConfig?.colors?.tertiaryColor ?: settings.colors.tertiaryColor)
                    })
                    put("white_label", settings.whiteLabel)
                    put("hostapp", JSONObject().apply {
                        put("name", settings.hostappInfo.name)
                        put("logo", settings.hostappInfo.logo)
                    })
                })
            }.toString()
        )
    }

    fun getInitialData(requestId: String, token: String, viewModel: BoxoViewModel): String {
        val app = viewModel.miniapp
        val openMiniappDuration = viewModel.openMiniappDuration
        return jsFunc(
            name = "AppBoxoWebAppGetInitData",
            data = JSONObject().apply {
                put("request_id", requestId)
                put("app_id", app.appId)
                put("token", token)
                put("client_id", boxo.config.clientId)
                put("sandbox_mode", boxo.config.isSandbox)
                put("language", boxo.config.language)
                put("open_miniapp_duration", openMiniappDuration)
                app.data?.let {
                    put("data", JSONObject(it).apply { remove("appboxo_auth_token") })
                }
            }.toString()
        )
    }

    fun getFuncForCustomEvent(customEvent: CustomEvent): String {
        return jsFunc(
            name = "AppBoxoWebAppCustomEvent",
            data = customEvent.toJson()
        )
    }

    fun getFuncToSetNfcRecords(records: List<String>): String {
        return jsFunc(
            name = "AppBoxoWebGetNfcRecords",
            data = JSONObject().apply { put("records", JSONArray(records)) }.toString()
        )
    }

    fun logout(requestId: String): String {
        return jsFunc(
            name = "AppBoxoWebAppLogout",
            data = JSONObject().apply { put("request_id", requestId) }.toString()
        )
    }

    fun login(requestId: String, result: Boolean, json: String? = null): String {
        return jsFunc(
            name = "AppBoxoWebAppLogin",
            data = JSONObject().apply {
                put("request_id", requestId)
                json?.let { put("payload", JSONObject(it)) }
                if (!result) put("error_type", "auth_error")
            }.toString()
        )
    }

    fun getFuncForPaymentResult(data: PaymentData): String {
        return jsFunc(
            name = "AppBoxoWebAppPay",
            data = data.toJson()
        )
    }

    fun onRestore(): String {
        return """ 
                window.dispatchEvent(new window.CustomEvent('AppBoxoWebAppEvent', {
                    bubbles: false,
                    cancelable: false,
                    detail: {
                        type: 'AppBoxoWebAppOnRestore'
                    }
                }))
                """
    }

    fun transactionTrack(requestId: String, result: Boolean): String {
        return jsFunc(
            name = "AppBoxoWebAppTrack",
            data = JSONObject().apply {
                put("request_id", requestId)
                put("result", result)
                if (!result) put("error_type", "client_error")
            }.toString()
        )
    }

    fun getFunctionForTab(tabId: Int): String {
        return jsFunc(
            name = "AppBoxoWebAppTabBarItemClick",
            data = JSONObject().apply { put("tabId", tabId) }.toString()
        )
    }

    fun getFuncQRCodeReaderResult(qrCodeData: String): String {
        return jsFunc(
            name = "AppBoxoWebAppOpenQRCodeReaderResult",
            data = JSONObject().apply { put("code_data", qrCodeData) }.toString()
        )
    }

    fun getFuncQRCodeReaderFailed(): String {
        return jsFunc(
            name = "AppBoxoWebAppOpenQRCodeReaderFailed",
            data = JSONObject().apply {
                put("error_type", "error")
                put("error_data", "QR code read rejected")
            }.toString()
        )
    }

    fun actionSheetItemClick(id: Int): String {
        return jsFunc(
            name = "AppBoxoWebAppActionSheetItemClick",
            data = JSONObject().apply { put("id", id) }.toString()
        )
    }

    fun sendGeolocation(requestId: String, location: Location?): String {
        return jsFunc(
            name = "AppBoxoWebAppGetGeodata",
            data = JSONObject().apply {
                put("request_id", requestId)
                if (location != null) {
                    put("available", true)
                    put("lat", location.latitude)
                    put("long", location.longitude)
                } else {
                    put("available", false)
                }
            }.toString()
        )
    }

    fun alertButtonClick(requestId: String, id: Int): String {
        return jsFunc(
            name = "AppBoxoWebAppShowAlert",
            data = JSONObject().apply {
                put("request_id", requestId)
                put("id", id)
            }.toString()
        )
    }

    fun showImagesResult(result: Boolean): String {
        return jsFunc(
            name = "AppBoxoWebAppShowImages",
            data = JSONObject().apply { put("result", result) }.toString()
        )
    }

    fun storageGetResult(requestId: String, keys: Map<String, *>): String {
        return jsFunc(
            name = "AppBoxoWebAppStorageGet",
            data = JSONObject().apply {
                put("request_id", requestId)
                put("result", JSONObject().apply {
                    keys.forEach { (key, value) ->
                        if (value != null) put(key, value)
                    }
                })
            }.toString()
        )
    }

    fun storageGetKeysResult(requestId: String, keys: List<String>): String {
        return jsFunc(
            name = "AppBoxoWebAppStorageGetKeys",
            data = JSONObject().apply {
                put("request_id", requestId)
                put("keys", JSONArray(keys))
            }.toString()
        )
    }

    fun storageSuccessResult(requestId: String): String {
        return jsFunc(
            name = "AppBoxoWebAppStorageSet",
            data = JSONObject().apply {
                put("request_id", requestId)
                put("result", true)
            }.toString()
        )
    }

    fun setClipboardSuccess(requestId: String): String {
        return jsFunc(
            name = "AppBoxoWebAppSetClipboard",
            data = JSONObject().apply {
                put("request_id", requestId)
                put("result", true)
            }.toString()
        )
    }

    fun getClipboard(requestId: String, data: String): String {
        return jsFunc(
            name = "AppBoxoWebAppGetClipboard",
            data = JSONObject().apply {
                put("request_id", requestId)
                put("data", data)
            }.toString()
        )
    }

    fun getSystemInfo(requestId: String, systemInfo: Map<String, *>): String {
        return jsFunc(
            name = "AppBoxoWebAppGetSystemInfo",
            data = JSONObject(systemInfo).apply {
                put("request_id", requestId)
            }.toString()
        )
    }

    fun openLocationResult(requestId: String, result: Boolean): String {
        return jsFunc(
            name = "AppBoxoWebAppOpenLocation",
            data = JSONObject().apply {
                put("request_id", requestId)
                put("result", result)
            }.toString()
        )
    }

    fun chooseLocation(requestId: String, lat: Double, lng: Double): String {
        return jsFunc(
            name = "AppBoxoWebAppChooseLocation",
            data = JSONObject().apply {
                put("request_id", requestId)
                put("latitude", lat)
                put("longitude", lng)
            }.toString()
        )
    }

    fun startAccelerometer(requestId: String, supported: Boolean): String {
        return jsFunc(
            name = "AppBoxoWebAppStartAccelerometer",
            data = JSONObject().apply {
                put("request_id", requestId)
                put("result", supported)
            }.toString()
        )
    }

    fun stopAccelerometer(requestId: String): String {
        return jsFunc(
            name = "AppBoxoWebAppStopAccelerometer",
            data = JSONObject().apply {
                put("request_id", requestId)
                put("result", true)
            }.toString()
        )
    }

    fun onAccelerometerChange(x: Float, y: Float, z: Float): String {
        return jsFunc(
            name = "AppBoxoWebAppOnAccelerometerChange",
            data = JSONObject().apply {
                put("x", x)
                put("y", y)
                put("z", z)
            }.toString()
        )
    }

    fun startGyroscope(requestId: String, supported: Boolean): String {
        return jsFunc(
            name = "AppBoxoWebAppStartGyroscope",
            data = JSONObject().apply {
                put("request_id", requestId)
                put("result", supported)
            }.toString()
        )
    }

    fun stopGyroscope(requestId: String): String {
        return jsFunc(
            name = "AppBoxoWebAppStopGyroscope",
            data = JSONObject().apply {
                put("request_id", requestId)
                put("result", true)
            }.toString()
        )
    }

    fun onGyroscopeChange(x: Float, y: Float, z: Float): String {
        return jsFunc(
            name = "AppBoxoWebAppOnGyroscopeChange",
            data = JSONObject().apply {
                put("x", x)
                put("y", y)
                put("z", z)
            }.toString()
        )
    }

    fun startCompass(requestId: String, result: Boolean): String {
        return jsFunc(
            name = "AppBoxoWebAppStartCompass",
            data = JSONObject().apply {
                put("request_id", requestId)
                put("result", result)
            }.toString()
        )
    }

    fun stopCompass(requestId: String): String {
        return jsFunc(
            name = "AppBoxoWebAppStopCompass",
            data = JSONObject().apply {
                put("request_id", requestId)
                put("result", true)
            }.toString()
        )
    }

    fun onCompassChange(azimuth: Int): String {
        return jsFunc(
            name = "AppBoxoWebAppOnCompassChange",
            data = JSONObject().apply { put("direction", azimuth) }.toString()
        )
    }

    fun startPullToRefresh(): String {
        return jsFunc(
            name = "AppBoxoWebAppStartPullToRefresh",
            data = JSONObject().toString()
        )
    }

    val removeCookies = """
        function deleteAllCookies() {
            const cookies = document.cookie.split(";");
            for (var i = 0; i < cookies.length; i++) {
                const cookie = cookies[i];
                const eqPos = cookie.indexOf("=");
                const name = eqPos > -1 ? cookie.substr(0, eqPos) : cookie;
                document.cookie = name + "=;expires=Thu, 01 Jan 1970 00:00:00 GMT";
            }
        };
        deleteAllCookies();
    """
}