package io.boxo.js.jsInterfaces

import android.os.Handler
import io.boxo.js.events.IStorageEvents
import io.boxo.log.exceptions.InvalidJsonException
import io.boxo.sdk.Boxo
import org.json.JSONObject

internal class StorageJsI(
    private val appId: String,
    private val handler: Handler,
    private val events: IStorageEvents
) {
    fun storageGet(body: String) {
        runCatching {
            val json = JSONObject(body)
            val requestId = json.getString("request_id")
            val keysJson = json.getJSONArray("keys")
            val keys = mutableListOf<String>()
            for (i in 0 until keysJson.length()) keys.add(keysJson.getString(i))
            Pair(requestId, keys.toTypedArray())
        }
            .onFailure { Boxo.logger.error(appId, InvalidJsonException(it)) }
            .onSuccess { (requestId, keys) ->
                handler.post { events.storageGet(requestId, keys) }
            }
    }

    fun storageGetKeys(body: String) {
        runCatching {
            val json = JSONObject(body)
            val requestId = json.getString("request_id")
            val count = runCatching { json.getInt("count") }.getOrThrow()
            val offset = runCatching { json.getInt("offset") }.getOrDefault(0)
            Triple(requestId, count, offset)
        }
            .onFailure { Boxo.logger.error(appId, InvalidJsonException(it)) }
            .onSuccess { (requestId, count, offset) ->
                handler.post { events.storageGetKeys(requestId, count, offset) }
            }
    }

    fun storageSet(body: String) {
        runCatching {
            val json = JSONObject(body)
            val requestId = json.getString("request_id")
            val key = runCatching { json.getString("key") }.getOrThrow()
            val value: Any = runCatching { json.get("value") }.getOrThrow()
            Triple(requestId, key, value)
        }
            .onFailure { Boxo.logger.error(appId, InvalidJsonException(it)) }
            .onSuccess { (requestId, key, value) ->
                handler.post { events.storageSet(requestId, key, value) }
            }
    }

    fun storageRemove(body: String) {
        runCatching {
            val json = JSONObject(body)
            val requestId = json.getString("request_id")
            val key = runCatching { json.getString("key") }.getOrThrow()
            Pair(requestId, key)
        }
            .onFailure { Boxo.logger.error(appId, InvalidJsonException(it)) }
            .onSuccess { (requestId, key) -> handler.post { events.storageRemove(requestId, key) } }
    }

    fun storageClear(body: String) {
        runCatching {
            val json = JSONObject(body)
            json.getString("request_id")
        }
            .onFailure { Boxo.logger.error(appId, InvalidJsonException(it)) }
            .onSuccess { requestId -> handler.post { events.storageClear(requestId) } }
    }
}