package io.boxo.js.params

import android.os.Parcelable
import kotlinx.parcelize.Parcelize
import org.json.JSONObject

@Parcelize
data class TabBar(
    val show: Boolean? = null,
    val activeTab: Int? = null,
    val list: Array<Tab>? = null,
    val badges: Array<Badge>? = null,
    val options: Options? = null
) : Parcelable {
    fun merge(tabBar: TabBar): TabBar {
        var newTabBar = copy()
        tabBar.show?.let { newTabBar = newTabBar.copy(show = it) }
        tabBar.activeTab?.let { newTabBar = newTabBar.copy(activeTab = it) }
        tabBar.list?.let { newTabBar = newTabBar.copy(list = it) }
        tabBar.badges?.let { newTabBar = newTabBar.copy(badges = it) }
        tabBar.options?.let { newOptions ->
            var options = newTabBar.options?.copy() ?: Options()
            newOptions.background?.let { options = options.copy(background = it) }
            newOptions.color?.let { options = options.copy(color = it) }
            newOptions.selectedColor?.let { options = options.copy(selectedColor = it) }
            newOptions.hasBorder?.let { options = options.copy(hasBorder = it) }
            newOptions.borderColor?.let { options = options.copy(borderColor = it) }
            newTabBar = newTabBar.copy(options = options)
        }
        return newTabBar
    }

    @Parcelize
    data class Tab(
        val tabId: Int,
        val tabName: String,
        val tabIcon: String
    ) : Parcelable

    @Parcelize
    data class Badge(
        val tabId: Int,
        val background: String,
        val color: String,
        val value: String?
    ) : Parcelable

    @Parcelize
    data class Options(
        val background: String? = null,
        val color: String? = null,
        val selectedColor: String? = null,
        val hasBorder: Boolean? = null,
        val borderColor: String? = null
    ) : Parcelable
}

internal fun JSONObject.toTabBar(): TabBar {
    return TabBar(
        show = runCatching { getBoolean("show") }.getOrNull(),
        activeTab = runCatching { getInt("activeTab") }.getOrNull(),
        list = runCatching {
            val jsonArray = getJSONArray("list")
            val tabs = mutableListOf<TabBar.Tab>()
            for (i in 0 until jsonArray.length()) {
                (jsonArray.get(i) as JSONObject).apply {
                    tabs.add(
                        TabBar.Tab(
                            tabId = runCatching { getInt("tabId") }.getOrDefault(0),
                            tabName = runCatching { getString("tabName") }.getOrDefault(""),
                            tabIcon = runCatching { getString("tabIcon") }.getOrDefault("")
                        )
                    )
                }
            }
            tabs.toTypedArray()
        }.getOrNull(),
        badges = runCatching {
            val jsonArray = getJSONArray("badges")
            val badges = mutableListOf<TabBar.Badge>()
            for (i in 0 until jsonArray.length()) {
                (jsonArray.get(i) as JSONObject).apply {
                    badges.add(
                        TabBar.Badge(
                            tabId = runCatching { getInt("tabId") }.getOrDefault(0),
                            background = runCatching { getString("background") }.getOrDefault(""),
                            color = runCatching { getString("color") }.getOrDefault(""),
                            value = runCatching { getString("value") }.getOrNull()
                        )
                    )
                }
            }
            badges.toTypedArray()
        }.getOrNull(),
        options = runCatching {
            val optionsJson = getJSONObject("options")
            optionsJson.run {
                TabBar.Options(
                    color = runCatching { getString("color") }.getOrNull(),
                    selectedColor = runCatching { getString("selectedColor") }.getOrNull(),
                    background = runCatching { getString("background") }.getOrNull(),
                    hasBorder = runCatching { getBoolean("hasBorder") }.getOrNull(),
                    borderColor = runCatching { getString("borderColor") }.getOrNull()
                )
            }
        }.getOrNull()
    )
}