package io.boxo.sdk

import android.app.Activity
import android.app.ActivityManager
import android.app.Application
import android.content.Context
import android.os.Bundle
import io.boxo.ServiceLocator
import io.boxo.log.DefaultLogger
import io.boxo.log.Logger
import io.boxo.ui.main.BoxoActivity
import kotlinx.coroutines.GlobalScope
import kotlinx.coroutines.launch
import java.io.File
import java.lang.ref.WeakReference

object Boxo {

    private val miniapps = hashMapOf<String, Miniapp>()
    private var _config: Config = Config.Builder().build()
    val config
        get() = _config

    private var _logger: Logger = DefaultLogger(false)

    internal val logger
        get() = ServiceLocator.logger

    private var topActivity: WeakReference<Activity>? = null

    internal val miniappStates = mutableMapOf<String, Bundle?>()

    private lateinit var application: Application


    /**
     * Set the global config
     */
    fun setConfig(config: Config): Boxo {
        _config = config
        return this
    }

    /**
     * Set logger.
     * by default is
     * @see DefaultLogger
     * @see Logger
     * Use Logger for custom implementation
     */
    fun setLogger(logger: Logger): Boxo {
        _logger = logger
        return this
    }

    /**
     * This method must be called on hostapp logout.
     */
    fun logout() {
        val activityManager =
            application.getSystemService(Context.ACTIVITY_SERVICE) as ActivityManager
        activityManager.appTasks
            .filter { it.taskInfo.baseIntent.hasExtra(BoxoActivity.APP_ID) }
            .forEach { it.finishAndRemoveTask() }

        val sharedPreferenceFile = File(application.applicationInfo.dataDir, "shared_prefs")
        val listFiles: Array<File>? = sharedPreferenceFile.listFiles()
        if (listFiles.isNullOrEmpty()) return
        for (file in listFiles) {
            if (file.name.startsWith("BoxoSDK_")) {
                val name = file.name.removeSuffix(".xml")
                application.getSharedPreferences(name, Context.MODE_PRIVATE).edit().clear().apply()
            }
        }
    }

    /**
     * Hide opened miniapps.
     */
    fun hideMiniapps() {
        val activityManager =
            application.getSystemService(Context.ACTIVITY_SERVICE) as ActivityManager
        val backActivities = activityManager.appTasks
            .filter { it.taskInfo.baseIntent.hasExtra(BoxoActivity.APP_ID).not() }
        if (backActivities.isNotEmpty()) {
            val task = backActivities.first()
            task.moveToFront()
            topActivity?.get()?.overridePendingTransition(0, 0)
        }
    }

    fun getMiniapps(callback: MiniappListCallback) {
        ServiceLocator.networkService.getMiniapps(callback)
    }

    fun prepareMiniapps(appIds: List<String>, callback: PreparedMiniappsCallback) {
        GlobalScope.launch {
            val preparedList = mutableListOf<Map<String, String>>()
            appIds.forEach { appId ->
                runCatching {
                    ServiceLocator.networkService.getMiniappSettings(appId, "")
                }.onSuccess { settings ->
                    ServiceLocator.storage.saveMiniappSettings(appId, settings)
                    preparedList.add(mapOf("app_id" to appId, "url" to settings.miniappInfo.url))
                }
            }
            callback.onFinish(preparedList)
        }
    }

    /**
     * get or create miniapp
     * @param appId id of miniapp
     */
    fun getMiniapp(appId: String): Miniapp {
        val id = appId.replace("/", "")
        val app = miniapps[id]
        return if (app == null) {
            val miniApp = Miniapp(id)
            miniapps[id] = miniApp
            miniApp
        } else {
            app
        }
    }

    /**
     * get created miniapp.
     * @param appId id of miniapp
     * @return miniapp or null
     */
    fun getExistingMiniapp(appId: String): Miniapp? = miniapps[appId]

    internal fun destroy(appId: String) {
        if (miniapps[appId]?.config?.saveState == false)
            miniappStates.remove(appId)
        miniapps.remove(appId)
    }

    /**
     * This method is must be called in your Application.onCreate
     */
    fun init(application: Application): Boxo {
        ServiceLocator.init(application)
        this.application = application
        application.registerActivityLifecycleCallbacks(object :
            Application.ActivityLifecycleCallbacks {
            override fun onActivityPaused(activity: Activity) {}
            override fun onActivityStarted(activity: Activity) {}
            override fun onActivityDestroyed(activity: Activity) {}
            override fun onActivitySaveInstanceState(activity: Activity, outState: Bundle) {}
            override fun onActivityStopped(activity: Activity) {}
            override fun onActivityCreated(activity: Activity, savedInstanceState: Bundle?) {}

            override fun onActivityResumed(activity: Activity) {
                topActivity = WeakReference(activity)
            }
        })
        return this
    }
}