package io.boxo.sdk

class Config private constructor(
    val clientId: String = "",
    val userId: String = "",
    val isMultitaskMode: Boolean = false,
    val theme: Theme = Theme.SYSTEM,
    val isDebug: Boolean = false,
    val isStaging: Boolean = false,
    val isSandbox: Boolean = false,
    val permissionsPage: Boolean = true,
    val showClearCache: Boolean = true,
    val showAboutPage: Boolean = true,
    val miniappSettingsExpirationTime: Int = 60,
    val language: String = "en"
) {
    fun toBuilder(): Builder = Builder()
        .setClientId(clientId)
        .setUserId(userId)
        .multitaskMode(isMultitaskMode)
        .setTheme(theme)
        .debug(isDebug)
        .sandboxMode(isSandbox)
        .staging(isStaging)
        .permissionsPage(permissionsPage)
        .showClearCache(showClearCache)
        .showAboutPage(showAboutPage)
        .setMiniappSettingsExpirationTime(miniappSettingsExpirationTime)
        .setLanguage(language)

    class Builder {
        private var clientId: String = ""
        private var userId: String = ""
        private var isMultitaskMode: Boolean = false
        private var isDebug: Boolean = false
        private var isSandbox: Boolean = false
        private var isStaging: Boolean = false
        private var theme: Theme = Theme.SYSTEM
        private var permissionsPage: Boolean = true
        private var showClearCache: Boolean = true
        private var showAboutPage: Boolean = true
        private var miniappSettingsExpirationTime: Int = 60
        private var language: String = "en"

        /**
         * Set the hostapp "clientId"
         */
        fun setClientId(clientId: String): Builder {
            this.clientId = clientId
            return this
        }

        /**
         * Set the "userId". By default UUID.randomUUID()
         */
        fun setUserId(id: String): Builder {
            this.userId = id
            return this
        }

        /**
         * When multi-task mode enabled,
         * a miniapp launches as a new task in the Recents screen
         * and if the miniapp in not closed, then it will be restored.
         */
        fun multitaskMode(enable: Boolean): Builder {
            this.isMultitaskMode = enable
            return this
        }

        /**
         * Set the global theme for miniapps.
         * @param theme by default Theme.SYSTEM
         */
        fun setTheme(theme: Theme): Builder {
            this.theme = theme
            return this
        }

        fun debug(enable: Boolean): Builder {
            this.isDebug = enable
            return this
        }

        fun sandboxMode(enable: Boolean): Builder {
            this.isSandbox = enable
            return this
        }

        fun staging(flag: Boolean): Builder {
            this.isStaging = flag
            return this
        }

        fun permissionsPage(enable: Boolean): Builder {
            permissionsPage = enable
            return this
        }

        fun showClearCache(enable: Boolean): Builder {
            showClearCache = enable
            return this
        }

        fun showAboutPage(enable: Boolean): Builder {
            showAboutPage = enable
            return this
        }

        fun setMiniappSettingsExpirationTime(sec: Int): Builder {
            miniappSettingsExpirationTime = sec
            return this
        }

        fun setLanguage(language: String): Builder {
            this.language = language
            return this
        }

        fun build(): Config {
            return Config(
                clientId,
                userId,
                isMultitaskMode,
                theme,
                isDebug,
                isStaging,
                isSandbox,
                permissionsPage,
                showClearCache,
                showAboutPage,
                miniappSettingsExpirationTime,
                language
            )
        }
    }

    enum class Theme { DARK, LIGHT, SYSTEM }
}