package io.boxo.sdk

import android.app.Activity
import android.app.ActivityManager
import android.content.Context
import android.content.Intent
import android.net.Uri
import android.webkit.WebView
import androidx.core.app.ActivityOptionsCompat
import io.boxo.R
import io.boxo.data.models.PageAnimation
import io.boxo.js.params.CustomEvent
import io.boxo.js.params.PaymentData
import io.boxo.ui.main.BoxoActivity

class Miniapp constructor(
    val appId: String,
    var data: Map<String, Any>? = null
) {
    val boxoConfig: Config
        get() = Boxo.config
    internal var customEventListener: CustomEventListener? = null
    internal var lifecycleListener: LifecycleListener? = null
    internal var urlChangeListener: UrlChangeListener? = null
    internal var customActionMenuItemClickListener: CustomActionMenuItemClickListener? = null
    internal var authListener: AuthListener? = null
    internal var paymentEventListener: PaymentEventListener? = null
    internal var customUrlLoadingHandler: CustomUrlLoadingHandler? = null

    internal var sendCustomEvent: ((customEvent: CustomEvent) -> Unit)? = null
    internal var onHideCustomActionMenuItem: (() -> Unit)? = null
    internal var onShowCustomActionMenuItem: (() -> Unit)? = null
    internal var onGetAuthCode: ((String) -> Unit)? = null
    internal var sendPaymentResult: ((paymentData: PaymentData) -> Unit)? = null
    internal var onDestroy: (() -> Unit)? = null
    internal var onNFCTagRead: ((data: List<String>) -> Unit)? = null

    internal var config: MiniappConfig? = null
    internal val theme
        get() = config?.theme ?: boxoConfig.theme

    internal var isClosing = false

    /**
     * Set the miniapp config
     * @see MiniappConfig.Builder
     */
    fun setConfig(config: MiniappConfig): Miniapp {
        this.config = config
        return this
    }

    /**
     * set custom data
     * @param data data to pass to miniapp.
     */
    fun setData(data: Map<String, Any>?): Miniapp {
        this.data = data
        return this
    }

    /**
     * Open the miniapp.
     * @see Config.isMultitaskMode when multi-task mode enabled,
     * the miniapp launches as a new task in the Recents screen
     * and if the miniapp in not closed, then it will be restored.
     */
    fun open(context: Context) {
        val activityManager = context.getSystemService(Context.ACTIVITY_SERVICE) as ActivityManager

        val task = activityManager.appTasks
            .find { it.taskInfo.baseIntent.getStringExtra(BoxoActivity.APP_ID) == appId }
        if (task != null) {
            task.moveToFront()
        } else {
            launchMiniappActivity(context)
        }

        lifecycleListener?.onLaunch(this)
    }

    private fun launchMiniappActivity(context: Context) {
        val intent = createIntent(context).apply {
            putExtra(BoxoActivity.APP_ID, appId)
            addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
            if (boxoConfig.isMultitaskMode) {
                addFlags(Intent.FLAG_ACTIVITY_NEW_DOCUMENT)
                addFlags(Intent.FLAG_ACTIVITY_MULTIPLE_TASK)
            }
        }
        val bundle = ActivityOptionsCompat.makeCustomAnimation(
            context, when (config?.pageAnimation) {
                PageAnimation.LEFT_TO_RIGHT -> R.anim.boxo_slide_left_right_in
                PageAnimation.RIGHT_TO_LEFT -> R.anim.boxo_slide_right_left_in
                PageAnimation.TOP_TO_BOTTOM -> R.anim.boxo_slide_top_bottom_in
                PageAnimation.FADE_IN -> R.anim.boxo_fade_in
                else -> R.anim.boxo_slide_bottom_top_in
            }, R.anim.boxo_no_animation
        ).toBundle()

        context.startActivity(intent, bundle)
    }

    /**
     * Set the lifecycle listener
     * @see LifecycleListener
     */
    fun setLifecycleListener(listener: LifecycleListener?): Miniapp {
        lifecycleListener = listener
        return this
    }

    /**
     * Set the custom event listener
     * @see CustomEventListener
     */
    fun setCustomEventListener(listener: CustomEventListener?): Miniapp {
        customEventListener = listener
        return this
    }

    /**
     *  Set the custom event listener
     * @see CustomEventListener
     */
    fun setCustomEventListener(listener: (BoxoActivity, Miniapp, CustomEvent) -> Unit): Miniapp {
        customEventListener = object : CustomEventListener {
            override fun handle(
                miniAppActivity: BoxoActivity,
                miniapp: Miniapp,
                customEvent: CustomEvent
            ) {
                listener.invoke(miniAppActivity, miniapp, customEvent)
            }
        }
        return this
    }

    /**
     * Send custom event
     * @see CustomEvent
     */
    fun sendEvent(customEvent: CustomEvent) {
        sendCustomEvent?.invoke(customEvent)
    }

    /**
     * Set the url change listener
     * @see UrlChangeListener
     */
    fun setUrlChangeListener(listener: UrlChangeListener?): Miniapp {
        urlChangeListener = listener
        return this
    }

    /**
     * Set the url change listener
     * @see UrlChangeListener
     */
    fun setUrlChangeListener(listener: (BoxoActivity, Miniapp, Uri) -> Unit): Miniapp {
        urlChangeListener = object : UrlChangeListener {
            override fun handle(miniAppActivity: BoxoActivity, miniapp: Miniapp, uri: Uri) {
                listener.invoke(miniAppActivity, miniapp, uri)
            }
        }
        return this
    }

    /**
     * Set the custom action menu item click listener
     * @see CustomActionMenuItemClickListener
     */
    fun setCustomActionMenuItemClickListener(listener: CustomActionMenuItemClickListener?): Miniapp {
        customActionMenuItemClickListener = listener
        return this
    }

    /**
     * Set the custom action menu item click listener
     * @see CustomActionMenuItemClickListener
     */
    fun setCustomActionMenuItemClickListener(listener: (BoxoActivity, Miniapp) -> Unit): Miniapp {
        customActionMenuItemClickListener = object : CustomActionMenuItemClickListener {
            override fun onClick(boxoActivity: BoxoActivity, miniapp: Miniapp) {
                listener.invoke(boxoActivity, miniapp)
            }
        }
        return this
    }

    fun setAuthListener(listener: AuthListener?): Miniapp {
        authListener = listener
        return this
    }

    fun setAuthListener(listener: (BoxoActivity, Miniapp) -> Unit): Miniapp {
        authListener = object : AuthListener {
            override fun onAuth(boxoActivity: BoxoActivity, miniapp: Miniapp) {
                listener.invoke(boxoActivity, miniapp)
            }
        }
        return this
    }

    /**
     * Set the payment event listener
     * @see PaymentEventListener
     */
    fun setPaymentEventListener(listener: PaymentEventListener?): Miniapp {
        paymentEventListener = listener
        return this
    }

    /**
     * Set the payment event listener
     * @see PaymentEventListener
     */
    fun setPaymentEventListener(listener: (BoxoActivity, Miniapp, PaymentData) -> Unit): Miniapp {
        paymentEventListener = object : PaymentEventListener {
            override fun handle(
                miniAppActivity: BoxoActivity,
                miniapp: Miniapp,
                paymentData: PaymentData
            ) {
                listener.invoke(miniAppActivity, miniapp, paymentData)
            }
        }
        return this
    }

    fun setCustomUrlLoadingHandler(handler: CustomUrlLoadingHandler): Miniapp {
        customUrlLoadingHandler = handler
        return this
    }

    /**
     * Send payment result
     * @see PaymentData
     */
    fun sendPaymentResult(paymentData: PaymentData) {
        sendPaymentResult?.invoke(paymentData)
    }

    /**
     * Show ActionMenu custom item
     */
    fun showCustomActionMenuItem() {
        onShowCustomActionMenuItem?.invoke()
    }

    /**
     * Hide ActionMenu custom item
     */
    fun hideCustomActionMenuItem() {
        onHideCustomActionMenuItem?.invoke()
    }

    fun setAuthCode(authCode: String) {
        onGetAuthCode?.invoke(authCode)
    }

    /**
     * Remove all listeners to avoid memory leak on your
     * @see Activity.onDestroy
     */
    fun removeAllListeners() {
        customEventListener = null
        lifecycleListener = null
        urlChangeListener = null
        paymentEventListener = null
        customActionMenuItemClickListener = null
        authListener = null
        customUrlLoadingHandler = null
    }

    /**
     * Close the miniapp.
     */
    fun close() {
        isClosing = true
        onDestroy?.invoke()
        removeAllListeners()

        sendCustomEvent = null
        sendPaymentResult = null
        onHideCustomActionMenuItem = null
        onShowCustomActionMenuItem = null
        onGetAuthCode = null
        onDestroy = null
        Boxo.destroy(appId)
    }

    private fun createIntent(context: Context): Intent {
        return Intent(context, BoxoActivity::class.java)
    }

    /**
     * This interface uses to handle custom events from miniapp.
     */
    interface CustomEventListener {
        /**
         * @param miniAppActivity use it to launch dialogs, activities, toasts...
         * @param miniapp current miniapp
         * @param customEvent event from miniapp
         */
        fun handle(miniAppActivity: BoxoActivity, miniapp: Miniapp, customEvent: CustomEvent)
    }

    /**
     * This interface uses to handle miniapp lifecycle
     */
    interface LifecycleListener {
        /**
         * Called when the miniapp will launch with Boxo.open(...)
         */
        fun onLaunch(miniapp: Miniapp)

        /**
         * Called when the miniapp will start interacting with the user
         */
        fun onResume(miniapp: Miniapp)

        /**
         * Called when the miniapp loses foreground state
         */
        fun onPause(miniapp: Miniapp)

        /**
         * Called when clicked close button in miniapp or when destroyed miniapp activity
         */
        fun onClose(miniapp: Miniapp)
        fun onError(miniapp: Miniapp, message: String)
        fun onUserInteraction(miniapp: Miniapp)
    }

    interface UrlChangeListener {
        fun handle(miniappActivity: BoxoActivity, miniapp: Miniapp, uri: Uri)
    }

    interface CustomActionMenuItemClickListener {
        fun onClick(boxoActivity: BoxoActivity, miniapp: Miniapp)
    }

    interface AuthListener {
        fun onAuth(boxoActivity: BoxoActivity, miniapp: Miniapp)
    }

    /**
     * This interface uses to handle payment events.
     */
    interface PaymentEventListener {
        /**
         * @param miniAppActivity use it to launch dialogs, activities, toasts...
         * @param miniapp current miniapp
         * @param paymentData payment data from miniapp
         */
        fun handle(miniAppActivity: BoxoActivity, miniapp: Miniapp, paymentData: PaymentData)
    }

    interface CustomUrlLoadingHandler {
        fun shouldOverrideUrlLoading(view: WebView?, url: String?): Boolean
    }
}
