package io.boxo.sdk

import androidx.annotation.DrawableRes
import io.boxo.data.models.Colors
import io.boxo.data.models.PageAnimation

/**
 * This class contains config for miniapp.
 * @property colors main colors in theme
 * @property theme miniapp theme
 * @property customActionMenuItemIcon Drawable resource icon of third button in action menu
 * @property extraParams - for adding extra query params to each request
 * @property urlSuffix - add to miniapp url suffix
 * @property isSplashEnabled - enable/disable miniapp loading page
 * @property saveState - enable/disable saving state on close miniapp
 */
class MiniappConfig private constructor(
    val colors: Colors?,
    val theme: Config.Theme?,
    val customActionMenuItemIcon: Int?,
    val extraParams: Map<String, String>,
    val urlSuffix: String,
    val isSplashEnabled: Boolean,
    val saveState: Boolean,
    val pageAnimation: PageAnimation
) {
    class Builder {
        private var colors: Colors? = null
        private var theme: Config.Theme? = null
        private var customActionMenuItemIcon: Int? = null
        private var extraParams: Map<String, String> = emptyMap()
        private var urlSuffix = ""
        private var isSplashEnabled = true
        private var saveState = true
        private var pageAnimation = PageAnimation.BOTTOM_TO_TOP

        /**
         * Set the colors for the miniaap,
         * if the miniaap supports color themes, the miniaap will appear with this color theme.
         */
        fun setColors(
            primaryColor: String,
            secondaryColor: String,
            tertiaryColor: String
        ): Builder {
            this.colors = Colors(primaryColor, secondaryColor, tertiaryColor)
            return this
        }

        /**
         * Set the theme to this miniapp.
         * If theme is not specified, then theme from global config will be used.
         * @see Boxo.setConfig(Config)
         * @param theme
         */
        fun setTheme(theme: Config.Theme): Builder {
            this.theme = theme
            return this
        }

        /**
         * Set the icon to third action menu item. The resource should refer to a Drawable object.
         * @param resId The identifier of the resource.
         */
        fun setCustomActionMenuItem(@DrawableRes resId: Int): Builder {
            customActionMenuItemIcon = resId
            return this
        }

        /**
         * set url suffix for first launch.
         * @param suffix
         */
        fun setUrlSuffix(suffix: String): Builder {
            this.urlSuffix = suffix
            return this
        }

        /**
         * Set extra query params to each request
         */
        fun setExtraUrlParams(params: Map<String, String>): Builder {
            this.extraParams = params
            return this
        }


        fun enableSplash(flag: Boolean): Builder {
            this.isSplashEnabled = flag
            return this
        }

        fun saveState(enable: Boolean): Builder {
            this.saveState = enable
            return this
        }

        /**
         * Set the page animation for open/exit miniapp page.
         * Animation will work only when disabled multitask mode.
         * @see Config.isMultitaskMode
         * @param animation PageAnimation
         */
        fun pageAnimation(animation: PageAnimation): Builder {
            this.pageAnimation = animation
            return this
        }

        fun build(): MiniappConfig {
            return MiniappConfig(
                colors,
                theme,
                customActionMenuItemIcon,
                extraParams,
                urlSuffix,
                isSplashEnabled,
                saveState,
                pageAnimation
            )
        }
    }

    /** Constructs a new builder, copying the attributes **/
    fun buildUpon(): Builder {
        return Builder().apply {
            colors?.apply { setColors(primaryColor, secondaryColor, tertiaryColor) }
            theme?.apply { setTheme(this) }
            customActionMenuItemIcon?.apply { setCustomActionMenuItem(this) }
            setExtraUrlParams(extraParams)
            setUrlSuffix(urlSuffix)
            enableSplash(isSplashEnabled)
            saveState(saveState)
            pageAnimation(pageAnimation)
        }
    }
}