package io.boxo.ui.main

import android.annotation.SuppressLint
import android.app.Activity
import android.app.ActivityManager
import android.app.DownloadManager
import android.content.Context
import android.content.Intent
import android.graphics.Bitmap
import android.net.Uri
import android.nfc.NdefRecord
import android.nfc.NfcAdapter
import android.nfc.Tag
import android.nfc.tech.Ndef
import android.os.Build
import android.os.Bundle
import android.os.Environment
import android.os.Handler
import android.view.View
import android.view.ViewGroup
import android.view.WindowManager
import android.webkit.MimeTypeMap
import android.webkit.URLUtil
import android.webkit.WebResourceError
import android.webkit.WebResourceRequest
import android.webkit.WebView
import android.webkit.WebView.setWebContentsDebuggingEnabled
import android.webkit.WebViewClient
import android.widget.ImageView
import androidx.activity.OnBackPressedCallback
import androidx.activity.result.contract.ActivityResultContracts
import androidx.activity.viewModels
import androidx.appcompat.app.AppCompatActivity
import androidx.constraintlayout.widget.ConstraintLayout
import androidx.core.view.ViewCompat
import androidx.core.view.WindowInsetsCompat
import androidx.lifecycle.AbstractSavedStateViewModelFactory
import androidx.lifecycle.SavedStateHandle
import androidx.lifecycle.ViewModel
import androidx.savedstate.SavedStateRegistryOwner
import io.boxo.R
import io.boxo.ServiceLocator
import io.boxo.data.models.ActionButtonTheme
import io.boxo.data.models.PageAnimation
import io.boxo.data.models.Status
import io.boxo.events.AuthEvents
import io.boxo.events.ClipboardEvents
import io.boxo.events.CustomEvents
import io.boxo.events.LocationEvents
import io.boxo.events.MiniappEvents
import io.boxo.events.PaymentEvents
import io.boxo.events.SensorEvents
import io.boxo.events.StorageEvents
import io.boxo.events.SystemEvents
import io.boxo.events.TransactionEvents
import io.boxo.events.UiEvents
import io.boxo.events.VibrationEvents
import io.boxo.js.JSFunctions
import io.boxo.js.jsInterfaces.BoxoJsInterface
import io.boxo.js.jsInterfaces.AuthJsI
import io.boxo.js.jsInterfaces.ClipboardJsI
import io.boxo.js.jsInterfaces.CustomEventsJsI
import io.boxo.js.jsInterfaces.LocationJsI
import io.boxo.js.jsInterfaces.MiniappJsI
import io.boxo.js.jsInterfaces.PaymentEventsJsI
import io.boxo.js.jsInterfaces.SensorJsI
import io.boxo.js.jsInterfaces.StorageJsI
import io.boxo.js.jsInterfaces.SystemJsI
import io.boxo.js.jsInterfaces.TransactionJsI
import io.boxo.js.jsInterfaces.UiJsI
import io.boxo.js.jsInterfaces.VibrationJsI
import io.boxo.js.params.ActionButton
import io.boxo.sdk.Boxo
import io.boxo.ui.about.MiniappAboutActivity
import io.boxo.ui.dialog.ContextMenuDialog
import io.boxo.ui.main.states.SettingsState
import io.boxo.ui.settings.MiniappSettingsActivity
import io.boxo.ui.view.ActionButtonsView
import io.boxo.ui.view.ErrorView
import io.boxo.ui.view.FloatingActionButtonsView
import io.boxo.ui.view.PullToRefreshLayout
import io.boxo.utils.ActivityResultHelper
import io.boxo.utils.Permissions
import io.boxo.utils.UiConst
import io.boxo.utils.extensions.bindView
import io.boxo.utils.extensions.fullscreen
import io.boxo.utils.extensions.goHome
import io.boxo.utils.extensions.invisible
import io.boxo.utils.extensions.lightStatusBar
import io.boxo.utils.extensions.setDefaultTheme
import io.boxo.utils.extensions.setMiniappLocale
import io.boxo.utils.extensions.show
import io.boxo.utils.webUrl
import com.squareup.picasso.Callback
import com.squareup.picasso.Picasso
import java.lang.Exception
import java.util.Arrays


class BoxoActivity : AppCompatActivity(), NfcAdapter.ReaderCallback {
    companion object {
        const val APP_ID = "boxo_app_id"
    }

    inner class ViewModelFactory(
        private val owner: SavedStateRegistryOwner,
    ) : AbstractSavedStateViewModelFactory(owner, null) {
        override fun <T : ViewModel> create(
            key: String,
            modelClass: Class<T>,
            state: SavedStateHandle
        ): T {
            val appId = intent.getStringExtra(APP_ID) ?: ""
            if (appId.isBlank()) finish()
            return BoxoViewModel(
                appId,
                ServiceLocator.networkService,
                ServiceLocator.storage,
                ServiceLocator.permissions,
                Boxo,
                ServiceLocator.context
            ) as T
        }
    }

    private val viewModel: BoxoViewModel by viewModels { ViewModelFactory(this) }

    private var isWebViewReady: Boolean = false
    private var onActivityResult: ((requestCode: Int, resultCode: Int, data: Intent?) -> Unit)? =
        null

    private val storage
        get() = ServiceLocator.storage
    private val permissionsStorage
        get() = ServiceLocator.permissions

    private val permissions by lazy { Permissions(this) }
    private val miniappEvents by lazy {
        MiniappEvents(this, viewModel, jsFunctions, storage, ::evaluateJavascript, Boxo)
    }
    private val customEvents by lazy { CustomEvents(this, viewModel) }
    private val paymentEvents by lazy { PaymentEvents(this, viewModel) }
    private val authEvents by lazy {
        AuthEvents(
            this,
            viewModel,
            jsFunctions,
            storage,
            permissionsStorage,
            ::evaluateJavascript
        )
    }

    private val embedBrowserDelegate = lazy { BoxoChromeCustomTab(this) }
    internal val embedBrowser by embedBrowserDelegate

    private val forResult by lazy { ActivityResultHelper(this) }

    private val jsFunctions: JSFunctions
        get() = ServiceLocator.jsFunctions

    private val locationEvents by lazy {
        LocationEvents(
            this,
            viewModel,
            permissionsStorage,
            permissions,
            jsFunctions,
            forResult,
            ::evaluateJavascript
        )
    }
    private val sensorEvents by lazy { SensorEvents(this, jsFunctions, ::evaluateJavascript) }
    private val transactionEvents by lazy {
        TransactionEvents(
            viewModel,
            jsFunctions,
            ::evaluateJavascript
        )
    }
    private val storageEvents by lazy {
        StorageEvents(
            this,
            jsFunctions,
            viewModel,
            ::evaluateJavascript
        )
    }
    private val systemEvents by lazy {
        SystemEvents(
            this,
            permissions,
            jsFunctions,
            ::evaluateJavascript
        )
    }
    private val uiEvents by lazy {
        UiEvents(
            this,
            viewModel,
            jsFunctions,
            permissionsStorage,
            forResult,
            this::evaluateJavascript
        )
    }
    private val vibrationEvents by lazy { VibrationEvents(this) }
    private val clipboardEvents by lazy { ClipboardEvents(this, jsFunctions, ::evaluateJavascript) }

    private val javascriptInterface by lazy {
        val appId = viewModel.appId
        val uiHandler = Handler(mainLooper)
        BoxoJsInterface(
            customEventsJsI = CustomEventsJsI(appId, uiHandler, customEvents, Boxo),
            authJsI = AuthJsI(appId, uiHandler, authEvents, Boxo),
            uiJsI = UiJsI(appId, uiHandler, uiEvents),
            miniappJsI = MiniappJsI(appId, uiHandler, miniappEvents, Boxo),
            storageJsI = StorageJsI(appId, uiHandler, storageEvents),
            locationJsI = LocationJsI(appId, uiHandler, locationEvents),
            sensorJsI = SensorJsI(appId, uiHandler, sensorEvents),
            systemJsI = SystemJsI(appId, uiHandler, systemEvents),
            paymentEventsJsI = PaymentEventsJsI(appId, uiHandler, paymentEvents, Boxo),
            transactionJsI = TransactionJsI(appId, uiHandler, transactionEvents),
            vibrationJsI = VibrationJsI(appId, uiHandler, vibrationEvents),
            clipboardJsI = ClipboardJsI(appId, uiHandler, clipboardEvents)
        )
    }

    private val root by bindView<ViewGroup>(R.id.root)
    private val webView by bindView<WebView>(R.id.webview)
    private val pullToRefreshLayout by bindView<PullToRefreshLayout>(R.id.pull_layout)
    private val logoContainer by bindView<ConstraintLayout>(R.id.logo_container)
    private val appLogo by bindView<ImageView>(R.id.app_logo)
    private val boxoLogo by bindView<ImageView>(R.id.imagePowered)
    private val errorView by bindView<ErrorView>(R.id.error_view)
    private val actionButtons by bindView<ActionButtonsView>(R.id.action_buttons)
    private val floatingActionButtons by bindView<FloatingActionButtonsView>(R.id.floating_action_buttons)

    private var nfcAdapter: NfcAdapter? = null

    private val minLoadProgress by lazy { if (viewModel.miniapp.config?.isSplashEnabled == true) 50 else 0 }

    private val webChromeClient = object : BoxoWebViewClient(this@BoxoActivity, permissions) {
        override fun onProgressChanged(view: WebView, newProgress: Int) {
            if (view.progress >= minLoadProgress) hideSplash()
        }
    }

    private val webViewClient = object : WebViewClient() {
        private var lastUrl = ""

        override fun shouldOverrideUrlLoading(
            view: WebView?,
            request: WebResourceRequest
        ): Boolean {
            val url = request.webUrl
            return load(url, view)
        }

        @Deprecated("Deprecated")
        override fun shouldOverrideUrlLoading(view: WebView?, url: String?): Boolean {
            return load(url, view)
        }

        private fun load(url: String?, view: WebView?): Boolean {
            if (viewModel.miniapp.customUrlLoadingHandler != null) {
                return viewModel.miniapp.customUrlLoadingHandler!!.shouldOverrideUrlLoading(
                    view,
                    url
                )
            } else {
                if (url == "about:blank") return false
                if (url != null && URLUtil.isNetworkUrl(url)) {
                    return if (viewModel.miniappSettings.disableUrlWhitelisting) {
                        view?.loadUrl(viewModel.urlWithExtraParams(url))
                        true
                    } else {
                        val urlInWhiteList =
                            viewModel.miniappSettings.whiteListedUrls.any { url.startsWith(it) }
                        if (urlInWhiteList) {
                            view?.loadUrl(viewModel.urlWithExtraParams(url))
                            true
                        } else {
                            embedBrowser.openExternalPage(url)
                            true
                        }
                    }
                }

                return runCatching {
                    startActivity(Intent(Intent.ACTION_VIEW, Uri.parse(url)))
                    true
                }.getOrDefault(true)
            }
        }

        override fun doUpdateVisitedHistory(view: WebView?, url: String?, isReload: Boolean) {
            super.doUpdateVisitedHistory(view, url, isReload)
            url?.also {
                if (lastUrl != it) {
                    viewModel.miniapp.urlChangeListener?.handle(
                        this@BoxoActivity,
                        viewModel.miniapp,
                        Uri.parse(it)
                    )
                }
                lastUrl = it
            }
        }

        override fun onPageStarted(view: WebView?, url: String?, favicon: Bitmap?) {
            if (storage.isFirstLaunch(viewModel.appId)) {
                webView.apply {
                    evaluateJavascript(
                        "window.localStorage.clear(); " +
                                jsFunctions.removeCookies
                    )
                }
                storage.miniappLaunched(viewModel.appId)
            }
            errorView.invisible()
            super.onPageStarted(view, url, favicon)
        }

        override fun onPageFinished(view: WebView?, url: String?) {
            evaluateJavascript(viewModel.miniappSettings.script)
        }

        private fun onError() {
            webView.stopLoading()
            errorView.noInternetError { webView.reload() }
        }

        override fun onReceivedError(
            view: WebView,
            request: WebResourceRequest,
            error: WebResourceError?
        ) {
            super.onReceivedError(view, request, error)
            if (request.isForMainFrame && error != null) {
                onError()
            }
        }

        @Deprecated(" >= API level 23")
        override fun onReceivedError(
            view: WebView?,
            errorCode: Int,
            description: String?,
            failingUrl: String?
        ) {
            onError()
        }
    }

    private val onBackPressedCallback = object : OnBackPressedCallback(true) {
        override fun handleOnBackPressed() {
            onBackPress()
        }
    }

    override fun attachBaseContext(newBase: Context?) {
        setMiniappLocale()
        super.attachBaseContext(newBase)
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        setDefaultTheme(Boxo)
        super.onCreate(savedInstanceState)
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
            ViewCompat.setOnApplyWindowInsetsListener(window?.decorView!!) { v, insets ->
                val imeHeight = insets.getInsets(WindowInsetsCompat.Type.ime()).bottom
                val navigationBarHeight =
                    insets.getInsets(WindowInsetsCompat.Type.navigationBars()).bottom
                root.setPadding(0, 0, 0, imeHeight - navigationBarHeight)
                ViewCompat.onApplyWindowInsets(v, insets)
            }
        } else {
            @Suppress("DEPRECATION")
            window.setSoftInputMode(WindowManager.LayoutParams.SOFT_INPUT_ADJUST_RESIZE)
        }

        setContentView(R.layout.boxo_activity_web_view)
        init()
        initNFC()
        onBackPressedDispatcher.addCallback(this, onBackPressedCallback)
    }

    private fun init() {
        fullscreen()
        initStart()
        initAppLogo(viewModel.miniappLogo)
        enableBranding(viewModel.boxoBrandingVisible)
        viewModel.settingsState.observe(this) { state ->
            when (state.state) {
                SettingsState.State.LOADING_MINIAPP_SETTINGS -> {
                    logoContainer.show()
                    errorView.invisible()
                }

                SettingsState.State.INTERNET_ERROR -> {
                    errorView.noInternetError { reload() }
                }

                SettingsState.State.DEFAULT_ERROR -> {
                    errorView.defaultError { reload() }
                }

                SettingsState.State.NOT_AVAILABLE_ERROR -> {
                    if (state.message.isNotBlank())
                        errorView.textError(state.message) { reload() }
                    else
                        errorView.miniappNotAvailableError { reload() }
                }

                SettingsState.State.SUCCESS -> {
                    requestedOrientation = viewModel.screenOrientation
                    initAppLogo(viewModel.miniappLogo)
                    enableBranding(viewModel.boxoBrandingVisible)
                    initWebView()
                    initActionButtons()
                    loaded()
                }
            }
        }
        errorView.setOnClose { closeMiniapp() }
        viewModel.loginState.observe(this) { loginState -> authEvents.handleLoginState(loginState) }
        viewModel.exit.observe(this) { finishAndRemoveTask() }

        viewModel.miniapp.sendCustomEvent =
            { customEvent -> evaluateJavascript(jsFunctions.getFuncForCustomEvent(customEvent)) }
        viewModel.miniapp.sendPaymentResult =
            { paymentData -> evaluateJavascript(jsFunctions.getFuncForPaymentResult(paymentData)) }
        viewModel.miniapp.onNFCTagRead =
            { data -> evaluateJavascript(jsFunctions.getFuncToSetNfcRecords(data)) }

        viewModel.miniapp.onDestroy = { finishAndRemoveTask() }
        isWebViewReady = false
    }

    private fun initStart() {
        ViewCompat.setOnApplyWindowInsetsListener(webView) { _, insets ->
            UiConst.statusBarHeight = insets.systemWindowInsetTop
            insets.consumeSystemWindowInsets()
        }
        uiEvents.onCreate()
        lightStatusBar()
    }

    private fun loaded() {
        uiEvents.initAppbar(webView)
    }

    private fun onWebViewReady() {
        uiEvents.ready()
    }

    override fun onResume() {
        super.onResume()
        if (isWebViewReady) evaluateJavascript(jsFunctions.onRestore())
        viewModel.onResume()
        webView.onResume()
        enableNfcReader()
    }

    override fun onPause() {
        super.onPause()
        if (isWebViewReady) viewModel.sessionEnd()
        viewModel.onPause()
        if (!viewModel.isReloading && !viewModel.hasSettingsChanges)
            Bundle().apply {
                Boxo.miniappStates[viewModel.appId] = this
                webView.saveState(this)
            }
        else
            Boxo.miniappStates[viewModel.appId] = null
        webView.onPause()
        disableNfcReader()
    }

    override fun onUserInteraction() {
        super.onUserInteraction()
        viewModel.onUserInteraction()
    }

    private fun hideSplash() {
        if (isWebViewReady.not()) {
            isWebViewReady = true
            pullToRefreshLayout.show()
            logoContainer.invisible()
            boxoLogo.invisible()
            onWebViewReady()
        }
    }

    private fun initAppLogo(logo: String) {
        if (logo.isNotBlank()) {
            Picasso.get()
                .load(logo)
                .noFade()
                .into(appLogo, object : Callback {
                    override fun onSuccess() {
                        runCatching { appLogo.show() }
                    }

                    override fun onError(e: Exception?) {
                    }
                })
        }
    }

    private fun enableBranding(enable: Boolean) {
        if (boxoLogo.visibility != View.VISIBLE) {
            if (enable) boxoLogo.show() else boxoLogo.visibility = View.INVISIBLE
            errorView.boxoLogoVisible = enable
        }
    }

    fun closeMiniapp() {
        if (viewModel.hasSettingsChanges) {
            clearCache()
            finishAndRemoveTask()
        } else if (Boxo.config.isMultitaskMode) {
            finishAndRemoveTask()
            val activityManager = getSystemService(Context.ACTIVITY_SERVICE) as ActivityManager
            val backActivities = activityManager.appTasks
            if (backActivities.isNotEmpty()) {
                backActivities.first().moveToFront()
            } else goHome()
        } else {
            finishAndRemoveTask()
            val exitAnim = when (viewModel.miniapp.config?.pageAnimation) {
                PageAnimation.LEFT_TO_RIGHT -> R.anim.boxo_slide_left_right_exit
                PageAnimation.RIGHT_TO_LEFT -> R.anim.boxo_slide_right_left_exit
                PageAnimation.TOP_TO_BOTTOM -> R.anim.boxo_slide_top_bottom_exit
                PageAnimation.FADE_IN -> R.anim.boxo_fade_out
                else -> R.anim.boxo_slide_bottom_top_exit
            }
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.UPSIDE_DOWN_CAKE) {
                overrideActivityTransition(
                    OVERRIDE_TRANSITION_CLOSE,
                    0,
                    exitAnim
                )
            } else {
                @Suppress("DEPRECATION")
                overridePendingTransition(0, exitAnim)
            }
        }
    }

    @SuppressLint("SetJavaScriptEnabled")
    private fun initWebView() {
        webView.webChromeClient = webChromeClient
        webView.webViewClient = webViewClient
        webView.apply {
            settings.javaScriptEnabled = true
            settings.loadWithOverviewMode = true
            settings.useWideViewPort = true
            settings.databaseEnabled = true
            settings.domStorageEnabled = true
            settings.setSupportZoom(true)
            settings.allowContentAccess = true
            settings.setSupportMultipleWindows(true)
            settings.javaScriptCanOpenWindowsAutomatically = true
            settings.userAgentString = settings.userAgentString.replace("; wv", "")
            setWebContentsDebuggingEnabled(Boxo.config.isDebug)
        }
        webView.addJavascriptInterface(javascriptInterface, "AppboxoJs")
        webView.setDownloadListener { url, userAgent, contentDisposition, mimetype, contentLength ->
            runCatching {
                val uri = Uri.parse(url)
                val fileName = runCatching {
                    val path = uri.path!!.split("/")
                    var name = path[path.size - 1]
                    if (!name.contains(".")) {
                        name += "." + MimeTypeMap.getSingleton().getExtensionFromMimeType(mimetype)
                    }
                    name
                }.getOrDefault(
                    "miniapp_file_${System.currentTimeMillis()}.${
                        MimeTypeMap.getSingleton().getExtensionFromMimeType(mimetype)
                    }"
                )

                val request = DownloadManager.Request(uri).apply {
                    setDestinationInExternalPublicDir(Environment.DIRECTORY_DOWNLOADS, fileName)
                    setNotificationVisibility(DownloadManager.Request.VISIBILITY_VISIBLE_NOTIFY_COMPLETED)
                    addRequestHeader("User-Agent", userAgent)
                }
                val downloadManager = getSystemService(DOWNLOAD_SERVICE) as DownloadManager
                downloadManager.enqueue(request)
            }
        }
        if (viewModel.miniappSettings.status != Status.MAINTENANCE) {
            val state = Boxo.miniappStates[viewModel.appId]
            if (state == null || state.isEmpty)
                webView.loadUrl(viewModel.firstLaunchUrl())
            else
                webView.restoreState(state)
        } else {
            webView.loadUrl(viewModel.miniappSettings.miniappInfo.maintenancePageUrl)
        }
    }

    private val getMiniappSettingsResult =
        registerForActivityResult(ActivityResultContracts.StartActivityForResult()) { result ->
            if (result.resultCode == Activity.RESULT_OK) {
                if (result.data?.getBooleanExtra("revoke", false) == true) {
                    storage.clearMiniappData(viewModel.appId)
                    clearCache()
                    reload()
                    recreate()
                }
            }
        }

    private fun showMoreMenu() {
        ContextMenuDialog(
            style = viewModel.miniappSettings.hostappInfo.menuStyle,
            settings = Boxo.config.permissionsPage,
            clearCacheEnabled = Boxo.config.showClearCache,
            aboutPageEnabled = Boxo.config.showAboutPage
        ).apply {
            onAboutClick = {
                startActivity(
                    Intent(context, MiniappAboutActivity::class.java)
                        .putExtra(APP_ID, viewModel.appId)
                )
            }
            onReloadClick = {
                reload()
                recreate()
            }
            onClearCacheClick = {
                clearCache()
                reload()
                recreate()
            }
            onSettingsClick = {
                getMiniappSettingsResult.launch(
                    Intent(context, MiniappSettingsActivity::class.java)
                        .putExtra(APP_ID, viewModel.appId)
                )
            }
        }.show(supportFragmentManager, null)
    }

    private fun reload() {
        viewModel.reload()
    }

    private fun clearCache() {
        evaluateJavascript("window.localStorage.clear();")
        evaluateJavascript(jsFunctions.removeCookies)
        webView.clearHistory()
        webView.clearCache(true)
    }

    private fun evaluateJavascript(funcBody: String) {
        webView.evaluateJavascript(funcBody, null)
    }

    private fun backFromSecondWebView(): Boolean {
        return if (webView.childCount > 0) {
            webChromeClient.authWebViewBackPress()
        } else false
    }

    internal fun onBackPress() {
        if (backFromSecondWebView().not()) {
            if (webView.canGoBack()) {
                webView.goBack()
            } else {
                closeMiniapp()
            }
        }
    }

    @SuppressLint("MissingSuperCall")
    override fun onRequestPermissionsResult(
        requestCode: Int,
        permissions: Array<String>,
        grantResults: IntArray
    ) {
        this.permissions.onRequestPermissionsResult(this, requestCode, permissions, grantResults)
    }

    override fun onActivityResult(requestCode: Int, resultCode: Int, data: Intent?) {
        onActivityResult?.invoke(requestCode, resultCode, data)
        val result = forResult.onActivityResult(requestCode, resultCode, data)
        if (!result) {
            super.onActivityResult(requestCode, resultCode, data)
        }
    }

    fun doOnActivityResult(action: (requestCode: Int, resultCode: Int, data: Intent?) -> Unit) {
        onActivityResult = action
    }

    fun doOnActivityResult(activityResult: BoxoActivityResult) {
        onActivityResult = activityResult::onActivityResult
    }

    override fun onDestroy() {
        webView.destroy()
        if (embedBrowserDelegate.isInitialized()) embedBrowser.destroy()
        super.onDestroy()
    }

    private fun initActionButtons() {
        with(viewModel.miniappSettings) {
            val isLight = actionButtonsTheme == ActionButtonTheme.LIGHT
            val actionButton = ActionButton(
                isLight,
                style = runCatching { ActionButton.Style.valueOf(hostappInfo.menuStyle) }
                    .getOrDefault(ActionButton.Style.default),
                visible = true
            )
            uiEvents.setActionButtons(actionButton)
        }
        val miniapp = viewModel.miniapp
        actionButtons.apply {
            setOnMoreClickListener { showMoreMenu() }
            setOnCloseClickListener { closeMiniapp() }
            setOnCustomButtonClickListener {
                miniapp.customActionMenuItemClickListener?.onClick(
                    this@BoxoActivity,
                    miniapp
                )
            }
        }
        floatingActionButtons.apply {
            setOnMoreClickListener { showMoreMenu() }
            setOnCloseClickListener { closeMiniapp() }
            setOnCustomButtonClickListener {
                miniapp.customActionMenuItemClickListener?.onClick(
                    this@BoxoActivity,
                    miniapp
                )
            }
        }
        miniapp.onHideCustomActionMenuItem = {
            actionButtons.hideCustomButton()
            floatingActionButtons.hideCustomButton()
        }
        miniapp.onShowCustomActionMenuItem = {
            miniapp.config?.customActionMenuItemIcon?.let { id ->
                actionButtons.showCustomButton(id)
                floatingActionButtons.showCustomButton(id)
            }
        }
    }

    private fun initNFC() {
        nfcAdapter = NfcAdapter.getDefaultAdapter(this)
    }

    private fun enableNfcReader() {
        val flags = NfcAdapter.FLAG_READER_NFC_A or
                NfcAdapter.FLAG_READER_NFC_B or
                NfcAdapter.FLAG_READER_NFC_F or
                NfcAdapter.FLAG_READER_NFC_V
        nfcAdapter?.enableReaderMode(this, this, flags, null)
    }

    private fun disableNfcReader() {
        nfcAdapter?.disableForegroundDispatch(this)
    }

    override fun onTagDiscovered(tag: Tag) {
        if (tag.techList.contains(Ndef::class.java.canonicalName)) {
            val ndef = Ndef.get(tag)
            if (ndef != null) {
                val ndefMessage = ndef.cachedNdefMessage
                if (ndefMessage != null) {
                    val records = ndefMessage.records
                    val dataList = records.map { readNdefRecord(it) }
                    window.decorView.post {
                        viewModel.miniapp.onNFCTagRead?.invoke(dataList)
                    }
                }
            }
        }
    }

    private fun readNdefRecord(record: NdefRecord): String {
        val payload = record.payload
        val data = try {
            if (record.tnf == NdefRecord.TNF_WELL_KNOWN) {
                if (Arrays.equals(record.type, NdefRecord.RTD_URI)) {
                    record.toUri().toString()
                } else {
                    val textEncoding =
                        if (payload[0].toInt() and 128 == 0) Charsets.UTF_8 else Charsets.UTF_16
                    val languageCodeLength: Int = payload[0].toInt() and 51
                    String(
                        payload, languageCodeLength + 1, payload.size - languageCodeLength - 1,
                        textEncoding
                    )
                }
            } else payload.toString()
        } catch (e: StringIndexOutOfBoundsException) {
            payload.toString()
        }
        return data
    }

    override fun finishAndRemoveTask() {
        sensorEvents.close()
        super.finishAndRemoveTask()
    }
}
