package io.boxo.ui.main

import android.content.Context
import android.content.pm.ActivityInfo
import android.net.Uri
import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import io.boxo.R
import io.boxo.data.models.MiniappSettings
import io.boxo.data.models.Status
import io.boxo.data.network.NetworkService
import io.boxo.data.storage.IBoxoPermissions
import io.boxo.data.storage.IBoxoStorage
import io.boxo.log.exceptions.BoxoException
import io.boxo.log.exceptions.BoxoLoginException
import io.boxo.sdk.Boxo
import io.boxo.sdk.Miniapp
import io.boxo.ui.main.states.LoginState
import io.boxo.ui.main.states.SettingsState
import io.boxo.utils.extensions.mutable
import kotlinx.coroutines.GlobalScope
import kotlinx.coroutines.launch
import java.net.SocketTimeoutException
import java.net.UnknownHostException

internal class BoxoViewModel(
    val appId: String,
    private val networkService: NetworkService,
    private val storage: IBoxoStorage,
    private val permissions: IBoxoPermissions,
    private val boxo: Boxo,
    private val context: Context
) : ViewModel() {

    var isReloading: Boolean = false
    val miniapp: Miniapp
    lateinit var miniappSettings: MiniappSettings
    val miniappLogo: String
        get() {
            return if (::miniappSettings.isInitialized) miniappSettings.miniappInfo.logo
            else storage.getMiniappSettings(miniapp.appId)?.miniappInfo?.logo ?: ""
        }
    val boxoBrandingVisible: Boolean
        get() {
            return if (::miniappSettings.isInitialized) miniappSettings.brandingEnabled
            else storage.getMiniappSettings(miniapp.appId)?.brandingEnabled ?: false
        }
    private val miniappUrl: String
        get() = miniappSettings.miniappInfo.url
    val isFullscreen: Boolean
        get() = if (::miniappSettings.isInitialized) miniappSettings.isFullscreen() else true
    val screenOrientation: Int
        get() = when (miniappSettings.orientation) {
            0 -> ActivityInfo.SCREEN_ORIENTATION_PORTRAIT
            1 -> ActivityInfo.SCREEN_ORIENTATION_LANDSCAPE
            else -> ActivityInfo.SCREEN_ORIENTATION_USER
        }
    val defaultStatusBar: String
        get() = if (::miniappSettings.isInitialized) miniappSettings.statusBarBackground
        else storage.getMiniappSettings(miniapp.appId)?.statusBarBackground ?: "#00FFFFFF"

    internal val loginState: LiveData<LoginState> = MutableLiveData()
    internal val settingsState: LiveData<SettingsState> = MutableLiveData()
    internal val exit: LiveData<String> = MutableLiveData()

    private var resumeDate: Long = 0
    var hasSettingsChanges: Boolean = false

    private var startTime = System.currentTimeMillis()
    private var openedTime = System.currentTimeMillis()
    val openMiniappDuration
        get() = openedTime - startTime

    init {
        val app = boxo.getExistingMiniapp(appId)
        if (app == null)
            exit.mutable().value = "Miniapp is null"
        miniapp = app ?: Miniapp("")
        viewModelScope.launch { loadMiniappSettings() }
        loginState.mutable().value = LoginState.idle()
        miniapp.onGetAuthCode = { login(it) }
        sessionStart()
    }

    private fun login(authCode: String) {
        viewModelScope.launch {
            if (loginState.value!!.state == LoginState.State.IDLE) {
                loginState.mutable().value = LoginState.loading()
                runCatching {
                    networkService.login(
                        appId,
                        authCode,
                        miniappSettings.miniappInfo.isShopboxo
                    )
                }
                    .onFailure { e ->
                        boxo.logger.error(appId, e)
                        when (e) {
                            is UnknownHostException,
                            is SocketTimeoutException ->
                                loginState.mutable().value =
                                    LoginState.error(context.getString(R.string.boxo_login_internet_connection_error))

                            is BoxoLoginException ->
                                loginState.mutable().value = LoginState.error(e.message, e.data)

                            else ->
                                loginState.mutable().value =
                                    LoginState.error(context.getString(R.string.boxo_login_unknown_error))
                        }
                        loginState.mutable().value = LoginState.idle()
                    }
                    .onSuccess { data ->
                        permissions.allowUserDataPermission(appId, true)
                        loginState.mutable().value = LoginState.success(data)
                        loginState.mutable().value = LoginState.idle()
                    }
            }
        }
    }

    fun reload() {
        isReloading = true
        startTime = System.currentTimeMillis()
        viewModelScope.launch {
            loadMiniappSettings(true)
            isReloading = false
        }
    }

    private fun sessionStart() = viewModelScope.launch {
        runCatching {
            networkService.sendSessionStart(appId, System.currentTimeMillis())
        }
    }

    fun sessionEnd() = GlobalScope.launch {
        runCatching {
            networkService.sendSessionEnd(appId, resumeDate, System.currentTimeMillis())
        }
    }

    fun onResume() {
        resumeDate = System.currentTimeMillis()
        miniapp.lifecycleListener?.onResume(miniapp)
    }

    fun onPause() {
        miniapp.lifecycleListener?.onPause(miniapp)
    }

    fun onUserInteraction() {
        miniapp.lifecycleListener?.onUserInteraction(miniapp)
    }

    private fun changeSettingsState() {
        when (miniappSettings.status) {
            Status.VERIFIED -> {
                openedTime = System.currentTimeMillis()
                settingsState.mutable().value = SettingsState.success()
            }

            Status.MAINTENANCE -> {
                if (miniappSettings.miniappInfo.maintenancePageUrl.isNotBlank()) {
                    openedTime = System.currentTimeMillis()
                    settingsState.mutable().value = SettingsState.success()
                } else
                    settingsState.mutable().value = SettingsState.notAvailableError()
            }

            else -> {
                settingsState.mutable().value = SettingsState.notAvailableError()
            }
        }
    }

    private suspend fun loadMiniappSettings(isReload: Boolean = false) {
        settingsState.mutable().value = SettingsState.loading()
        val savedMiniappSettings = storage.getMiniappSettings(appId)
        val savedTimeMillis = storage.miniappSettingsSavedTime(appId)
        if (!isReload &&
            savedMiniappSettings != null &&
            (System.currentTimeMillis() - savedTimeMillis) < Boxo.config.miniappSettingsExpirationTime * 1000L
        ) {
            miniappSettings = savedMiniappSettings
            changeSettingsState()
        } else {
            val token = (miniapp.data?.get("appboxo_auth_token") as? String) ?: ""
            runCatching { networkService.getMiniappSettings(miniapp.appId, token) }
                .onFailure { e ->
                    boxo.logger.error(appId, e)
                    when (e) {
                        is UnknownHostException, is SocketTimeoutException -> {
                            settingsState.mutable().value = SettingsState.internetError()
                        }

                        is BoxoException -> {
                            settingsState.mutable().value =
                                SettingsState.notAvailableError(e.message)
                            miniapp.lifecycleListener?.onError(miniapp, e.message)
                        }

                        else -> {
                            settingsState.mutable().value = SettingsState.defaultError()
                            miniapp.lifecycleListener?.onError(
                                miniapp,
                                "miniapp_settings_unknown_error"
                            )
                        }
                    }
                }
                .onSuccess { settings ->
                    miniappSettings = settings
                    storage.saveMiniappSettings(miniapp.appId, miniappSettings)
                    changeSettingsState()
                    permissions.allowUserDataPermission(appId, settings.isConsented)
                }
        }
    }

    fun transactionTrack(payload: String, onResult: (Boolean) -> Unit) {
        viewModelScope.launch {
            runCatching { networkService.sendTransactionEvent(appId, payload) }
                .onFailure { e ->
                    Boxo.logger.error(appId, e)
                    onResult(false)
                }
                .onSuccess(onResult)
        }
    }

    fun firstLaunchUrl(): String {
        val url = miniappUrl + (miniapp.config?.urlSuffix ?: "")
        return urlWithExtraParams(url)
    }

    fun urlWithExtraParams(url: String? = null): String {
        return if (miniapp.config?.extraParams?.isNotEmpty() == true)
            Uri.parse(url ?: miniappUrl).buildUpon().apply {
                miniapp.config!!.extraParams.forEach { appendQueryParameter(it.key, it.value) }
            }.build().toString()
        else url ?: miniappUrl
    }

    override fun onCleared() {
        miniapp.lifecycleListener?.onClose(miniapp)
        if (miniapp.isClosing.not()) miniapp.close()
        super.onCleared()
    }
}