package io.boxo.ui.main;

import android.annotation.SuppressLint
import android.app.Activity
import android.content.Intent
import android.graphics.Bitmap
import android.net.Uri
import android.os.Message
import android.provider.MediaStore
import android.webkit.GeolocationPermissions
import android.webkit.URLUtil
import android.webkit.ValueCallback
import android.webkit.WebChromeClient
import android.webkit.WebResourceRequest
import android.webkit.WebView
import android.webkit.WebViewClient
import android.widget.FrameLayout
import androidx.activity.result.ActivityResult
import androidx.activity.result.ActivityResultLauncher
import androidx.activity.result.contract.ActivityResultContracts
import androidx.core.content.FileProvider
import io.boxo.ui.dialog.ImageTypeDialog
import io.boxo.utils.Permissions
import io.boxo.utils.webUrl
import java.io.File
import java.io.IOException
import java.text.SimpleDateFormat
import java.util.Date

internal open class BoxoWebViewClient(
    private val activity: BoxoActivity,
    private val permissions: Permissions
) : WebChromeClient() {
    companion object {
        private val socialHosts = listOf(
            "facebook.com",
            "google.com",
            "twitter.com",
            "appleid.apple.com"
        )
    }

    private var activityLauncher: ActivityResultLauncher<Intent> =
        activity.registerForActivityResult(ActivityResultContracts.StartActivityForResult()) { result ->
            activityListener?.invoke(result)
        }
    private var activityListener: ((result: ActivityResult) -> Unit)? = null

    private var parentWebView: WebView? = null
    private var webViewForAuth: WebView? = null

    override fun onShowFileChooser(
        webView: WebView?,
        filePathCallback: ValueCallback<Array<Uri>>,
        fileChooserParams: FileChooserParams
    ): Boolean {
        val acceptTypes = fileChooserParams.acceptTypes
        val captureEnabled = fileChooserParams.isCaptureEnabled
        val capturePhoto = captureEnabled && acceptTypes.contains("image/*")
        if ((capturePhoto)) {
            showMediaCaptureOrFilePicker(filePathCallback, fileChooserParams)
        } else {
            openFilePicker(filePathCallback, fileChooserParams)
        }

        return true
    }

    private fun showMediaCaptureOrFilePicker(
        filePathCallback: ValueCallback<Array<Uri>>,
        fileChooserParams: FileChooserParams
    ) {
        ImageTypeDialog().apply {
            onFileClick = { openFilePicker(filePathCallback, fileChooserParams) }
            onCameraClick = {
                if (permissions.hasCameraPermission()) {
                    openImageCapturePicker(filePathCallback)
                } else {
                    permissions.camera(this@BoxoWebViewClient.activity,
                        onAccess = {
                            openImageCapturePicker(filePathCallback)
                        }, onDenied = {
                            filePathCallback.onReceiveValue(null)
                        }, onForeverDenied = {
                            filePathCallback.onReceiveValue(null)
                        })
                }
            }
            onDismiss = { filePathCallback.onReceiveValue(null) }
        }.show(activity.supportFragmentManager, "file_dialog")
    }

    @SuppressLint("QueryPermissionsNeeded")
    private fun openImageCapturePicker(filePathCallback: ValueCallback<Array<Uri>>) {
        val takePictureIntent = Intent(MediaStore.ACTION_IMAGE_CAPTURE)

        val imageFileUri: Uri
        try {
            imageFileUri = createImageFileUri()
        } catch (ex: Exception) {
            filePathCallback.onReceiveValue(null)
            return
        }
        takePictureIntent.putExtra(MediaStore.EXTRA_OUTPUT, imageFileUri)
        activityListener = { activityResult ->
            var result: Array<Uri>? = null
            if (activityResult.resultCode == Activity.RESULT_OK) {
                result = arrayOf(imageFileUri)
            }
            filePathCallback.onReceiveValue(result)
        }
        activityLauncher.launch(takePictureIntent)
    }

    private fun openFilePicker(
        filePathCallback: ValueCallback<Array<Uri>>,
        fileChooserParams: FileChooserParams
    ) {
        runCatching {
            activityListener = { activityResult ->
                val resultIntent: Intent? = activityResult.data
                val result = FileChooserParams.parseResult(activityResult.resultCode, resultIntent)
                filePathCallback.onReceiveValue(result)
            }
            activityLauncher.launch(fileChooserParams.createIntent())
        }.onFailure {
            filePathCallback.onReceiveValue(null)
        }
    }

    @Throws(IOException::class)
    private fun createImageFileUri(): Uri {
        val photoFile = createImageFile(activity)
        return FileProvider.getUriForFile(
            activity,
            "${activity.packageName}.boxo.provider",
            photoFile
        )
    }

    @Throws(IOException::class)
    private fun createImageFile(activity: Activity): File {
        val timeStamp: String = SimpleDateFormat("yyyyMMdd_HHmmss").format(Date())
        val imageFileName = "JPEG_" + timeStamp + "_"
        val storageDir = File(activity.cacheDir, "boxo")
        if (storageDir.exists().not()) storageDir.mkdir()
        return File.createTempFile(imageFileName, ".jpg", storageDir)
    }

    override fun onCreateWindow(
        view: WebView,
        isDialog: Boolean,
        isUserGesture: Boolean,
        resultMsg: Message
    ): Boolean {
        parentWebView = view
        webViewForAuth = WebView(activity)
        webViewForAuth?.layoutParams = FrameLayout.LayoutParams(
            FrameLayout.LayoutParams.MATCH_PARENT,
            FrameLayout.LayoutParams.MATCH_PARENT
        )
        view.addView(webViewForAuth)
        initWebViewForAuth()
        val transport = resultMsg.obj as WebView.WebViewTransport
        transport.webView = webViewForAuth
        resultMsg.sendToTarget()
        return true
    }

    override fun onGeolocationPermissionsShowPrompt(
        origin: String?,
        callback: GeolocationPermissions.Callback?
    ) {
        permissions.location(onAccess = {
            callback?.invoke(origin, true, false)
        }, onDenied = {
            callback?.invoke(origin, false, false)
        }, onForeverDenied = {
            callback?.invoke(origin, false, false)
        })
    }

    override fun getDefaultVideoPoster(): Bitmap? {
        return Bitmap.createBitmap(10, 10, Bitmap.Config.ARGB_8888);
    }

    @SuppressLint("SetJavaScriptEnabled")
    private fun initWebViewForAuth() {
        webViewForAuth?.apply {
            settings.apply {
                javaScriptEnabled = true
                userAgentString = userAgentString.replace("; wv", "")
            }
            webChromeClient = object : WebChromeClient() {
                override fun onCloseWindow(window: WebView?) {
                    closeWebViewForAuth()
                    super.onCloseWindow(window)
                }
            }
            webViewClient = object : WebViewClient() {
                override fun shouldOverrideUrlLoading(
                    view: WebView?,
                    request: WebResourceRequest
                ): Boolean {
                    val url = request.webUrl
                    return load(view, url)
                }

                @Deprecated("For old android versions")
                override fun shouldOverrideUrlLoading(view: WebView?, url: String?): Boolean {
                    return load(view, url)
                }

                private fun load(view: WebView?, url: String?): Boolean {
                    if (url != null && URLUtil.isNetworkUrl(url)) {
                        val host = Uri.parse(url).host!!
                        if (socialHosts.any { host.contains(it) }) {
                            return false
                        }
                    }
                    closeWebViewForAuth()
                    url?.let { parentWebView?.loadUrl(it) }
                    return true
                }
            }
        }
    }

    private fun closeWebViewForAuth() {
        parentWebView?.removeView(webViewForAuth)
    }

    fun authWebViewBackPress(): Boolean {
        return if (webViewForAuth?.canGoBack() == true) {
            webViewForAuth?.goBack()
            true
        } else if (webViewForAuth != null) {
            closeWebViewForAuth()
            true
        } else false
    }
}