package io.boxo.ui.map

import android.annotation.SuppressLint
import android.app.Activity
import android.content.Context
import android.content.Intent
import android.location.Location
import android.location.LocationManager
import android.os.Bundle
import android.widget.Button
import android.widget.ImageButton
import android.widget.ImageView
import androidx.fragment.app.FragmentActivity
import io.boxo.R
import io.boxo.js.params.LocationParam
import io.boxo.utils.ActivityResultHelper
import io.boxo.utils.Permissions
import io.boxo.utils.UiConst
import io.boxo.utils.extensions.*
import com.google.android.gms.location.LocationCallback
import com.google.android.gms.location.LocationRequest
import com.google.android.gms.location.LocationResult
import com.google.android.gms.location.LocationServices
import com.google.android.gms.location.Priority
import com.google.android.gms.maps.CameraUpdateFactory
import com.google.android.gms.maps.GoogleMap
import com.google.android.gms.maps.OnMapReadyCallback
import com.google.android.gms.maps.SupportMapFragment
import com.google.android.gms.maps.model.BitmapDescriptorFactory
import com.google.android.gms.maps.model.LatLng
import com.google.android.gms.maps.model.MarkerOptions


internal class BoxoMapActivity : FragmentActivity(), OnMapReadyCallback {

    private val locationManager by lazy { getSystemService(Context.LOCATION_SERVICE) as LocationManager }
    private val isLocationEnabled: Boolean
        get() = locationManager.isProviderEnabled(LocationManager.GPS_PROVIDER)
    private val fusedLocationClient by lazy { LocationServices.getFusedLocationProviderClient(this) }

    private val forResult by lazy { ActivityResultHelper(this) }
    private val permissions by lazy { Permissions(this) }
    private var map: GoogleMap? = null

    private val backBtn by bindView<ImageButton>(R.id.back_btn)
    private val chooseBtn by bindView<Button>(R.id.choose_button)
    private val myLocationBtn by bindView<ImageButton>(R.id.my_location)
    private val currentMarker by bindView<ImageView>(R.id.current_pos)

    override fun attachBaseContext(newBase: Context?) {
        setMiniappLocale()
        super.attachBaseContext(newBase)
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        setContentView(R.layout.boxo_activity_map)
        fullscreen()
        lightStatusBar()
        backBtn.setMarginTop(UiConst.statusBarHeight)
        backBtn.setOnClickListener { onBackPressedDispatcher.onBackPressed() }

        val mapFragment = supportFragmentManager.findFragmentById(R.id.map) as SupportMapFragment
        mapFragment.getMapAsync(this)
    }

    override fun onMapReady(map: GoogleMap) {
        this.map = map
        myLocationBtn.setOnClickListener { moveToMyLocation(zoom = false, animation = true) }
        val location = intent?.getSerializableExtra("location") as? LocationParam
        if (location != null)
            openLocation(location)
        else
            chooseLocation()
    }

    private fun openLocation(location: LocationParam) {
        chooseBtn.hide()
        currentMarker.hide()
        val position = LatLng(location.latitude, location.longitude)
        map?.addMarker(
            MarkerOptions()
                .icon(BitmapDescriptorFactory.fromResource(R.drawable.boxo_map_marker))
                .anchor(0.5f, 1f)
                .position(position)
        )
        map?.moveCamera(CameraUpdateFactory.newLatLngZoom(position, location.scale?.toFloat() ?: 14f))
        setResult(Activity.RESULT_OK)
    }

    private fun chooseLocation() {
        chooseBtn.show()
        currentMarker.show()
        chooseBtn.setOnClickListener {
            intent.putExtra("latitude", map!!.cameraPosition.target.latitude)
                .putExtra("longitude", map!!.cameraPosition.target.longitude)
            setResult(Activity.RESULT_OK, intent)
            finish()
        }
        moveToMyLocation()
    }

    private fun moveToLocation(location: Location, zoom: Boolean, animation: Boolean) {
        val position = LatLng(location.latitude, location.longitude)
        if (animation) {
            if (zoom)
                map?.animateCamera(CameraUpdateFactory.newLatLngZoom(position, 14f))
            else
                map?.animateCamera(CameraUpdateFactory.newLatLng(position))
        } else {
            if (zoom)
                map?.moveCamera(CameraUpdateFactory.newLatLngZoom(position, 14f))
            else
                map?.moveCamera(CameraUpdateFactory.newLatLng(position))
        }

    }

    @SuppressLint("MissingPermission")
    private fun moveToMyLocation(zoom: Boolean = true, animation: Boolean = false) {
        permissions.location(onAccess = {
            if (isLocationEnabled) {
                map?.isMyLocationEnabled = true
                map?.uiSettings?.isMyLocationButtonEnabled = false
                fusedLocationClient.lastLocation
                    .addOnCompleteListener {
                        if (it.result != null) {
                            moveToLocation(it.result!!, zoom, animation)
                        } else {
                            fusedLocationClient.requestLocationUpdates(
                                LocationRequest.Builder(5 * 1000)
                                    .setPriority(Priority.PRIORITY_HIGH_ACCURACY).build(),
                                object : LocationCallback() {
                                    override fun onLocationResult(result: LocationResult) {
                                        result.lastLocation?.also { location ->
                                            moveToLocation(location, zoom, animation)
                                            fusedLocationClient.removeLocationUpdates(this)
                                        }
                                    }
                                },
                                null
                            )
                        }
                    }
            } else {
                showDialog(
                    message = getString(R.string.boxo_system_location_disabled_text),
                    positive = getString(R.string.boxo_open_location_settings_btn),
                    onPositiveClick = { forResult.openLocationSettings { moveToMyLocation() } }
                )
            }
        }, onDenied = {}, onForeverDenied = {
            showDialog(
                message = getString(R.string.boxo_location_permission_denied_text),
                positive = getString(R.string.boxo_open_settings_btn),
                onPositiveClick = { forResult.openSettings { moveToMyLocation() } }
            )
        })
    }

    override fun onActivityResult(requestCode: Int, resultCode: Int, data: Intent?) {
        val result = forResult.onActivityResult(requestCode, resultCode, data)
        if (!result) super.onActivityResult(requestCode, resultCode, data)
    }

    override fun onRequestPermissionsResult(requestCode: Int, permissions: Array<String>, grantResults: IntArray) {
        super.onRequestPermissionsResult(requestCode, permissions, grantResults)
        this.permissions.onRequestPermissionsResult(this, requestCode, permissions, grantResults)
    }
}