package io.boxo.ui.qrscanner

import android.app.Activity
import android.content.Context
import android.content.Intent
import android.os.Bundle
import android.view.ViewGroup
import android.widget.ImageView
import android.widget.TextView
import io.boxo.R
import io.boxo.utils.Permissions
import io.boxo.utils.extensions.bindView
import io.boxo.utils.extensions.fullscreen
import io.boxo.utils.extensions.openSettings
import io.boxo.utils.extensions.setMiniappLocale
import io.boxo.utils.extensions.showDialog
import com.google.zxing.Result
import me.dm7.barcodescanner.zxing.ZXingScannerView
import me.dm7.barcodescanner.zxing.ZXingScannerView.ResultHandler


class QRScannerActivity : Activity(), ResultHandler {
    private var _scannerView: ZXingScannerView? = null

    private val flashIcon by bindView<ImageView>(R.id.flash_icon)
    private val flashText by bindView<TextView>(R.id.flash_text)

    private val permissions by lazy { Permissions(this) }

    override fun attachBaseContext(newBase: Context?) {
        setMiniappLocale()
        super.attachBaseContext(newBase)
    }

    override fun onCreate(state: Bundle?) {
        super.onCreate(state)
        setContentView(R.layout.boxo_activity_qr_scanner)
        fullscreen()
        requestCameraPermission()
    }

    private fun requestCameraPermission() {
        permissions.camera(this, onAccess = {
            init()
            start()
        }, onDenied = {
            finish()
        }, onForeverDenied = {
            showDialog(
                message = getString(R.string.boxo_camera_permission_denied_text),
                positive = getString(R.string.boxo_open_settings_btn),
                onNegativeClick = { finish() },
                onPositiveClick = { openSettings(SETTINGS_REQUEST_CODE) }
            )
        })
    }

    private fun init() {
        val contentFrame = findViewById<ViewGroup>(R.id.content_frame)
        _scannerView = ZXingScannerView(this)
        contentFrame.addView(_scannerView)
        flashIcon.setOnClickListener {
            _scannerView?.toggleFlash()
            if (_scannerView!!.flash) {
                flashIcon.setImageResource(R.drawable.boxo_flash_on)
                flashText.setText(R.string.boxo_flash_on)
            } else {
                flashIcon.setImageResource(R.drawable.boxo_flash_off)
                flashText.setText(R.string.boxo_flash_off)
            }
        }
    }

    private fun start() {
        _scannerView?.setResultHandler(this)
        _scannerView?.startCamera()
    }

    override fun onResume() {
        super.onResume()
        start()
    }

    override fun onPause() {
        super.onPause()
        _scannerView?.stopCamera()
    }

    override fun onRequestPermissionsResult(requestCode: Int, permissions: Array<String>, grantResults: IntArray) {
        this.permissions.onRequestPermissionsResult(this, requestCode, permissions, grantResults)
    }

    override fun onActivityResult(requestCode: Int, resultCode: Int, data: Intent?) {
        if (requestCode == SETTINGS_REQUEST_CODE)
            requestCameraPermission()
        else
            super.onActivityResult(requestCode, resultCode, data)
    }

    override fun handleResult(rawResult: Result) {
        setResult(RESULT_OK, intent.putExtra(QR_RESULT, rawResult.text))
        finish()
    }

    companion object {
        private const val SETTINGS_REQUEST_CODE = 1001
        const val QR_RESULT = "qr_result"
    }
}