package io.boxo.ui.settings

import android.app.Activity
import android.content.Context
import android.content.Intent
import android.os.Bundle
import android.widget.ImageButton
import androidx.activity.viewModels
import androidx.appcompat.app.AppCompatActivity
import androidx.appcompat.widget.SwitchCompat
import androidx.core.view.isVisible
import androidx.lifecycle.AbstractSavedStateViewModelFactory
import androidx.lifecycle.SavedStateHandle
import androidx.lifecycle.ViewModel
import androidx.savedstate.SavedStateRegistryOwner
import io.boxo.R
import io.boxo.ServiceLocator
import io.boxo.sdk.Boxo
import io.boxo.ui.main.BoxoActivity.Companion.APP_ID
import io.boxo.utils.extensions.applyBoxoStatusBarByTheme
import io.boxo.utils.extensions.bindView
import io.boxo.utils.extensions.setDefaultTheme
import io.boxo.utils.extensions.setMiniappLocale

internal class MiniappSettingsActivity : AppCompatActivity() {
    private val appId by lazy { intent.getStringExtra(APP_ID)!! }

    inner class ViewModelFactory(
        private val owner: SavedStateRegistryOwner,
    ) : AbstractSavedStateViewModelFactory(owner, null) {
        override fun <T : ViewModel> create(
            key: String, modelClass: Class<T>, state: SavedStateHandle
        ): T {
            if (appId.isBlank()) finish()
            return MiniappSettingsViewModel(
                appId,
                ServiceLocator.networkService,
                ServiceLocator.permissions,
                ServiceLocator.storage
            ) as T
        }
    }

    private val viewModel: MiniappSettingsViewModel by viewModels { ViewModelFactory(this) }

    private val backBtn by bindView<ImageButton>(R.id.back_button)
    private val userInfoSwitch by bindView<SwitchCompat>(R.id.user_info_switch)
    private val locationSwitch by bindView<SwitchCompat>(R.id.location_switch)
    private val cameraSwitch by bindView<SwitchCompat>(R.id.camera_switch)

    override fun attachBaseContext(newBase: Context?) {
        setMiniappLocale()
        super.attachBaseContext(newBase)
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        setDefaultTheme(Boxo)
        applyBoxoStatusBarByTheme(Boxo)
        setContentView(R.layout.boxo_activity_settings)
        backBtn.setOnClickListener { finish() }
        viewModel.hasLocationPermission.observe(this) {
            locationSwitch.setOnCheckedChangeListener(null)
            locationSwitch.isChecked = it
            setLocationCheckedChangeListener()
        }
        viewModel.hasCameraPermission.observe(this) {
            cameraSwitch.setOnCheckedChangeListener(null)
            cameraSwitch.isChecked = it
            setCameraCheckedChangeListener()
        }
        userInfoSwitch.isVisible = viewModel.isConsentEnabled
        viewModel.isUserDataConsented.observe(this) { isConsented ->
            userInfoSwitch.setOnCheckedChangeListener(null)
            userInfoSwitch.isChecked = isConsented
            userInfoSwitch.isEnabled = isConsented
            setUserDataCheckedChangeListener()
        }
    }

    private fun setLocationCheckedChangeListener() {
        locationSwitch.setOnCheckedChangeListener { _, isChecked ->
            viewModel.allowLocation(isChecked)
        }
    }

    private fun setCameraCheckedChangeListener() {
        cameraSwitch.setOnCheckedChangeListener { _, isChecked ->
            viewModel.allowCamera(isChecked)
        }
    }

    private fun setUserDataCheckedChangeListener() {
        userInfoSwitch.setOnCheckedChangeListener { _, isChecked ->
            if (!isChecked) {
                viewModel.revokeUserConsent()
                setResult(Activity.RESULT_OK, Intent().apply {
                    putExtra("revoke", true)
                })
            }
        }
    }
}