package io.boxo.ui.settings

import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import io.boxo.data.network.NetworkService
import io.boxo.data.storage.IBoxoPermissions
import io.boxo.data.storage.IBoxoStorage
import kotlinx.coroutines.launch

internal class MiniappSettingsViewModel(
    val appId: String,
    private val networkService: NetworkService,
    private val permissionStorage: IBoxoPermissions,
    private val storage: IBoxoStorage
) : ViewModel() {
    val isConsentEnabled = storage.getMiniappSettings(appId)?.hostappInfo?.isConsentEnabled ?: true
    val hasLocationPermission = MutableLiveData(permissionStorage.hasLocationPermission(appId))
    val hasCameraPermission = MutableLiveData(permissionStorage.hasCameraPermission(appId))
    val isUserDataConsented = MutableLiveData(permissionStorage.hasUserDataPermission(appId))

    init {
        getUserConsent()
    }

    private fun getUserConsent() {
        viewModelScope.launch {
            runCatching { networkService.getUserConsent(appId) }
                .onSuccess {
                    val isConsented = it.optBoolean("is_consented")
                    permissionStorage.allowUserDataPermission(appId, isConsented)
                    isUserDataConsented.postValue(isConsented)
                }
        }
    }

    fun allowLocation(checked: Boolean) {
        hasLocationPermission.postValue(checked)
        permissionStorage.allowLocationPermission(appId, checked)
    }

    fun allowCamera(checked: Boolean) {
        hasCameraPermission.postValue(checked)
        permissionStorage.allowCameraPermission(appId, checked)
    }

    fun revokeUserConsent() {
        viewModelScope.launch {
            permissionStorage.allowUserDataPermission(appId, false)
            runCatching { networkService.revokeUserConsent(appId) }
                .onSuccess { isUserDataConsented.postValue(false) }
        }
    }
}