package io.boxo.ui.view

import android.content.Context
import android.graphics.Color
import android.util.AttributeSet
import android.view.View
import android.widget.FrameLayout
import android.widget.ImageView
import android.widget.LinearLayout
import androidx.annotation.ColorInt
import androidx.annotation.DrawableRes
import androidx.constraintlayout.widget.ConstraintLayout
import androidx.constraintlayout.widget.ConstraintSet
import androidx.core.view.isVisible
import androidx.core.view.setPadding
import io.boxo.R
import io.boxo.js.params.ActionButton
import io.boxo.utils.extensions.addBackgroundCircleRipple
import io.boxo.utils.extensions.bindView
import io.boxo.utils.extensions.hide
import io.boxo.utils.extensions.setMargin
import io.boxo.utils.extensions.show
import io.boxo.utils.extensions.toPx

internal class ActionButtonsView : FrameLayout {
    private val moreBtn by bindView<ImageView>(R.id.more_button)
    private val closeBtn by bindView<ImageView>(R.id.close_button)
    private val customBtn by bindView<ImageView>(R.id.custom_button)
    private val divider by bindView<View>(R.id.divider)
    private val dividerCustomBtn by bindView<View>(R.id.divider_custom_button)
    private val container by bindView<LinearLayout>(R.id.container)

    private var closeClickListener: ((View) -> Unit)? = null
    private var moreClickListener: ((View) -> Unit)? = null
    private var customButtonClickListener: ((View) -> Unit)? = null

    private var isLight = false
    private var style: String = ActionButton.Style.default.name
    private var visible: Boolean = true

    private var cMainHeight = 48.toPx
    private var cHeight = 32.toPx
    private var cWidthWithCustomBtn = 128.toPx
    private var cWidth = 84.toPx

    constructor(context: Context) : this(context, null)
    constructor(context: Context, attrs: AttributeSet?) : this(context, attrs, 0)
    constructor(context: Context, attrs: AttributeSet?, defStyleAttr: Int) : super(
        context,
        attrs,
        defStyleAttr
    ) {
        init()
    }

    private fun setSizes(
        containerHeight: Int,
        buttonWidth: Int,
        buttonHeight: Int,
        dividerSize: Int,
        moreButtonEnabled: Boolean
    ) {
        cMainHeight = containerHeight.toPx
        cHeight = buttonHeight.toPx
        cWidth = (if (moreButtonEnabled) 2 * buttonWidth + dividerSize else buttonWidth).toPx
        cWidthWithCustomBtn = (if (moreButtonEnabled) 3 * buttonWidth + 2 * dividerSize
        else 2 * buttonWidth + dividerSize).toPx
    }

    private fun init() {
        inflate(context, R.layout.boxo_action_buttons, this)
    }

    fun setOnMoreClickListener(listener: (View) -> Unit) {
        moreClickListener = listener
        moreBtn.setOnClickListener(moreClickListener)
    }

    fun setOnCloseClickListener(listener: (View) -> Unit) {
        closeClickListener = listener
        closeBtn.setOnClickListener(closeClickListener)
    }

    fun setOnCustomButtonClickListener(listener: (View) -> Unit) {
        customButtonClickListener = listener
        customBtn.setOnClickListener(customButtonClickListener)
    }

    fun showCustomButton(@DrawableRes id: Int) {
        customBtn.setImageResource(id)
        customBtn.show()
        dividerCustomBtn.show()
        updateContainerSize()
    }

    fun hideCustomButton() {
        customBtn.hide()
        dividerCustomBtn.hide()
        updateContainerSize()
    }

    private fun updateContainerSize() {
        layoutParams.apply {
            height = cMainHeight
        }
        container.layoutParams.apply {
            height = cHeight
            width = if (customBtn.isVisible) cWidthWithCustomBtn else cWidth
        }
        moreBtn.changeSize(cHeight)
        customBtn.changeSize(cHeight)
        closeBtn.changeSize(cHeight)
    }

    private fun buttonPaddings(size: Int) {
        moreBtn.setPadding(size)
        customBtn.setPadding(size)
        closeBtn.setPadding(size)
    }

    private fun View.changeSize(size: Int) {
        layoutParams.apply {
            width = size
            height = size
        }
    }

    private fun dividersForDefault() {
        divider.layoutParams.apply { width = 1.toPx }
        dividerCustomBtn.layoutParams.apply { width = 1.toPx }
        divider.setBackgroundResource(R.color.boxo_menu_divider_color)
        dividerCustomBtn.setBackgroundResource(R.color.boxo_menu_divider_color)
    }

    private fun transparentDivider(size: Int) {
        divider.layoutParams.apply { width = size.toPx }
        dividerCustomBtn.layoutParams.apply { width = size.toPx }
        divider.setBackgroundResource(0)
        dividerCustomBtn.setBackgroundResource(0)
    }

    private fun changePosition(pos: Int, margin: Int) {
        val constraintLayout = parent as? ConstraintLayout ?: return
        val constraintSet = ConstraintSet()
        constraintSet.clone(constraintLayout)
        constraintSet.apply {
            connect(
                R.id.action_buttons,
                if (pos == 0) ConstraintSet.END else ConstraintSet.START,
                ConstraintSet.PARENT_ID,
                if (pos == 0) ConstraintSet.END else ConstraintSet.START,
                margin.toPx
            )
            clear(R.id.action_buttons, if (pos == 0) ConstraintSet.START else ConstraintSet.END)
            applyTo(constraintLayout)
        }
    }

    private fun topLeft(margin: Int = 14) = changePosition(1, margin)

    private fun topRight(margin: Int = 14) = changePosition(0, margin)

    private fun iconColors(isLight: Boolean, @ColorInt forLight: Int, @ColorInt forDark: Int) {
        if (isLight) {
            moreBtn.setColorFilter(forLight)
            closeBtn.setColorFilter(forLight)
        } else {
            moreBtn.setColorFilter(forDark)
            closeBtn.setColorFilter(forDark)
        }
    }

    fun set(actionButton: ActionButton) {
        actionButton.isLight?.also { isLight = it }
        actionButton.style?.name?.also { style = it }
        actionButton.visible?.also { visible ->
            this.visible = visible
        }
        post {
            when (style) {
                ActionButton.Style.style_1.name -> {
                    setSizes(72, 40, 40, 8, true)
                    transparentDivider(8)
                    topRight()
                    moreBtn.isVisible = true
                    divider.isVisible = true
                    container.setBackgroundResource(0)
                    moreBtn.setBackgroundResource(R.drawable.boxo_menu_style1_bg)
                    closeBtn.setBackgroundResource(R.drawable.boxo_menu_style1_bg)
                    customBtn.setBackgroundResource(R.drawable.boxo_menu_style1_bg)
                    moreBtn.setImageResource(R.drawable.boxo_ic_menu_dots)
                    closeBtn.setImageResource(R.drawable.boxo_ic_cross)
                    buttonPaddings(10.toPx)
                    iconColors(isLight, Color.BLACK, Color.BLACK)
                    container.setMargin(left = 4.toPx, right = 4.toPx)
                }

                ActionButton.Style.style_2.name -> {
                    setSizes(48, 24, 24, 12, true)
                    transparentDivider(12)
                    topRight()
                    moreBtn.isVisible = true
                    divider.isVisible = true
                    moreBtn.setBackgroundResource(R.drawable.boxo_action_transparent_bg)
                    closeBtn.setBackgroundResource(R.drawable.boxo_action_transparent_bg)
                    customBtn.setBackgroundResource(R.drawable.boxo_action_transparent_bg)
                    moreBtn.setImageResource(R.drawable.boxo_ic_style_2_more)
                    closeBtn.setImageResource(R.drawable.boxo_ic_style_2_close)
                    container.setBackgroundColor(Color.TRANSPARENT)
                    buttonPaddings(0.toPx)
                    val color = Color.parseColor("#3F3D44")
                    iconColors(isLight, color, color)
                    container.setMargin()
                }

                ActionButton.Style.style_3.name -> {
                    setSizes(48, 40, 40, 8, true)
                    transparentDivider(8)
                    topRight()
                    moreBtn.isVisible = true
                    divider.isVisible = true
                    moreBtn.setBackgroundResource(R.drawable.boxo_action_transparent_bg)
                    closeBtn.setBackgroundResource(R.drawable.boxo_action_transparent_bg)
                    customBtn.setBackgroundResource(R.drawable.boxo_action_transparent_bg)
                    moreBtn.setImageResource(R.drawable.boxo_ic_menu_dots)
                    closeBtn.setImageResource(R.drawable.boxo_ic_cross)
                    container.setBackgroundColor(Color.TRANSPARENT)
                    buttonPaddings(10.toPx)
                    iconColors(isLight, Color.BLACK, Color.BLACK)
                    container.setMargin()
                }

                ActionButton.Style.style_4.name -> {
                    setSizes(64, 36, 36, 8, false)
                    transparentDivider(8)
                    topLeft(8)
                    moreBtn.isVisible = false
                    divider.isVisible = false
                    closeBtn.setBackgroundResource(R.drawable.boxo_action_transparent_bg)
                    customBtn.setBackgroundResource(R.drawable.boxo_action_transparent_bg)
                    closeBtn.setImageResource(R.drawable.boxo_ic_style_4_back)
                    container.setBackgroundColor(Color.TRANSPARENT)
                    buttonPaddings(0.toPx)
                    val color = Color.parseColor("#373E4D")
                    iconColors(isLight, color, color)
                    container.setMargin()
                }

                else -> {//default
                    setSizes(48, 41, 32, 1, true)
                    dividersForDefault()
                    topRight()
                    moreBtn.isVisible = true
                    divider.isVisible = true
                    moreBtn.addBackgroundCircleRipple()
                    closeBtn.addBackgroundCircleRipple()
                    customBtn.addBackgroundCircleRipple()
                    moreBtn.setImageResource(R.drawable.boxo_ic_more)
                    closeBtn.setImageResource(R.drawable.boxo_ic_close)
                    container.setBackgroundResource(
                        if (isLight) R.drawable.boxo_ab_default_light
                        else R.drawable.boxo_ab_default_dark
                    )
                    buttonPaddings(3.toPx)
                    iconColors(isLight, Color.BLACK, Color.WHITE)
                    container.setMargin()
                }
            }
            updateContainerSize()
            isVisible = visible
        }
    }
}