package io.boxo.ui.view

import android.app.Activity
import android.content.Context
import android.graphics.Color
import android.os.Parcel
import android.os.Parcelable
import android.util.AttributeSet
import android.view.View
import android.widget.FrameLayout
import android.widget.ImageButton
import android.widget.TextView
import androidx.constraintlayout.widget.ConstraintLayout
import androidx.constraintlayout.widget.ConstraintLayout.LayoutParams.BOTTOM
import androidx.constraintlayout.widget.ConstraintLayout.LayoutParams.TOP
import androidx.constraintlayout.widget.ConstraintSet
import androidx.core.graphics.ColorUtils
import io.boxo.R
import io.boxo.js.params.NavigationBar
import io.boxo.utils.UiConst
import io.boxo.utils.extensions.*


internal class BoxoAppBar : FrameLayout {
    private val toolbar by bindView<View>(R.id.toolbar)
    private val backBtn by bindView<ImageButton>(R.id.back_button)
    private val titleTV by bindView<TextView>(R.id.title)
    private var contentView: View? = null
    private var _lastScrollY = 0
    private var isFullscreen = false

    private var _changeBackgroundOnScroll = false
    private var _frontColorWhenTransparent = Color.BLACK
    private var _backgroundColor: Int = Color.WHITE
    private var _isBackgroundTransparent = false
    private var _frontColor: Int = Color.BLACK
    private var _toolbarShown = false
    private var _backButton = false
    private var _ready = false

    private var _navigationBar: NavigationBar? = null

    constructor(context: Context) : this(context, null)
    constructor(context: Context, attrs: AttributeSet?) : this(context, attrs, 0)
    constructor(context: Context, attrs: AttributeSet?, defStyleAttr: Int) : super(context, attrs, defStyleAttr) {
        inflate(context, R.layout.boxo_appbar, this)
    }

    fun init(view: View, isFullscreen: Boolean, statusBarBackground: String) {
        contentView = view
        this.isFullscreen = isFullscreen

        contentView?.viewTreeObserver?.addOnScrollChangedListener {
            _lastScrollY = contentView!!.scrollY
            onContentScroll()
        }
        if (!isFullscreen) {
            (parent as View).setMarginTop(UiConst.statusBarHeight)
            setMarginTop(0)
        }
        set(_navigationBar ?: NavigationBar(background = statusBarBackground))
    }

    fun ready() {
        _ready = true
        apply()
    }

    fun setNavigationOnClickListener(listener: (View) -> Unit) {
        backBtn.setOnClickListener(listener)
    }

    fun set(navigationBar: NavigationBar) {
        _navigationBar = _navigationBar?.merge(navigationBar) ?: navigationBar
        navigationBar.background?.also { color -> _backgroundColor = Color.parseColor(color) }
        navigationBar.frontColor?.also { color -> _frontColor = Color.parseColor(color) }
        navigationBar.isBackgroundTransparent?.also { _isBackgroundTransparent = it }
        navigationBar.changeBackgroundOnScroll?.also { _changeBackgroundOnScroll = it }
        navigationBar.frontColorWhenTransparent?.also { _frontColorWhenTransparent = Color.parseColor(it) }
        navigationBar.title?.also { title -> titleTV.text = title }
        navigationBar.show?.also { show -> _toolbarShown = show }
        navigationBar.backButton?.also { _backButton = it }
        apply()
    }

    private fun apply() {
        if (_ready) {
            toolbar.show(_toolbarShown)
            backBtn.show(_backButton)
            if (_isBackgroundTransparent) {
                changeStatusTheme(_backgroundColor)
                if (_toolbarShown) {
                    setBackgroundColor(Color.TRANSPARENT)
                    contentUnderNavbar()
                    setToolbarColor(_frontColorWhenTransparent)
                    if (_changeBackgroundOnScroll)
                        changeColorsOnScroll()
                } else
                    contentUnderNavbar()
            } else {
                setBackgroundColor(_backgroundColor)
                changeStatusTheme(_backgroundColor)
                setToolbarColor(_frontColor)
                if (_toolbarShown)
                    contentBelowNavbar(_backgroundColor)
                else
                    contentUnderNavbar()
            }
        }
    }

    private fun contentUnderNavbar() {
        if (isFullscreen)
            statusBarColor(Color.TRANSPARENT)
        else
            statusBarColor(_backgroundColor)
        if (contentView != null)
            (parent as? ConstraintLayout)?.apply {
                val set = ConstraintSet()
                set.clone(this)
                set.connect(contentView!!.id, TOP, this.id, TOP)
                set.applyTo(this)
            }
    }

    private fun contentBelowNavbar(color: Int) {
        statusBarColor(color)
        if (contentView != null)
            (parent as? ConstraintLayout)?.apply {
                val set = ConstraintSet()
                set.clone(this)
                set.connect(contentView!!.id, TOP, R.id.app_bar, BOTTOM)
                set.applyTo(this)
            }
    }

    private fun setToolbarColor(color: Int) {
        titleTV.setTextColor(color)
        backBtn.setColorFilter(color)
    }

    private fun statusBarColor(color: Int) {
        (context as? Activity)?.setStatusBarColor(color)
    }

    private fun changeStatusTheme(color: Int) {
        (context as? Activity)?.changeStatusTheme(color)
    }

    private fun onContentScroll() {
        if (_changeBackgroundOnScroll && _toolbarShown) {
            changeColorsOnScroll()
        }
    }

    private fun changeColorsOnScroll() {
        var color = ColorUtils.setAlphaComponent(_backgroundColor, 0xFF)
        if (_lastScrollY < 256) {
            val alpha = _lastScrollY shl 24 or (-1 ushr 8)
            color = color and alpha
        }
        if (_lastScrollY > 128) {
            setToolbarColor(_frontColor)
            changeStatusTheme(_frontColorWhenTransparent)
        } else {
            setToolbarColor(_frontColorWhenTransparent)
            changeStatusTheme(_frontColor)
        }
        setBackgroundColor(color)
        statusBarColor(color)
    }

    override fun onSaveInstanceState(): Parcelable? {
        val superState: Parcelable? = super.onSaveInstanceState()
        superState?.let {
            val state = SavedState(superState)
            state.navBar = this._navigationBar
            return state
        } ?: run {
            return superState
        }
    }

    override fun onRestoreInstanceState(state: Parcelable?) {
        when (state) {
            is SavedState -> {
                super.onRestoreInstanceState(state.superState)
                this._navigationBar = state.navBar
                _navigationBar?.let { set(it) }
            }
            else -> {
                super.onRestoreInstanceState(state)
            }
        }
    }

    internal class SavedState : BaseSavedState {
        var navBar: NavigationBar? = null

        constructor(superState: Parcelable) : super(superState)


        constructor(source: Parcel) : super(source) {
            navBar = source.readParcelable(NavigationBar::class.java.classLoader)
        }

        override fun writeToParcel(parcel: Parcel, flags: Int) {
            super.writeToParcel(parcel, flags)
            parcel.writeParcelable(navBar, flags)
        }

        companion object CREATOR : Parcelable.Creator<SavedState> {
            override fun createFromParcel(parcel: Parcel): SavedState {
                return SavedState(parcel)
            }

            override fun newArray(size: Int): Array<SavedState?> {
                return newArray(size)
            }
        }
    }
}