package io.boxo.ui.view

import android.content.Context
import android.graphics.Color
import android.graphics.Rect
import android.util.AttributeSet
import android.view.MotionEvent
import android.view.View
import android.view.ViewGroup
import android.widget.FrameLayout
import android.widget.ImageView
import android.widget.LinearLayout
import androidx.annotation.DrawableRes
import androidx.core.view.isVisible
import io.boxo.R
import io.boxo.js.params.ActionButton
import io.boxo.utils.extensions.bindView
import io.boxo.utils.extensions.hide
import io.boxo.utils.extensions.show
import io.boxo.utils.extensions.toPx
import kotlin.math.abs
import kotlin.math.max
import kotlin.math.min

internal class FloatingActionButtonsView : FrameLayout {
    companion object {
        const val DRAG_TOLERANCE = 16
        const val DURATION_MILLIS = 250L
    }

    private var widgetInitialX = 0f
    private var widgetDX = 0f
    private var widgetInitialY = 0f
    private var widgetDY = 0f

    private val viewParent by lazy { parent as View }
    private val parentHeight by lazy { viewParent.height }
    private val parentWidth by lazy { viewParent.width }
    private val xMax by lazy { parentWidth - width }
    private val xMiddle by lazy { parentWidth / 2 }
    private val yMax by lazy { parentHeight - height }

    private val moreBtn by bindView<ImageView>(R.id.more_button)
    private val closeBtn by bindView<ImageView>(R.id.close_button)
    private val customBtn by bindView<ImageView>(R.id.custom_button)
    private val container by bindView<LinearLayout>(R.id.container)

    private var closeClickListener: ((View) -> Unit)? = null
    private var moreClickListener: ((View) -> Unit)? = null
    private var customButtonClickListener: ((View) -> Unit)? = null

    private var isLight = false
    private var visible = false

    constructor(context: Context) : this(context, null)
    constructor(context: Context, attrs: AttributeSet?) : this(context, attrs, 0)
    constructor(context: Context, attrs: AttributeSet?, defStyleAttr: Int) : super(context, attrs, defStyleAttr) {
        init()
    }

    private fun init() {
        inflate(context, R.layout.boxo_action_buttons_floating, this)
    }

    fun setOnMoreClickListener(listener: (View) -> Unit) {
        moreClickListener = listener
    }

    fun setOnCloseClickListener(listener: (View) -> Unit) {
        closeClickListener = listener
    }

    fun setOnCustomButtonClickListener(listener: (View) -> Unit) {
        customButtonClickListener = listener
    }

    fun showCustomButton(@DrawableRes id: Int) {
        container.layoutParams.apply {
            height = 150.toPx
            width = 40.toPx
        }
        customBtn.setImageResource(id)
        customBtn.show()
        setFloatingActionButtonsBg()
    }

    fun hideCustomButton() {
        container.layoutParams.apply {
            height = 100.toPx
            width = 40.toPx
        }
        customBtn.hide()
        setFloatingActionButtonsBg()
    }

    fun set(actionButton: ActionButton) {
        actionButton.isLight?.also { isLight = it }
        actionButton.visible?.also { visible ->
            this.visible = visible
        }
        post {
            if (isLight) {
                moreBtn.setColorFilter(Color.BLACK)
                closeBtn.setColorFilter(Color.BLACK)
            } else {
                moreBtn.setColorFilter(Color.WHITE)
                closeBtn.setColorFilter(Color.WHITE)
            }
            setFloatingActionButtonsBg()
            isVisible = visible
        }
    }

    private fun setFloatingActionButtonsBg(rawX: Float = 0f) {
        val x = if (rawX > 0) rawX else x
        if (x >= xMiddle)
            container.setBackgroundResource(
                if (isLight)
                    if (customBtn.visibility == View.VISIBLE) R.drawable.boxo_ab_3_light_right
                    else R.drawable.boxo_ab_light_right
                else
                    if (customBtn.visibility == View.VISIBLE) R.drawable.boxo_ab_3_dark_right
                    else R.drawable.boxo_ab_dark_right
            )
        else
            container.setBackgroundResource(
                if (isLight)
                    if (customBtn.visibility == View.VISIBLE) R.drawable.boxo_ab_3_light_left
                    else R.drawable.boxo_ab_light_left
                else
                    if (customBtn.visibility == View.VISIBLE) R.drawable.boxo_ab_3_dark_left
                    else R.drawable.boxo_ab_dark_left
            )
    }

    override fun onTouchEvent(event: MotionEvent): Boolean {
        when (event.actionMasked) {
            MotionEvent.ACTION_DOWN -> {
                widgetDX = x - event.rawX
                widgetDY = y - event.rawY
                widgetInitialX = x
                widgetInitialY = y
            }
            MotionEvent.ACTION_MOVE -> {
                var newX = event.rawX + widgetDX
                newX = max(0F, newX)
                newX = min(xMax.toFloat(), newX)
                x = newX

                var newY = event.rawY + widgetDY
                newY = max(0F, newY)
                newY = min(yMax.toFloat(), newY)
                y = newY

                if (parentWidth - container.width - 30 > x && x > 30)
                    container.setBackgroundResource(
                        if (isLight)
                            if (customBtn.visibility == View.VISIBLE) R.drawable.boxo_ab_3_light_center
                            else R.drawable.boxo_ab_light_center
                        else
                            if (customBtn.visibility == View.VISIBLE) R.drawable.boxo_ab_3_dark_center
                            else R.drawable.boxo_ab_dark_center
                    )
            }
            MotionEvent.ACTION_UP -> {
                if (event.rawX >= xMiddle) {
                    animate().x(xMax.toFloat())
                        .setDuration(DURATION_MILLIS)
                        .setUpdateListener { setFloatingActionButtonsBg(event.rawX) }
                        .start()
                } else {
                    animate().x(0F).setDuration(DURATION_MILLIS)
                        .setUpdateListener { setFloatingActionButtonsBg(event.rawX) }
                        .start()
                }
                if (abs(x - widgetInitialX) <= DRAG_TOLERANCE && abs(y - widgetInitialY) <= DRAG_TOLERANCE) {
                    when (container.getViewByCoordinates(event.x, event.y)) {
                        moreBtn -> {
                            moreClickListener?.invoke(moreBtn)
                        }
                        closeBtn -> {
                            closeClickListener?.invoke(closeBtn)
                        }
                        customBtn -> {
                            customButtonClickListener?.invoke(customBtn)
                        }
                    }
                }
            }
            else -> return false
        }
        return true
    }

    private fun ViewGroup.getViewByCoordinates(x: Float, y: Float): View? {
        (childCount - 1 downTo 0)
            .map { this.getChildAt(it) }
            .forEach {
                val bounds = Rect()
                it.getHitRect(bounds)
                if (bounds.contains(x.toInt(), y.toInt())) {
                    return it
                }
            }
        return null
    }
}