package io.boxo.utils

import android.app.Activity
import android.content.Intent
import android.net.Uri
import android.provider.Settings
import io.boxo.js.params.LocationParam
import io.boxo.ui.gallery.GalleryActivity
import io.boxo.ui.map.BoxoMapActivity
import io.boxo.ui.qrscanner.QRScannerActivity

internal class ActivityResultHelper(val activity: Activity) {
    private val QR_CODE_REQUEST_CODE = 1000
    private val LOCATION_PERMISSION_SETTINGS_REQUEST_CODE = 1001
    private val LOCATION_SETTINGS_REQUEST_CODE = 1002
    private val GALLERY_REQUEST_CODE = 1003
    private val OPEN_LOCATION_REQUEST_CODE = 1004
    private val CHOOSE_LOCATION_REQUEST_CODE = 1005

    private var onQrCodeResult: ((qrCode: String?) -> Unit)? = null
    private var onImageGalleryResult: ((result: Boolean) -> Unit)? = null
    private var onSettingsResult: (() -> Unit)? = null
    private var onLocationSettingsResult: (() -> Unit)? = null
    private var onOpenLocationResult: ((result: Boolean) -> Unit)? = null
    private var onChooseLocationResult: ((lat: Double, lng: Double) -> Unit)? = null


    fun openQrCodeReader(onResult: (qrCode: String?) -> Unit) {
        onQrCodeResult = onResult
        activity.startActivityForResult(
            Intent(activity, QRScannerActivity::class.java),
            QR_CODE_REQUEST_CODE
        )
    }

    fun openImageGallery(images: List<String>, onResult: (result: Boolean) -> Unit) {
        onImageGalleryResult = onResult
        activity.startActivityForResult(
            Intent(activity, GalleryActivity::class.java).putStringArrayListExtra("images", ArrayList(images)),
            GALLERY_REQUEST_CODE
        )
    }

    fun openLocationSettings(onResult: () -> Unit) {
        onLocationSettingsResult = onResult
        activity.startActivityForResult(
            Intent(Settings.ACTION_LOCATION_SOURCE_SETTINGS),
            LOCATION_SETTINGS_REQUEST_CODE
        )
    }

    fun openSettings(onResult: () -> Unit) {
        onSettingsResult = onResult
        activity.startActivityForResult(
            Intent(Settings.ACTION_APPLICATION_DETAILS_SETTINGS).setData(
                Uri.fromParts("package", activity.packageName, null)
            ), LOCATION_PERMISSION_SETTINGS_REQUEST_CODE
        )
    }

    fun openOpenLocation(location: LocationParam, onResult: (Boolean) -> Unit) {
        onOpenLocationResult = onResult
        activity.startActivityForResult(
            Intent(activity, BoxoMapActivity::class.java)
                .putExtra("location", location),
            OPEN_LOCATION_REQUEST_CODE
        )
    }

    fun openChooseLocation(onResult: (lat: Double, lng: Double) -> Unit) {
        onChooseLocationResult = onResult
        activity.startActivityForResult(Intent(activity, BoxoMapActivity::class.java), CHOOSE_LOCATION_REQUEST_CODE)
    }

    fun onActivityResult(requestCode: Int, resultCode: Int, data: Intent?): Boolean {
        return when (requestCode) {
            QR_CODE_REQUEST_CODE -> {
                val result = data?.getStringExtra(QRScannerActivity.QR_RESULT)
                onQrCodeResult?.invoke(result)
                true
            }

            GALLERY_REQUEST_CODE -> {
                onImageGalleryResult?.invoke(resultCode == Activity.RESULT_OK)
                true
            }

            LOCATION_PERMISSION_SETTINGS_REQUEST_CODE -> {
                onSettingsResult?.invoke()
                true
            }

            LOCATION_SETTINGS_REQUEST_CODE -> {
                onLocationSettingsResult?.invoke()
                true
            }

            OPEN_LOCATION_REQUEST_CODE -> {
                onOpenLocationResult?.invoke(resultCode == Activity.RESULT_OK)
                true
            }

            CHOOSE_LOCATION_REQUEST_CODE -> {
                val lat = data?.getDoubleExtra("latitude", 0.0) ?: 0.0
                val lng = data?.getDoubleExtra("longitude", 0.0) ?: 0.0
                onChooseLocationResult?.invoke(lat, lng)
                true
            }
            else -> false
        }
    }

}