package io.boxo.utils

import android.content.Context
import android.content.Context.SENSOR_SERVICE
import android.content.Context.WINDOW_SERVICE
import android.hardware.Sensor
import android.hardware.Sensor.*
import android.hardware.SensorEvent
import android.hardware.SensorEventListener
import android.hardware.SensorManager
import android.view.Surface.*
import android.view.WindowManager
import io.boxo.sdk.Boxo

internal class Compass private constructor(private val context: Context) : SensorEventListener {

    var onNewAzimuth: ((azimuth: Int) -> Unit)? = null

    private var sensorManager: SensorManager = context.getSystemService(SENSOR_SERVICE) as SensorManager
    private var _rotationVectorSensor: Sensor?
    private var _magnetometerSensor: Sensor?
    private var _accelerometerSensor: Sensor?

    private var _useRotationVectorSensor = false

    private var _rotationVector = FloatArray(5)
    private var _geomagnetic = FloatArray(3)
    private var _gravity = FloatArray(3)

    private var _azimuthSensibility: Float = 0.toFloat()
    private var _lastAzimuthDegrees: Float = 0.toFloat()

    init {
        _magnetometerSensor = sensorManager.getDefaultSensor(TYPE_MAGNETIC_FIELD)
        _accelerometerSensor = sensorManager.getDefaultSensor(TYPE_ACCELEROMETER)
        _rotationVectorSensor = sensorManager.getDefaultSensor(TYPE_ROTATION_VECTOR)
    }

    // Check that the device has the required sensors
    private fun hasRequiredSensors(): Boolean {
        return if (_rotationVectorSensor != null) {
            Boxo.logger.debug("Sensor.TYPE_ROTATION_VECTOR found")
            true
        } else if (_magnetometerSensor != null && _accelerometerSensor != null) {
            Boxo.logger.debug("Sensor.TYPE_MAGNETIC_FIELD and Sensor.TYPE_ACCELEROMETER found")
            true
        } else {
            Boxo.logger.debug("The device does not have the required sensors")
            false
        }
    }

    private fun start(azimuthSensibility: Float) {
        _azimuthSensibility = azimuthSensibility
        if (_rotationVectorSensor != null) {
            sensorManager.registerListener(this, _rotationVectorSensor, SensorManager.SENSOR_DELAY_UI)
        }
        if (_magnetometerSensor != null) {
            sensorManager.registerListener(this, _magnetometerSensor, SensorManager.SENSOR_DELAY_UI)
        }
        if (_accelerometerSensor != null) {
            sensorManager.registerListener(this, _accelerometerSensor, SensorManager.SENSOR_DELAY_UI)
        }
    }

    fun start() {
        start(1f)
    }

    fun stop() {
        _azimuthSensibility = 0f
        sensorManager.unregisterListener(this)
    }

    // SensorEventListener
    override fun onSensorChanged(event: SensorEvent) {
        makeCalculations(event)
    }

    private fun makeCalculations(event: SensorEvent) {
        synchronized(this) {
            // Get the orientation array with Sensor.TYPE_ROTATION_VECTOR if possible (more precise), otherwise with Sensor.TYPE_MAGNETIC_FIELD and Sensor.TYPE_ACCELEROMETER combined
            val orientation = FloatArray(3)
            if (event.sensor.type == TYPE_ROTATION_VECTOR) {
                // Only use rotation vector sensor if it is working on this device
                if (!_useRotationVectorSensor) {
                    Boxo.logger.debug("Using Sensor.TYPE_ROTATION_VECTOR (more precise compass data)")
                    _useRotationVectorSensor = true
                }
                // Smooth values
                _rotationVector = exponentialSmoothing(
                    event.values, _rotationVector,
                    ROTATION_VECTOR_SMOOTHING_FACTOR
                )
                // Calculate the rotation matrix
                val rotationMatrix = FloatArray(9)
                SensorManager.getRotationMatrixFromVector(rotationMatrix, event.values)
                // Calculate the orientation
                SensorManager.getOrientation(rotationMatrix, orientation)
            } else if (!_useRotationVectorSensor && (event.sensor.type == TYPE_MAGNETIC_FIELD || event.sensor.type == TYPE_ACCELEROMETER)) {
                if (event.sensor.type == TYPE_MAGNETIC_FIELD) {
                    _geomagnetic = exponentialSmoothing(
                        event.values, _geomagnetic,
                        GEOMAGNETIC_SMOOTHING_FACTOR
                    )
                }
                if (event.sensor.type == TYPE_ACCELEROMETER) {
                    _gravity = exponentialSmoothing(
                        event.values, _gravity,
                        GRAVITY_SMOOTHING_FACTOR
                    )
                }
                // Calculate the rotation and inclination matrix
                val rotationMatrix = FloatArray(9)
                val inclinationMatrix = FloatArray(9)
                SensorManager.getRotationMatrix(rotationMatrix, inclinationMatrix, _gravity, _geomagnetic)
                // Calculate the orientation
                SensorManager.getOrientation(rotationMatrix, orientation)
            } else {
                return
            }

            // Calculate azimuth, pitch and roll values from the orientation[] array
            // Correct values depending on the screen rotation
            val screenRotation =
                (context.getSystemService(WINDOW_SERVICE) as WindowManager).defaultDisplay.rotation
            var mAzimuthDegrees = Math.toDegrees(orientation[0].toDouble()).toFloat()
            if (screenRotation == ROTATION_0) {
                val mRollDegrees = Math.toDegrees(orientation[2].toDouble()).toFloat()
                if (mRollDegrees >= 90 || mRollDegrees <= -90) {
                    mAzimuthDegrees += 180f
                }
            } else if (screenRotation == ROTATION_90) {
                mAzimuthDegrees += 90f
            } else if (screenRotation == ROTATION_180) {
                mAzimuthDegrees += 180f
                val mRollDegrees = (-Math.toDegrees(orientation[2].toDouble())).toFloat()
                if (mRollDegrees >= 90 || mRollDegrees <= -90) {
                    mAzimuthDegrees += 180f
                }
            } else if (screenRotation == ROTATION_270) {
                mAzimuthDegrees += 270f
            }

            // Force azimuth value between 0° and 360°.
            mAzimuthDegrees = (mAzimuthDegrees + 360) % 360

            // Notify the compass listener if needed
            if (Math.abs(mAzimuthDegrees - _lastAzimuthDegrees) >= _azimuthSensibility || _lastAzimuthDegrees == 0f) {
                _lastAzimuthDegrees = mAzimuthDegrees
                onNewAzimuth?.invoke(Math.round(mAzimuthDegrees))
            }
        }
    }


    override fun onAccuracyChanged(sensor: Sensor, accuracy: Int) {
    }

    private fun exponentialSmoothing(newValue: FloatArray, lastValue: FloatArray?, alpha: Float): FloatArray {
        val output = FloatArray(newValue.size)
        if (lastValue == null) {
            return newValue
        }
        for (i in newValue.indices) {
            output[i] = lastValue[i] + alpha * (newValue[i] - lastValue[i])
        }
        return output
    }

    companion object {
        private const val ROTATION_VECTOR_SMOOTHING_FACTOR = 1f
        private const val GEOMAGNETIC_SMOOTHING_FACTOR = 1f
        private const val GRAVITY_SMOOTHING_FACTOR = 0.3f

        fun newInstance(context: Context): Compass? {
            val compass = Compass(context)
            return if (compass.hasRequiredSensors()) {
                compass
            } else {
                null
            }
        }
    }
}