package io.boxo.utils

import android.Manifest
import android.app.Activity
import android.content.pm.PackageManager
import androidx.core.app.ActivityCompat
import androidx.core.content.PermissionChecker

class Permissions(private val activity: Activity) {
    private val PERMISSION_REQUEST_CAMERA = 1000
    private val PERMISSION_REQUEST_LOCATION = 1001

    private var onLocationAccess: (() -> Unit)? = null
    private var onLocationDenied: (() -> Unit)? = null
    private var onLocationForeverDenied: (() -> Unit)? = null

    private var onCameraAccess: (() -> Unit)? = null
    private var onCameraDenied: (() -> Unit)? = null
    private var onCameraForeverDenied: (() -> Unit)? = null

    fun hasLocationPermission(): Boolean =
        ActivityCompat.checkSelfPermission(
            activity,
            Manifest.permission.ACCESS_FINE_LOCATION
        ) == PermissionChecker.PERMISSION_GRANTED

    fun hasCameraPermission(): Boolean =
        ActivityCompat.checkSelfPermission(activity, Manifest.permission.CAMERA) == PermissionChecker.PERMISSION_GRANTED


    fun location(onAccess: () -> Unit, onDenied: () -> Unit, onForeverDenied: () -> Unit) {
        onLocationAccess = onAccess
        onLocationDenied = onDenied
        onLocationForeverDenied = onForeverDenied
        if (hasLocationPermission()) {
            onAccess()
        } else {
            ActivityCompat.requestPermissions(
                activity,
                arrayOf(Manifest.permission.ACCESS_FINE_LOCATION),
                PERMISSION_REQUEST_LOCATION
            )
        }
    }

    fun camera(activity: Activity, onAccess: () -> Unit, onDenied: () -> Unit, onForeverDenied: () -> Unit) {
        onCameraAccess = onAccess
        onCameraDenied = onDenied
        onCameraForeverDenied = onForeverDenied
        if (hasCameraPermission()) {
            onAccess()
        } else {
            ActivityCompat.requestPermissions(
                activity, arrayOf(Manifest.permission.CAMERA),
                PERMISSION_REQUEST_CAMERA
            )
        }
    }

    fun onRequestPermissionsResult(
        activity: Activity,
        requestCode: Int,
        permissions: Array<String>,
        grantResults: IntArray
    ) {
        when (requestCode) {
            PERMISSION_REQUEST_LOCATION -> {
                if ((grantResults.isNotEmpty() && grantResults[0] == PackageManager.PERMISSION_GRANTED)) {
                    onLocationAccess?.invoke()
                } else {
                    val showRationale =
                        ActivityCompat.shouldShowRequestPermissionRationale(
                            activity, Manifest.permission.ACCESS_FINE_LOCATION
                        )
                    if (!showRationale)
                        onLocationForeverDenied?.invoke()
                    else
                        onLocationDenied?.invoke()
                }
            }

            PERMISSION_REQUEST_CAMERA -> {
                if ((grantResults.isNotEmpty() && grantResults[0] == PackageManager.PERMISSION_GRANTED)) {
                    onCameraAccess?.invoke()
                } else {
                    val showRationale =
                        ActivityCompat.shouldShowRequestPermissionRationale(activity, Manifest.permission.CAMERA)
                    if (!showRationale)
                        onCameraForeverDenied?.invoke()
                    else
                        onCameraDenied?.invoke()
                }
            }
        }
    }
}