package io.boxo.utils.extensions

import android.app.Activity
import android.content.Intent
import android.content.res.Configuration
import android.graphics.Color
import android.net.Uri
import android.os.Build
import android.provider.Settings
import android.view.View
import android.view.WindowInsetsController
import android.view.WindowManager
import androidx.annotation.IdRes
import androidx.appcompat.app.AlertDialog
import androidx.appcompat.app.AppCompatActivity
import androidx.appcompat.app.AppCompatDelegate
import androidx.core.content.ContextCompat
import io.boxo.R
import io.boxo.sdk.Boxo
import io.boxo.sdk.Config
import io.boxo.ui.main.BoxoActivity
import java.util.Locale

@Suppress("DEPRECATION")
fun Activity.lightStatusBar() {
    when {
        Build.VERSION.SDK_INT >= Build.VERSION_CODES.R -> {
            window.insetsController?.setSystemBarsAppearance(
                WindowInsetsController.APPEARANCE_LIGHT_STATUS_BARS,
                WindowInsetsController.APPEARANCE_LIGHT_STATUS_BARS
            )
        }

        Build.VERSION.SDK_INT >= Build.VERSION_CODES.M -> {
            window.decorView.systemUiVisibility =
                View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN or View.SYSTEM_UI_FLAG_LIGHT_STATUS_BAR
        }

        else -> {
            window.decorView.systemUiVisibility = View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN
        }
    }
}

@Suppress("DEPRECATION")
fun Activity.darkStatusBar() {
    when {
        Build.VERSION.SDK_INT >= Build.VERSION_CODES.R -> {
            window.insetsController?.setSystemBarsAppearance(
                0,
                WindowInsetsController.APPEARANCE_LIGHT_STATUS_BARS
            )
        }

        Build.VERSION.SDK_INT >= Build.VERSION_CODES.M -> {
            window.decorView.systemUiVisibility = View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN or
                    window.decorView.systemUiVisibility and View.SYSTEM_UI_FLAG_LIGHT_STATUS_BAR.inv()
        }

        else -> {
            window.decorView.systemUiVisibility = View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN
        }
    }
}

@Suppress("DEPRECATION")
fun Activity.fullscreen() {
    window.apply {
        clearFlags(WindowManager.LayoutParams.FLAG_TRANSLUCENT_STATUS)
        addFlags(WindowManager.LayoutParams.FLAG_DRAWS_SYSTEM_BAR_BACKGROUNDS)
        decorView.systemUiVisibility = View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN
        statusBarColor = Color.TRANSPARENT
    }
}

fun Activity.setStatusBarColor(color: Int) {
    window.statusBarColor = color
}

fun AppCompatActivity.setDefaultTheme(boxo: Boxo) {
    val appId = intent.getStringExtra(BoxoActivity.APP_ID) ?: ""
    when (boxo.getExistingMiniapp(appId)?.theme) {
        Config.Theme.DARK -> delegate.localNightMode = AppCompatDelegate.MODE_NIGHT_YES
        Config.Theme.LIGHT -> delegate.localNightMode = AppCompatDelegate.MODE_NIGHT_NO
        else -> delegate.localNightMode = AppCompatDelegate.MODE_NIGHT_FOLLOW_SYSTEM
    }
}

internal fun Activity.setMiniappLocale() {
    val config = Configuration()
    val locale = Locale(Boxo.config.language.ifBlank { "en" })
    config.setLocale(locale)
    applyOverrideConfiguration(config)
}

fun AppCompatActivity.applyBoxoStatusBarByTheme(boxo: Boxo) {
    val appId = intent.getStringExtra(BoxoActivity.APP_ID) ?: return
    val miniapp = boxo.getExistingMiniapp(appId) ?: return

    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M && miniapp.theme == Config.Theme.LIGHT)
        window.decorView.systemUiVisibility = View.SYSTEM_UI_FLAG_LIGHT_STATUS_BAR
    else setStatusBarColor(ContextCompat.getColor(this, R.color.boxo_status_bar))
}

fun Activity.changeStatusTheme(color: Int) {
    val luminance =
        (0.299 * Color.red(color) + 0.587 * Color.green(color) + 0.114 * Color.blue(color)) / 255
    if (luminance < 0.5)
        darkStatusBar()
    else
        lightStatusBar()
}

fun Activity.openSettings(code: Int) {
    startActivityForResult(
        Intent(Settings.ACTION_APPLICATION_DETAILS_SETTINGS).setData(
            Uri.fromParts("package", packageName, null)
        ), code
    )
}

fun Activity.goHome() {
    startActivity(Intent(Intent.ACTION_MAIN).apply {
        addCategory(Intent.CATEGORY_HOME)
        flags = Intent.FLAG_ACTIVITY_NEW_TASK
    })
}

fun Activity.openLocationSettings(code: Int) {
    startActivityForResult(Intent(Settings.ACTION_LOCATION_SOURCE_SETTINGS), code)
}

fun Activity.showDialog(
    title: String? = null,
    message: String,
    negative: String? = null,
    positive: String? = null,
    onNegativeClick: (() -> Unit)? = null,
    onPositiveClick: (() -> Unit)? = null
) {
    AlertDialog.Builder(this).apply {
        if (title.isNullOrEmpty().not()) setTitle(title)
        setMessage(message)
        setNegativeButton(negative ?: getString(android.R.string.cancel)) { d, _ ->
            d.dismiss()
            onNegativeClick?.invoke()
        }
        setPositiveButton(positive ?: getString(android.R.string.ok)) { d, _ ->
            d.dismiss()
            onPositiveClick?.invoke()
        }
        setOnCancelListener {
            onNegativeClick?.invoke()
        }
    }.show()
}

fun <ViewT : View> Activity.bindView(@IdRes idRes: Int): Lazy<ViewT> {
    return lazy(LazyThreadSafetyMode.NONE) { findViewById<ViewT>(idRes) }
}