/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.client.impl.oauth;

import com.auth0.jwt.JWT;
import com.auth0.jwt.algorithms.Algorithm;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectReader;
import io.camunda.client.CredentialsProvider;
import io.camunda.client.impl.CamundaClientCredentials;
import io.camunda.client.impl.oauth.OAuthCredentialsCache;
import io.camunda.client.impl.oauth.OAuthCredentialsProviderBuilder;
import io.camunda.client.impl.util.VersionUtil;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.Reader;
import java.io.UncheckedIOException;
import java.io.UnsupportedEncodingException;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.security.GeneralSecurityException;
import java.security.KeyManagementException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.UnrecoverableKeyException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.security.interfaces.RSAPrivateKey;
import java.security.interfaces.RSAPublicKey;
import java.time.Duration;
import java.util.Base64;
import java.util.Date;
import java.util.HashMap;
import java.util.UUID;
import java.util.stream.Collectors;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLSocketFactory;
import net.jcip.annotations.ThreadSafe;
import org.apache.hc.core5.ssl.SSLContextBuilder;
import org.apache.hc.core5.ssl.SSLContexts;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ThreadSafe
public final class OAuthCredentialsProvider
implements CredentialsProvider {
    private static final String HEADER_AUTH_KEY = "Authorization";
    private static final String JWT_ASSERTION_TYPE = "urn:ietf:params:oauth:client-assertion-type:jwt-bearer";
    private static final ObjectMapper JSON_MAPPER = new ObjectMapper().configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
    private static final ObjectReader CREDENTIALS_READER = JSON_MAPPER.readerFor(CamundaClientCredentials.class);
    private static final Logger LOG = LoggerFactory.getLogger(OAuthCredentialsProvider.class);
    private final URL authorizationServerUrl;
    private final String payload;
    private final String clientId;
    private final Path keystorePath;
    private final String keystorePassword;
    private final String keystoreKeyPassword;
    private final Path truststorePath;
    private final String truststorePassword;
    private final OAuthCredentialsCache credentialsCache;
    private final Duration connectionTimeout;
    private final Duration readTimeout;

    OAuthCredentialsProvider(OAuthCredentialsProviderBuilder builder) {
        this.authorizationServerUrl = builder.getAuthorizationServer();
        this.keystorePath = builder.getKeystorePath();
        this.keystorePassword = builder.getKeystorePassword();
        this.keystoreKeyPassword = builder.getKeystoreKeyPassword();
        this.truststorePath = builder.getTruststorePath();
        this.truststorePassword = builder.getTruststorePassword();
        this.clientId = builder.getClientId();
        this.payload = OAuthCredentialsProvider.createParams(builder);
        this.credentialsCache = new OAuthCredentialsCache(builder.getCredentialsCache());
        this.connectionTimeout = builder.getConnectTimeout();
        this.readTimeout = builder.getReadTimeout();
    }

    @Override
    public void applyCredentials(CredentialsProvider.CredentialsApplier applier) throws IOException {
        CamundaClientCredentials camundaClientCredentials = this.credentialsCache.computeIfMissingOrInvalid(this.clientId, this::fetchCredentials);
        String type = camundaClientCredentials.getTokenType();
        if (type == null || type.isEmpty()) {
            throw new IOException(String.format("Expected valid token type but was absent or invalid '%s'", type));
        }
        type = Character.toUpperCase(type.charAt(0)) + type.substring(1);
        applier.put(HEADER_AUTH_KEY, String.format("%s %s", type, camundaClientCredentials.getAccessToken()));
    }

    @Override
    public boolean shouldRetryRequest(CredentialsProvider.StatusCode statusCode) {
        try {
            return statusCode.isUnauthorized() && this.credentialsCache.withCache(this.clientId, value -> {
                CamundaClientCredentials fetchedCredentials = this.fetchCredentials();
                this.credentialsCache.put(this.clientId, fetchedCredentials).writeCache();
                return !fetchedCredentials.equals(value) || !value.isValid();
            }).orElse(false) != false;
        }
        catch (IOException e) {
            LOG.error("Failed while fetching credentials: ", (Throwable)e);
            return false;
        }
    }

    private static String createParams(OAuthCredentialsProviderBuilder builder) {
        HashMap<String, String> payload = new HashMap<String, String>();
        if (builder.sslClientCertConfigurationProvided()) {
            payload.put("client_assertion", OAuthCredentialsProvider.getClientAssertion(builder.getSslClientCertPath().toAbsolutePath().toString(), builder.getSslClientCertPassword(), builder.getClientId(), builder.getAudience()));
            payload.put("client_assertion_type", JWT_ASSERTION_TYPE);
        } else {
            payload.put("client_secret", builder.getClientSecret());
        }
        payload.put("client_id", builder.getClientId());
        payload.put("audience", builder.getAudience());
        payload.put("grant_type", "client_credentials");
        String scope = builder.getScope();
        if (scope != null) {
            payload.put("scope", scope);
        }
        return payload.entrySet().stream().map(e -> OAuthCredentialsProvider.encode((String)e.getKey()) + "=" + OAuthCredentialsProvider.encode((String)e.getValue())).collect(Collectors.joining("&"));
    }

    private static String encode(String param) {
        try {
            return URLEncoder.encode(param, StandardCharsets.UTF_8.name());
        }
        catch (UnsupportedEncodingException e) {
            throw new UncheckedIOException("Failed while encoding OAuth request parameters: ", e);
        }
    }

    private CamundaClientCredentials fetchCredentials() throws IOException {
        HttpURLConnection connection = (HttpURLConnection)this.authorizationServerUrl.openConnection();
        this.maybeConfigureCustomSSLContext(connection);
        connection.setRequestMethod("POST");
        connection.setRequestProperty("Content-Type", "application/x-www-form-urlencoded");
        connection.setRequestProperty("Accept", "application/json");
        connection.setDoOutput(true);
        connection.setReadTimeout(Math.toIntExact(this.readTimeout.toMillis()));
        connection.setConnectTimeout(Math.toIntExact(this.connectionTimeout.toMillis()));
        connection.setRequestProperty("User-Agent", "camunda-client-java/" + VersionUtil.getVersion());
        try (OutputStream os = connection.getOutputStream();){
            byte[] input = this.payload.getBytes(StandardCharsets.UTF_8);
            os.write(input, 0, input.length);
        }
        if (connection.getResponseCode() != 200) {
            throw new IOException(String.format("Failed while requesting access token with status code %d and message %s.", connection.getResponseCode(), connection.getResponseMessage()));
        }
        try (InputStream in = connection.getInputStream();){
            CamundaClientCredentials camundaClientCredentials;
            try (InputStreamReader reader = new InputStreamReader(in, StandardCharsets.UTF_8);){
                CamundaClientCredentials fetchedCredentials = (CamundaClientCredentials)CREDENTIALS_READER.readValue((Reader)reader);
                if (fetchedCredentials == null) {
                    throw new IOException("Expected valid credentials but got null instead.");
                }
                camundaClientCredentials = fetchedCredentials;
            }
            return camundaClientCredentials;
        }
    }

    private void maybeConfigureCustomSSLContext(HttpURLConnection connection) {
        if (connection instanceof HttpsURLConnection) {
            HttpsURLConnection httpsConnection = (HttpsURLConnection)connection;
            httpsConnection.setSSLSocketFactory(this.createSSLContext());
        }
    }

    private SSLSocketFactory createSSLContext() {
        if (this.keystorePath == null && this.truststorePath == null) {
            return SSLContexts.createSystemDefault().getSocketFactory();
        }
        SSLContextBuilder builder = SSLContexts.custom();
        try {
            if (this.keystorePath != null) {
                builder.loadKeyMaterial(this.keystorePath, this.keystorePassword == null ? null : this.keystorePassword.toCharArray(), this.keystoreKeyPassword == null ? new char[]{} : this.keystoreKeyPassword.toCharArray(), new OpenOption[0]);
            }
            if (this.truststorePath != null) {
                builder.loadTrustMaterial(this.truststorePath, this.truststorePassword == null ? null : this.truststorePassword.toCharArray());
            }
            return builder.build().getSocketFactory();
        }
        catch (IOException | KeyManagementException | KeyStoreException | NoSuchAlgorithmException | UnrecoverableKeyException | CertificateException e) {
            throw new RuntimeException("Failed to create SSL context", e);
        }
    }

    private static String getClientAssertion(String certPath, String certStorePassword, String clientId, String audience) {
        Algorithm algorithm;
        X509Certificate certificate;
        try (FileInputStream stream = new FileInputStream(certPath);){
            KeyStore keyStore = KeyStore.getInstance("PKCS12");
            char[] password = certStorePassword.toCharArray();
            keyStore.load(stream, password);
            String alias = keyStore.aliases().nextElement();
            RSAPrivateKey privateKey = (RSAPrivateKey)keyStore.getKey(alias, password);
            X509Certificate keyStoreCertificate = (X509Certificate)keyStore.getCertificate(alias);
            RSAPublicKey publicKey = (RSAPublicKey)keyStoreCertificate.getPublicKey();
            certificate = (X509Certificate)keyStore.getCertificate(alias);
            algorithm = Algorithm.RSA256((RSAPublicKey)publicKey, (RSAPrivateKey)privateKey);
        }
        catch (IOException | GeneralSecurityException e) {
            throw new RuntimeException("Failed to create client assertion", e);
        }
        Date now = new Date();
        String x5t = OAuthCredentialsProvider.generateX5tThumbprint(certificate);
        HashMap<String, String> header = new HashMap<String, String>();
        header.put("alg", "RSA256");
        header.put("typ", "JWT");
        header.put("x5t", x5t);
        return JWT.create().withHeader(header).withIssuer(clientId).withSubject(clientId).withAudience(new String[]{audience}).withIssuedAt(now).withNotBefore(now).withExpiresAt(new Date(now.getTime() + 300000L)).withJWTId(UUID.randomUUID().toString()).sign(algorithm);
    }

    private static String generateX5tThumbprint(X509Certificate certificate) {
        try {
            MessageDigest digest = MessageDigest.getInstance("SHA-1");
            byte[] encoded = digest.digest(certificate.getEncoded());
            return Base64.getUrlEncoder().withoutPadding().encodeToString(encoded);
        }
        catch (Exception e) {
            throw new RuntimeException("Failed to generate x5t thumbprint", e);
        }
    }
}

