/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.exporter;

import com.fasterxml.jackson.databind.ObjectMapper;
import io.camunda.zeebe.exporter.ElasticsearchExporter;
import io.camunda.zeebe.exporter.ElasticsearchExporterConfiguration;
import io.camunda.zeebe.exporter.ElasticsearchExporterException;
import io.camunda.zeebe.exporter.ElasticsearchMetrics;
import io.camunda.zeebe.exporter.dto.BulkItemError;
import io.camunda.zeebe.exporter.dto.BulkResponse;
import io.camunda.zeebe.exporter.dto.PutIndexTemplateResponse;
import io.camunda.zeebe.protocol.record.Record;
import io.camunda.zeebe.protocol.record.ValueType;
import io.camunda.zeebe.util.VersionUtil;
import io.prometheus.client.Histogram;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.time.Instant;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import org.apache.http.HttpHost;
import org.apache.http.auth.AuthScope;
import org.apache.http.auth.Credentials;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.client.CredentialsProvider;
import org.apache.http.impl.client.BasicCredentialsProvider;
import org.apache.http.impl.nio.client.HttpAsyncClientBuilder;
import org.apache.http.impl.nio.reactor.IOReactorConfig;
import org.elasticsearch.client.Request;
import org.elasticsearch.client.Response;
import org.elasticsearch.client.ResponseException;
import org.elasticsearch.client.RestClient;
import org.elasticsearch.client.RestClientBuilder;
import org.elasticsearch.xcontent.DeprecationHandler;
import org.elasticsearch.xcontent.NamedXContentRegistry;
import org.elasticsearch.xcontent.XContent;
import org.elasticsearch.xcontent.XContentParser;
import org.elasticsearch.xcontent.XContentType;
import org.slf4j.Logger;

public class ElasticsearchClient {
    public static final String INDEX_TEMPLATE_FILENAME_PATTERN = "/zeebe-record-%s-template.json";
    public static final String INDEX_DELIMITER = "_";
    public static final String ALIAS_DELIMITER = "-";
    private static final ObjectMapper MAPPER = new ObjectMapper();
    protected final RestClient client;
    private final ElasticsearchExporterConfiguration configuration;
    private final Logger log;
    private final DateTimeFormatter formatter;
    private List<String> bulkRequest;
    private ElasticsearchMetrics metrics;

    public ElasticsearchClient(ElasticsearchExporterConfiguration configuration, Logger log) {
        this(configuration, log, new ArrayList<String>());
    }

    ElasticsearchClient(ElasticsearchExporterConfiguration configuration, Logger log, List<String> bulkRequest) {
        this.configuration = configuration;
        this.log = log;
        this.client = this.createClient();
        this.bulkRequest = bulkRequest;
        this.formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd").withZone(ZoneOffset.UTC);
    }

    public void close() throws IOException {
        this.client.close();
    }

    public void index(Record<?> record) {
        if (this.metrics == null) {
            this.metrics = new ElasticsearchMetrics(record.getPartitionId());
        }
        this.bulk(this.newIndexCommand(record), record);
    }

    public void bulk(Map<String, Object> command, Record<?> record) {
        String serializedCommand;
        try {
            serializedCommand = MAPPER.writeValueAsString(command);
        }
        catch (IOException e) {
            throw new ElasticsearchExporterException("Failed to serialize bulk request command to JSON", e);
        }
        String jsonCommand = serializedCommand + "\n" + record.toJson();
        if (this.bulkRequest.isEmpty() || !this.bulkRequest.get(this.bulkRequest.size() - 1).equals(jsonCommand)) {
            this.bulkRequest.add(jsonCommand);
        }
    }

    public void flush() {
        if (this.bulkRequest.isEmpty()) {
            return;
        }
        int bulkSize = this.bulkRequest.size();
        this.metrics.recordBulkSize(bulkSize);
        int bulkMemorySize = this.getBulkMemorySize();
        this.metrics.recordBulkMemorySize(bulkMemorySize);
        try (Histogram.Timer timer = this.metrics.measureFlushDuration();){
            this.exportBulk();
            this.bulkRequest = new ArrayList<String>();
        }
        catch (ElasticsearchExporterException e) {
            this.metrics.recordFailedFlush();
            throw e;
        }
    }

    private void exportBulk() {
        BulkResponse bulkResponse;
        Response httpResponse;
        try {
            httpResponse = this.sendBulkRequest();
        }
        catch (ResponseException e) {
            throw new ElasticsearchExporterException("Elastic returned an error response on flush", e);
        }
        catch (IOException e) {
            throw new ElasticsearchExporterException("Failed to flush bulk", e);
        }
        try {
            bulkResponse = (BulkResponse)MAPPER.readValue(httpResponse.getEntity().getContent(), BulkResponse.class);
        }
        catch (IOException e) {
            throw new ElasticsearchExporterException("Failed to parse response when flushing", e);
        }
        if (bulkResponse.hasErrors()) {
            this.throwCollectedBulkError(bulkResponse);
        }
    }

    private void throwCollectedBulkError(BulkResponse bulkResponse) {
        ArrayList collectedErrors = new ArrayList();
        bulkResponse.getItems().stream().flatMap(item -> Optional.ofNullable(item.getIndex()).stream()).flatMap(index -> Optional.ofNullable(index.getError()).stream()).collect(Collectors.groupingBy(BulkItemError::getType)).forEach((errorType, errors) -> collectedErrors.add(String.format("Failed to flush %d item(s) of bulk request [type: %s, reason: %s]", errors.size(), errorType, ((BulkItemError)errors.get(0)).getReason())));
        throw new ElasticsearchExporterException("Failed to flush bulk request: " + collectedErrors);
    }

    private Response sendBulkRequest() throws IOException {
        Request request = new Request("POST", "/_bulk");
        request.setJsonEntity(String.join((CharSequence)"\n", this.bulkRequest) + "\n");
        return this.client.performRequest(request);
    }

    public boolean shouldFlush() {
        return this.bulkRequest.size() >= this.configuration.bulk.size || this.getBulkMemorySize() >= this.configuration.bulk.memoryLimit;
    }

    private int getBulkMemorySize() {
        return this.bulkRequest.stream().mapToInt(String::length).sum();
    }

    public boolean putIndexTemplate(ValueType valueType) {
        String templateName = this.indexPrefixForValueType(valueType);
        String aliasName = this.aliasNameForValueType(valueType);
        String filename = ElasticsearchClient.indexTemplateForValueType(valueType);
        return this.putIndexTemplate(templateName, aliasName, filename);
    }

    public boolean putIndexTemplate(String templateName, String aliasName, String filename) {
        Map<String, Object> template = this.getTemplateFromClasspath(filename);
        template.put("index_patterns", Collections.singletonList(templateName + "_*"));
        template.put("composed_of", Collections.singletonList(this.configuration.index.prefix));
        Object templateProperties = template.computeIfAbsent("template", key -> new HashMap());
        if (!(templateProperties instanceof Map)) {
            throw new IllegalStateException(String.format("Expected the 'template' field of the index template '%s' to be a map, but was '%s'", templateName, templateProperties.getClass()));
        }
        Map templatePropertyMap = (Map)templateProperties;
        templatePropertyMap.put("aliases", Collections.singletonMap(aliasName, Collections.emptyMap()));
        return this.putIndexTemplate(templateName, template);
    }

    public boolean putComponentTemplate(String templateName, String aliasName, String filename) {
        Map<String, Object> template = this.getTemplateFromClasspath(filename);
        return this.putComponentTemplate(templateName, template);
    }

    private Map<String, Object> getTemplateFromClasspath(String filename) {
        Map<String, Object> template;
        block9: {
            try (InputStream inputStream = ElasticsearchExporter.class.getResourceAsStream(filename);){
                if (inputStream != null) {
                    template = this.convertToMap(XContentType.JSON.xContent(), inputStream);
                    break block9;
                }
                throw new ElasticsearchExporterException("Failed to find index template in classpath " + filename);
            }
            catch (IOException e) {
                throw new ElasticsearchExporterException("Failed to load index template from classpath " + filename, e);
            }
        }
        return template;
    }

    private boolean putIndexTemplate(String templateName, Object body) {
        try {
            Request request = new Request("PUT", "/_index_template/" + templateName);
            request.setJsonEntity(MAPPER.writeValueAsString(body));
            Response response = this.client.performRequest(request);
            PutIndexTemplateResponse putIndexTemplateResponse = (PutIndexTemplateResponse)MAPPER.readValue(response.getEntity().getContent(), PutIndexTemplateResponse.class);
            return putIndexTemplateResponse.isAcknowledged();
        }
        catch (IOException e) {
            throw new ElasticsearchExporterException("Failed to put index template", e);
        }
    }

    private boolean putComponentTemplate(String templateName, Object body) {
        try {
            Request request = new Request("PUT", "/_component_template/" + templateName);
            request.setJsonEntity(MAPPER.writeValueAsString(body));
            Response response = this.client.performRequest(request);
            PutIndexTemplateResponse putIndexTemplateResponse = (PutIndexTemplateResponse)MAPPER.readValue(response.getEntity().getContent(), PutIndexTemplateResponse.class);
            return putIndexTemplateResponse.isAcknowledged();
        }
        catch (IOException e) {
            throw new ElasticsearchExporterException("Failed to put component template", e);
        }
    }

    private RestClient createClient() {
        HttpHost[] httpHosts = ElasticsearchClient.urlsToHttpHosts(this.configuration.url);
        RestClientBuilder builder = RestClient.builder((HttpHost[])httpHosts).setRequestConfigCallback(b -> b.setConnectTimeout(this.configuration.requestTimeoutMs).setSocketTimeout(this.configuration.requestTimeoutMs)).setHttpClientConfigCallback(this::setHttpClientConfigCallback);
        return builder.build();
    }

    private HttpAsyncClientBuilder setHttpClientConfigCallback(HttpAsyncClientBuilder builder) {
        builder.setDefaultIOReactorConfig(IOReactorConfig.custom().setIoThreadCount(1).build());
        if (this.configuration.hasAuthenticationPresent()) {
            this.setupBasicAuthentication(builder);
        }
        return builder;
    }

    private void setupBasicAuthentication(HttpAsyncClientBuilder builder) {
        BasicCredentialsProvider credentialsProvider = new BasicCredentialsProvider();
        credentialsProvider.setCredentials(AuthScope.ANY, (Credentials)new UsernamePasswordCredentials(this.configuration.getAuthentication().getUsername(), this.configuration.getAuthentication().getPassword()));
        builder.setDefaultCredentialsProvider((CredentialsProvider)credentialsProvider);
    }

    private static HttpHost[] urlsToHttpHosts(String urls) {
        return (HttpHost[])Arrays.stream(urls.split(",")).map(String::trim).map(ElasticsearchClient::urlToHttpHost).toArray(HttpHost[]::new);
    }

    private static HttpHost urlToHttpHost(String url) {
        URI uri;
        try {
            uri = new URI(url);
        }
        catch (URISyntaxException e) {
            throw new ElasticsearchExporterException("Failed to parse url " + url, e);
        }
        return new HttpHost(uri.getHost(), uri.getPort(), uri.getScheme());
    }

    protected String indexFor(Record<?> record) {
        Instant timestamp = Instant.ofEpochMilli(record.getTimestamp());
        return this.indexPrefixForValueTypeWithDelimiter(record.getValueType()) + this.formatter.format(timestamp);
    }

    protected String idFor(Record<?> record) {
        return record.getPartitionId() + ALIAS_DELIMITER + record.getPosition();
    }

    protected String typeFor(Record<?> record) {
        return "_doc";
    }

    protected String indexPrefixForValueTypeWithDelimiter(ValueType valueType) {
        return this.indexPrefixForValueType(valueType) + INDEX_DELIMITER;
    }

    private String aliasNameForValueType(ValueType valueType) {
        return this.configuration.index.prefix + ALIAS_DELIMITER + ElasticsearchClient.valueTypeToString(valueType);
    }

    private String indexPrefixForValueType(ValueType valueType) {
        String version = VersionUtil.getVersionLowerCase();
        return this.configuration.index.prefix + INDEX_DELIMITER + ElasticsearchClient.valueTypeToString(valueType) + INDEX_DELIMITER + version;
    }

    private static String valueTypeToString(ValueType valueType) {
        return valueType.name().toLowerCase().replaceAll(INDEX_DELIMITER, ALIAS_DELIMITER);
    }

    private static String indexTemplateForValueType(ValueType valueType) {
        return String.format(INDEX_TEMPLATE_FILENAME_PATTERN, ElasticsearchClient.valueTypeToString(valueType));
    }

    private Map<String, Object> convertToMap(XContent content, InputStream input) {
        Map map;
        block8: {
            XContentParser parser = content.createParser(NamedXContentRegistry.EMPTY, DeprecationHandler.THROW_UNSUPPORTED_OPERATION, input);
            try {
                map = parser.mapOrdered();
                if (parser == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (parser != null) {
                        try {
                            parser.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    throw new ElasticsearchExporterException("Failed to parse content to map", e);
                }
            }
            parser.close();
        }
        return map;
    }

    private Map<String, Object> newIndexCommand(Record<?> record) {
        HashMap<String, Object> command = new HashMap<String, Object>();
        HashMap<String, String> contents = new HashMap<String, String>();
        contents.put("_index", this.indexFor(record));
        contents.put("_id", this.idFor(record));
        contents.put("routing", String.valueOf(record.getPartitionId()));
        command.put("index", contents);
        return command;
    }
}

