/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.util.health;

import io.camunda.zeebe.util.health.FailureListener;
import io.camunda.zeebe.util.health.HealthMonitor;
import io.camunda.zeebe.util.health.HealthMonitorable;
import io.camunda.zeebe.util.health.HealthStatus;
import io.camunda.zeebe.util.sched.ActorControl;
import java.time.Duration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import org.slf4j.Logger;

public class CriticalComponentsHealthMonitor
implements HealthMonitor {
    private static final Duration HEALTH_MONITORING_PERIOD = Duration.ofSeconds(60L);
    private final Map<String, MonitoredComponent> monitoredComponents = new HashMap<String, MonitoredComponent>();
    private final Map<String, HealthStatus> componentHealth = new HashMap<String, HealthStatus>();
    private final Set<FailureListener> failureListeners = new HashSet<FailureListener>();
    private final ActorControl actor;
    private final Logger log;
    private volatile HealthStatus healthStatus = HealthStatus.UNHEALTHY;

    public CriticalComponentsHealthMonitor(ActorControl actor, Logger log) {
        this.actor = actor;
        this.log = log;
    }

    @Override
    public void startMonitoring() {
        this.actor.runAtFixedRate(HEALTH_MONITORING_PERIOD, this::updateHealth);
    }

    @Override
    public void monitorComponent(String componentName) {
        this.actor.run(() -> this.componentHealth.put(componentName, HealthStatus.UNHEALTHY));
    }

    @Override
    public void removeComponent(String componentName) {
        this.actor.run(() -> {
            MonitoredComponent monitoredComponent = this.monitoredComponents.remove(componentName);
            if (monitoredComponent != null) {
                this.componentHealth.remove(componentName);
                monitoredComponent.component.removeFailureListener(monitoredComponent);
            }
        });
    }

    @Override
    public void registerComponent(String componentName, HealthMonitorable component) {
        this.actor.run(() -> {
            MonitoredComponent monitoredComponent = new MonitoredComponent(componentName, component);
            this.monitoredComponents.put(componentName, monitoredComponent);
            this.componentHealth.put(componentName, component.getHealthStatus());
            component.addFailureListener(monitoredComponent);
            this.calculateHealth();
        });
    }

    @Override
    public HealthStatus getHealthStatus() {
        return this.healthStatus;
    }

    @Override
    public void addFailureListener(FailureListener failureListener) {
        this.actor.run(() -> this.failureListeners.add(failureListener));
    }

    @Override
    public void removeFailureListener(FailureListener failureListener) {
        this.actor.run(() -> this.failureListeners.remove(failureListener));
    }

    private void updateHealth() {
        this.componentHealth.keySet().forEach(component -> this.componentHealth.put((String)component, this.getHealth((String)component)));
        this.calculateHealth();
    }

    private void calculateHealth() {
        HealthStatus previousStatus = this.healthStatus;
        this.healthStatus = this.calculateStatus();
        if (previousStatus == this.healthStatus) {
            return;
        }
        switch (this.healthStatus) {
            case HEALTHY: {
                this.failureListeners.forEach(FailureListener::onRecovered);
                break;
            }
            case UNHEALTHY: {
                this.failureListeners.forEach(FailureListener::onFailure);
                break;
            }
            case DEAD: {
                this.failureListeners.forEach(FailureListener::onUnrecoverableFailure);
                break;
            }
            default: {
                this.log.warn("Unknown health status {}", (Object)this.healthStatus);
            }
        }
        this.logComponentStatus(this.healthStatus);
    }

    private void logComponentStatus(HealthStatus status) {
        this.log.debug("Detected '{}' components. The current health status of components: {}", (Object)status, this.componentHealth);
    }

    private HealthStatus calculateStatus() {
        if (this.componentHealth.containsValue((Object)HealthStatus.DEAD)) {
            return HealthStatus.DEAD;
        }
        if (this.componentHealth.containsValue((Object)HealthStatus.UNHEALTHY)) {
            return HealthStatus.UNHEALTHY;
        }
        return HealthStatus.HEALTHY;
    }

    private HealthStatus getHealth(String componentName) {
        MonitoredComponent monitoredComponent = this.monitoredComponents.get(componentName);
        if (monitoredComponent != null) {
            return monitoredComponent.component.getHealthStatus();
        }
        return HealthStatus.UNHEALTHY;
    }

    private final class MonitoredComponent
    implements FailureListener {
        private final String componentName;
        private final HealthMonitorable component;

        private MonitoredComponent(String componentName, HealthMonitorable component) {
            this.componentName = componentName;
            this.component = component;
        }

        @Override
        public void onFailure() {
            CriticalComponentsHealthMonitor.this.actor.run(this::onComponentFailure);
        }

        @Override
        public void onRecovered() {
            CriticalComponentsHealthMonitor.this.actor.run(this::onComponentRecovered);
        }

        @Override
        public void onUnrecoverableFailure() {
            CriticalComponentsHealthMonitor.this.actor.run(this::onComponentDied);
        }

        private void onComponentFailure() {
            if (!CriticalComponentsHealthMonitor.this.monitoredComponents.containsKey(this.componentName)) {
                return;
            }
            CriticalComponentsHealthMonitor.this.log.warn("{} failed, marking it as unhealthy", (Object)this.componentName);
            CriticalComponentsHealthMonitor.this.componentHealth.put(this.componentName, HealthStatus.UNHEALTHY);
            CriticalComponentsHealthMonitor.this.calculateHealth();
        }

        private void onComponentRecovered() {
            if (!CriticalComponentsHealthMonitor.this.monitoredComponents.containsKey(this.componentName)) {
                return;
            }
            CriticalComponentsHealthMonitor.this.log.info("{} recovered, marking it as healthy", (Object)this.componentName);
            CriticalComponentsHealthMonitor.this.componentHealth.put(this.componentName, HealthStatus.HEALTHY);
            CriticalComponentsHealthMonitor.this.calculateHealth();
        }

        private void onComponentDied() {
            if (!CriticalComponentsHealthMonitor.this.monitoredComponents.containsKey(this.componentName)) {
                return;
            }
            CriticalComponentsHealthMonitor.this.log.error("{} failed, marking it as dead", (Object)this.componentName);
            CriticalComponentsHealthMonitor.this.componentHealth.put(this.componentName, HealthStatus.DEAD);
            CriticalComponentsHealthMonitor.this.calculateHealth();
        }
    }
}

