/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.broker.exporter.metrics;

import io.camunda.zeebe.broker.exporter.metrics.ExecutionLatencyMetrics;
import io.camunda.zeebe.broker.system.configuration.ExporterCfg;
import io.camunda.zeebe.exporter.api.Exporter;
import io.camunda.zeebe.exporter.api.context.Controller;
import io.camunda.zeebe.protocol.record.Record;
import io.camunda.zeebe.protocol.record.RecordType;
import io.camunda.zeebe.protocol.record.ValueType;
import io.camunda.zeebe.protocol.record.intent.Intent;
import io.camunda.zeebe.protocol.record.intent.JobBatchIntent;
import io.camunda.zeebe.protocol.record.intent.JobIntent;
import io.camunda.zeebe.protocol.record.intent.ProcessInstanceIntent;
import io.camunda.zeebe.protocol.record.value.BpmnElementType;
import io.camunda.zeebe.protocol.record.value.JobBatchRecordValue;
import io.camunda.zeebe.protocol.record.value.ProcessInstanceRecordValue;
import java.time.Duration;
import java.util.Iterator;
import java.util.NavigableMap;
import java.util.SortedMap;
import java.util.TreeMap;
import org.agrona.collections.Long2LongHashMap;

public class MetricsExporter
implements Exporter {
    public static final Duration TIME_TO_LIVE = Duration.ofSeconds(10L);
    private final ExecutionLatencyMetrics executionLatencyMetrics = new ExecutionLatencyMetrics();
    private final Long2LongHashMap jobKeyToCreationTimeMap = new Long2LongHashMap(-1L);
    private final Long2LongHashMap processInstanceKeyToCreationTimeMap = new Long2LongHashMap(-1L);
    private final NavigableMap<Long, Long> creationTimeToJobKeyNavigableMap = new TreeMap<Long, Long>();
    private final NavigableMap<Long, Long> creationTimeToProcessInstanceKeyNavigableMap = new TreeMap<Long, Long>();
    private Controller controller;

    public void open(Controller controller) {
        this.controller = controller;
        controller.scheduleCancellableTask(TIME_TO_LIVE, this::cleanUp);
    }

    public void close() {
        this.jobKeyToCreationTimeMap.clear();
        this.processInstanceKeyToCreationTimeMap.clear();
        this.creationTimeToJobKeyNavigableMap.clear();
        this.creationTimeToProcessInstanceKeyNavigableMap.clear();
    }

    public void export(Record<?> record) {
        if (record.getRecordType() != RecordType.EVENT) {
            this.controller.updateLastExportedRecordPosition(record.getPosition());
            return;
        }
        int partitionId = record.getPartitionId();
        long recordKey = record.getKey();
        ValueType currentValueType = record.getValueType();
        if (currentValueType == ValueType.JOB) {
            this.handleJobRecord(record, partitionId, recordKey);
        } else if (currentValueType == ValueType.JOB_BATCH) {
            this.handleJobBatchRecord(record, partitionId);
        } else if (currentValueType == ValueType.PROCESS_INSTANCE) {
            this.handleProcessInstanceRecord(record, partitionId, recordKey);
        }
        this.controller.updateLastExportedRecordPosition(record.getPosition());
    }

    private void handleProcessInstanceRecord(Record<?> record, int partitionId, long recordKey) {
        Intent currentIntent = record.getIntent();
        if (currentIntent == ProcessInstanceIntent.ELEMENT_ACTIVATING && MetricsExporter.isProcessInstanceRecord(record)) {
            this.storeProcessInstanceCreation(record.getTimestamp(), recordKey);
        } else if (currentIntent == ProcessInstanceIntent.ELEMENT_COMPLETED && MetricsExporter.isProcessInstanceRecord(record)) {
            long creationTime = this.processInstanceKeyToCreationTimeMap.remove(recordKey);
            this.executionLatencyMetrics.observeProcessInstanceExecutionTime(partitionId, creationTime, record.getTimestamp());
        }
    }

    private void storeProcessInstanceCreation(long creationTime, long recordKey) {
        this.processInstanceKeyToCreationTimeMap.put(recordKey, creationTime);
        this.creationTimeToProcessInstanceKeyNavigableMap.put(creationTime, recordKey);
    }

    private void handleJobRecord(Record<?> record, int partitionId, long recordKey) {
        Intent currentIntent = record.getIntent();
        if (currentIntent == JobIntent.CREATED) {
            this.storeJobCreation(record.getTimestamp(), recordKey);
        } else if (currentIntent == JobIntent.COMPLETED) {
            long creationTime = this.jobKeyToCreationTimeMap.remove(recordKey);
            this.executionLatencyMetrics.observeJobLifeTime(partitionId, creationTime, record.getTimestamp());
        }
    }

    private void handleJobBatchRecord(Record<?> record, int partitionId) {
        Intent currentIntent = record.getIntent();
        if (currentIntent == JobBatchIntent.ACTIVATED) {
            JobBatchRecordValue value = (JobBatchRecordValue)record.getValue();
            Iterator iterator = value.getJobKeys().iterator();
            while (iterator.hasNext()) {
                long jobKey = (Long)iterator.next();
                long creationTime = this.jobKeyToCreationTimeMap.get(jobKey);
                this.executionLatencyMetrics.observeJobActivationTime(partitionId, creationTime, record.getTimestamp());
            }
        }
    }

    private void storeJobCreation(long creationTime, long recordKey) {
        this.jobKeyToCreationTimeMap.put(recordKey, creationTime);
        this.creationTimeToJobKeyNavigableMap.put(creationTime, recordKey);
    }

    private void cleanUp() {
        long currentTimeMillis = System.currentTimeMillis();
        long deadTime = currentTimeMillis - TIME_TO_LIVE.toMillis();
        this.clearMaps(deadTime, this.creationTimeToJobKeyNavigableMap, this.jobKeyToCreationTimeMap);
        this.clearMaps(deadTime, this.creationTimeToProcessInstanceKeyNavigableMap, this.processInstanceKeyToCreationTimeMap);
        this.controller.scheduleCancellableTask(TIME_TO_LIVE, this::cleanUp);
    }

    private void clearMaps(long deadTime, NavigableMap<Long, Long> timeToKeyMap, Long2LongHashMap keyToTimestampMap) {
        SortedMap<Long, Long> outOfScopeInstances = timeToKeyMap.headMap(deadTime);
        for (Long key : outOfScopeInstances.values()) {
            keyToTimestampMap.remove((Object)key);
        }
        outOfScopeInstances.clear();
    }

    public static ExporterCfg defaultConfig() {
        ExporterCfg exporterCfg = new ExporterCfg();
        exporterCfg.setClassName(MetricsExporter.class.getName());
        return exporterCfg;
    }

    public static String defaultExporterId() {
        return MetricsExporter.class.getSimpleName();
    }

    private static boolean isProcessInstanceRecord(Record<?> record) {
        ProcessInstanceRecordValue recordValue = (ProcessInstanceRecordValue)record.getValue();
        return BpmnElementType.PROCESS == recordValue.getBpmnElementType();
    }
}

