/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.journal.file;

import com.google.common.base.Preconditions;
import com.google.common.collect.Sets;
import io.camunda.zeebe.journal.Journal;
import io.camunda.zeebe.journal.JournalException;
import io.camunda.zeebe.journal.JournalReader;
import io.camunda.zeebe.journal.JournalRecord;
import io.camunda.zeebe.journal.file.JournalIndex;
import io.camunda.zeebe.journal.file.JournalMetrics;
import io.camunda.zeebe.journal.file.JournalSegment;
import io.camunda.zeebe.journal.file.JournalSegmentDescriptor;
import io.camunda.zeebe.journal.file.JournalSegmentFile;
import io.camunda.zeebe.journal.file.SegmentedJournalBuilder;
import io.camunda.zeebe.journal.file.SegmentedJournalReader;
import io.camunda.zeebe.journal.file.SegmentedJournalWriter;
import io.camunda.zeebe.journal.file.UnknownVersionException;
import io.camunda.zeebe.journal.file.record.CorruptedLogException;
import io.camunda.zeebe.util.FileUtil;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.DirectoryStream;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.Set;
import java.util.SortedMap;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.locks.StampedLock;
import org.agrona.DirectBuffer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class SegmentedJournal
implements Journal {
    public static final long ASQN_IGNORE = -1L;
    private static final ByteOrder ENDIANNESS = ByteOrder.LITTLE_ENDIAN;
    private static final int SEGMENT_BUFFER_FACTOR = 3;
    private static final int FIRST_SEGMENT_ID = 1;
    private static final int INITIAL_INDEX = 1;
    private final JournalMetrics journalMetrics;
    private final Logger log = LoggerFactory.getLogger(this.getClass());
    private final String name;
    private final File directory;
    private final int maxSegmentSize;
    private final NavigableMap<Long, JournalSegment> segments = new ConcurrentSkipListMap<Long, JournalSegment>();
    private final Collection<SegmentedJournalReader> readers = Sets.newConcurrentHashSet();
    private volatile JournalSegment currentSegment;
    private volatile boolean open = true;
    private final long minFreeDiskSpace;
    private final JournalIndex journalIndex;
    private final SegmentedJournalWriter writer;
    private final long lastWrittenIndex;
    private final StampedLock rwlock = new StampedLock();

    public SegmentedJournal(String name, File directory, int maxSegmentSize, long minFreeSpace, JournalIndex journalIndex, long lastWrittenIndex) {
        this.name = (String)Preconditions.checkNotNull((Object)name, (Object)"name cannot be null");
        this.directory = (File)Preconditions.checkNotNull((Object)directory, (Object)"directory cannot be null");
        this.maxSegmentSize = maxSegmentSize;
        this.journalMetrics = new JournalMetrics(name);
        this.minFreeDiskSpace = minFreeSpace;
        this.journalIndex = journalIndex;
        this.lastWrittenIndex = lastWrittenIndex;
        this.open();
        this.writer = new SegmentedJournalWriter(this);
    }

    public static SegmentedJournalBuilder builder() {
        return new SegmentedJournalBuilder();
    }

    @Override
    public JournalRecord append(long asqn, DirectBuffer data) {
        return this.writer.append(asqn, data);
    }

    @Override
    public JournalRecord append(DirectBuffer data) {
        return this.writer.append(-1L, data);
    }

    @Override
    public void append(JournalRecord record) {
        this.writer.append(record);
    }

    @Override
    public void deleteAfter(long indexExclusive) {
        this.journalMetrics.observeSegmentTruncation(() -> {
            long stamp = this.rwlock.writeLock();
            try {
                this.writer.deleteAfter(indexExclusive);
                this.resetAdvancedReaders(indexExclusive + 1L);
            }
            finally {
                this.rwlock.unlockWrite(stamp);
            }
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void deleteUntil(long index) {
        Map.Entry<Long, JournalSegment> segmentEntry = this.segments.floorEntry(index);
        if (segmentEntry != null) {
            SortedMap<Long, JournalSegment> compactSegments = this.segments.headMap(segmentEntry.getValue().index());
            if (compactSegments.isEmpty()) {
                this.log.debug("No segments can be deleted with index < {} (first log index: {})", (Object)index, (Object)this.getFirstIndex());
                return;
            }
            long stamp = this.rwlock.writeLock();
            try {
                this.log.debug("{} - Deleting log up from {} up to {} (removing {} segments)", new Object[]{this.name, this.getFirstIndex(), ((JournalSegment)compactSegments.get(compactSegments.lastKey())).index(), compactSegments.size()});
                for (JournalSegment segment : compactSegments.values()) {
                    this.log.trace("{} - Deleting segment: {}", (Object)this.name, (Object)segment);
                    segment.delete();
                    this.journalMetrics.decSegmentCount();
                }
                compactSegments.clear();
                this.journalIndex.deleteUntil(index);
            }
            finally {
                this.rwlock.unlockWrite(stamp);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void reset(long nextIndex) {
        long stamp = this.rwlock.writeLock();
        try {
            this.journalIndex.clear();
            this.writer.reset(nextIndex);
        }
        finally {
            this.rwlock.unlockWrite(stamp);
        }
    }

    @Override
    public long getLastIndex() {
        return this.writer.getLastIndex();
    }

    @Override
    public long getFirstIndex() {
        if (!this.segments.isEmpty()) {
            return this.segments.firstEntry().getValue().index();
        }
        return 0L;
    }

    @Override
    public boolean isEmpty() {
        return this.writer.getNextIndex() - this.getFirstSegment().index() == 0L;
    }

    @Override
    public void flush() {
        this.writer.flush();
    }

    @Override
    public JournalReader openReader() {
        long stamped = this.acquireReadlock();
        SegmentedJournalReader reader = new SegmentedJournalReader(this);
        this.readers.add(reader);
        this.releaseReadlock(stamped);
        return reader;
    }

    @Override
    public boolean isOpen() {
        return this.open;
    }

    @Override
    public void close() {
        this.segments.values().forEach(segment -> {
            this.log.debug("Closing segment: {}", segment);
            segment.close();
        });
        this.currentSegment = null;
        this.open = false;
    }

    private synchronized void open() {
        long startTime = System.currentTimeMillis();
        for (JournalSegment segment : this.loadSegments()) {
            this.segments.put(segment.descriptor().index(), segment);
            this.journalMetrics.incSegmentCount();
        }
        if (!this.segments.isEmpty()) {
            this.currentSegment = this.segments.lastEntry().getValue();
        } else {
            JournalSegmentDescriptor descriptor = JournalSegmentDescriptor.builder().withId(1L).withIndex(1L).withMaxSegmentSize(this.maxSegmentSize).build();
            this.currentSegment = this.createSegment(descriptor);
            this.segments.put(1L, this.currentSegment);
            this.journalMetrics.incSegmentCount();
        }
        this.journalMetrics.observeJournalOpenDuration(System.currentTimeMillis() - startTime);
        this.deleteDeferredFiles();
    }

    private void assertOpen() {
        Preconditions.checkState((this.currentSegment != null ? 1 : 0) != 0, (Object)"journal not open");
    }

    private void assertDiskSpace() {
        if (this.directory().getUsableSpace() < Math.max(this.maxSegmentSize() * 3L, this.minFreeDiskSpace)) {
            throw new JournalException.OutOfDiskSpace("Not enough space to allocate a new journal segment");
        }
    }

    private long maxSegmentSize() {
        return this.maxSegmentSize;
    }

    private File directory() {
        return this.directory;
    }

    private synchronized void resetCurrentSegment() {
        JournalSegment lastSegment = this.getLastSegment();
        if (lastSegment != null) {
            this.currentSegment = lastSegment;
        } else {
            JournalSegmentDescriptor descriptor = JournalSegmentDescriptor.builder().withId(1L).withIndex(1L).withMaxSegmentSize(this.maxSegmentSize).build();
            this.currentSegment = this.createSegment(descriptor);
            this.segments.put(1L, this.currentSegment);
            this.journalMetrics.incSegmentCount();
        }
    }

    JournalSegment resetSegments(long index) {
        this.assertOpen();
        for (JournalSegment segment : this.segments.values()) {
            segment.delete();
            this.journalMetrics.decSegmentCount();
        }
        this.segments.clear();
        JournalSegmentDescriptor descriptor = JournalSegmentDescriptor.builder().withId(1L).withIndex(index).withMaxSegmentSize(this.maxSegmentSize).build();
        this.currentSegment = this.createSegment(descriptor);
        this.segments.put(index, this.currentSegment);
        this.journalMetrics.incSegmentCount();
        return this.currentSegment;
    }

    JournalSegment getFirstSegment() {
        this.assertOpen();
        Map.Entry<Long, JournalSegment> segment = this.segments.firstEntry();
        return segment != null ? segment.getValue() : null;
    }

    JournalSegment getLastSegment() {
        this.assertOpen();
        Map.Entry<Long, JournalSegment> segment = this.segments.lastEntry();
        return segment != null ? segment.getValue() : null;
    }

    synchronized JournalSegment getNextSegment() {
        this.assertOpen();
        this.assertDiskSpace();
        JournalSegment lastSegment = this.getLastSegment();
        JournalSegmentDescriptor descriptor = JournalSegmentDescriptor.builder().withId(lastSegment != null ? lastSegment.descriptor().id() + 1L : 1L).withIndex(this.currentSegment.lastIndex() + 1L).withMaxSegmentSize(this.maxSegmentSize).build();
        this.currentSegment = this.createSegment(descriptor);
        this.segments.put(descriptor.index(), this.currentSegment);
        this.journalMetrics.incSegmentCount();
        return this.currentSegment;
    }

    JournalSegment getNextSegment(long index) {
        Map.Entry<Long, JournalSegment> nextSegment = this.segments.higherEntry(index);
        return nextSegment != null ? nextSegment.getValue() : null;
    }

    synchronized JournalSegment getSegment(long index) {
        this.assertOpen();
        if (this.currentSegment != null && index > this.currentSegment.index()) {
            return this.currentSegment;
        }
        Map.Entry<Long, JournalSegment> segment = this.segments.floorEntry(index);
        if (segment != null) {
            return segment.getValue();
        }
        return this.getFirstSegment();
    }

    synchronized void removeSegment(JournalSegment segment) {
        this.segments.remove(segment.index());
        this.journalMetrics.decSegmentCount();
        segment.delete();
        this.resetCurrentSegment();
    }

    protected Collection<JournalSegment> loadSegments() {
        this.directory.mkdirs();
        ArrayList<JournalSegment> segments = new ArrayList<JournalSegment>();
        List<File> files = this.getSortedLogSegments();
        for (int i = 0; i < files.size(); ++i) {
            File file = files.get(i);
            try {
                this.log.debug("Found segment file: {}", (Object)file.getName());
                JournalSegment segment = this.loadExistingSegment(file);
                if (i > 0) {
                    this.checkForIndexGaps((JournalSegment)segments.get(i - 1), segment);
                }
                segments.add(segment);
                continue;
            }
            catch (CorruptedLogException e) {
                if (this.handleSegmentCorruption(files, segments, i)) {
                    return segments;
                }
                throw e;
            }
        }
        return segments;
    }

    private void checkForIndexGaps(JournalSegment prevSegment, JournalSegment segment) {
        if (prevSegment.lastIndex() != segment.index() - 1L) {
            throw new CorruptedLogException(String.format("Log segment %s is not aligned with previous segment %s (last index: %d).", segment, prevSegment, prevSegment.lastIndex()));
        }
    }

    private boolean handleSegmentCorruption(List<File> files, List<JournalSegment> segments, int failedIndex) {
        long lastSegmentIndex = 0L;
        if (!segments.isEmpty()) {
            JournalSegment previousSegment = segments.get(segments.size() - 1);
            lastSegmentIndex = previousSegment.lastIndex();
        }
        if (this.lastWrittenIndex > lastSegmentIndex) {
            return false;
        }
        this.log.debug("Found corrupted segment after last ack'ed index {}. Deleting segments {} - {}", new Object[]{this.lastWrittenIndex, files.get(failedIndex).getName(), files.get(files.size() - 1).getName()});
        for (int i = failedIndex; i < files.size(); ++i) {
            File file = files.get(i);
            try {
                Files.delete(file.toPath());
                continue;
            }
            catch (IOException e) {
                throw new JournalException(String.format("Failed to delete log segment '%s' when handling corruption.", file.getName()), e);
            }
        }
        return true;
    }

    private List<File> getSortedLogSegments() {
        File[] files = this.directory.listFiles(file -> file.isFile() && JournalSegmentFile.isSegmentFile(this.name, file));
        if (files == null) {
            throw new IllegalStateException(String.format("Could not list files in directory '%s'. Either the path doesn't point to a directory or an I/O error occurred.", this.directory));
        }
        Arrays.sort(files, Comparator.comparingInt(f -> JournalSegmentFile.getSegmentIdFromPath(f.getName())));
        return Arrays.asList(files);
    }

    private void deleteDeferredFiles() {
        try (DirectoryStream<Path> segmentsToDelete = Files.newDirectoryStream(this.directory.toPath(), path -> JournalSegmentFile.isDeletedSegmentFile(this.name, path.getFileName().toString()));){
            segmentsToDelete.forEach(this::deleteDeferredFile);
        }
        catch (IOException e) {
            this.log.warn("Could not delete segment files marked for deletion in {}. This can result in unnecessary disk usage.", (Object)this.directory.toPath(), (Object)e);
        }
    }

    private void deleteDeferredFile(Path segmentFileToDelete) {
        try {
            Files.deleteIfExists(segmentFileToDelete);
        }
        catch (IOException e) {
            this.log.warn("Could not delete file {} which is marked for deletion. This can result in unnecessary disk usage.", (Object)segmentFileToDelete, (Object)e);
        }
    }

    private JournalSegmentDescriptor readDescriptor(File file) {
        JournalSegmentDescriptor journalSegmentDescriptor;
        block12: {
            FileChannel channel = FileChannel.open(file.toPath(), StandardOpenOption.READ);
            try {
                byte version = this.readVersion(channel, file.getName());
                int length = JournalSegmentDescriptor.getEncodingLengthForVersion(version);
                if (file.length() < (long)length) {
                    throw new CorruptedLogException(String.format("Expected segment '%s' with version %d to be at least %d bytes long but it only has %d.", file.getName(), version, length, file.length()));
                }
                ByteBuffer buffer = ByteBuffer.allocate(length);
                int readBytes = channel.read(buffer, 0L);
                if (readBytes != -1 && readBytes < length) {
                    throw new JournalException(String.format("Expected to read %d bytes of segment '%s' with %d version but only read %d bytes.", length, file.getName(), version, readBytes));
                }
                buffer.flip();
                journalSegmentDescriptor = new JournalSegmentDescriptor(buffer);
                if (channel == null) break block12;
            }
            catch (Throwable throwable) {
                try {
                    if (channel != null) {
                        try {
                            channel.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IndexOutOfBoundsException e) {
                    throw new JournalException(String.format("Expected to read descriptor of segment '%s', but nothing was read.", file.getName()), e);
                }
                catch (UnknownVersionException e) {
                    throw new CorruptedLogException(String.format("Couldn't read or recognize version of segment '%s'.", file.getName()), e);
                }
                catch (IOException e) {
                    throw new JournalException(e);
                }
            }
            channel.close();
        }
        return journalSegmentDescriptor;
    }

    private byte readVersion(FileChannel channel, String fileName) throws IOException {
        ByteBuffer buffer = ByteBuffer.allocate(1);
        int readBytes = channel.read(buffer);
        if (readBytes == 0) {
            throw new JournalException(String.format("Expected to read the version byte from segment '%s' but nothing was read.", fileName));
        }
        if (readBytes == -1) {
            throw new CorruptedLogException(String.format("Expected to read the version byte from segment '%s' but got EOF instead.", fileName));
        }
        return buffer.get(0);
    }

    public void closeReader(SegmentedJournalReader segmentedJournalReader) {
        this.readers.remove(segmentedJournalReader);
    }

    void resetAdvancedReaders(long index) {
        for (SegmentedJournalReader reader : this.readers) {
            if (reader.getNextIndex() <= index) continue;
            reader.unsafeSeek(index);
        }
    }

    public JournalMetrics getJournalMetrics() {
        return this.journalMetrics;
    }

    public JournalIndex getJournalIndex() {
        return this.journalIndex;
    }

    long acquireReadlock() {
        return this.rwlock.readLock();
    }

    void releaseReadlock(long stamp) {
        this.rwlock.unlockRead(stamp);
    }

    private JournalSegment createSegment(JournalSegmentDescriptor descriptor) {
        MappedByteBuffer mappedSegment;
        File segmentFile = JournalSegmentFile.createSegmentFile(this.name, this.directory, descriptor.id());
        try {
            mappedSegment = this.mapNewSegment(segmentFile, descriptor);
        }
        catch (IOException e) {
            throw new JournalException(String.format("Failed to map new segment %s", segmentFile), e);
        }
        try {
            descriptor.copyTo(mappedSegment);
            mappedSegment.force();
        }
        catch (InternalError e) {
            throw new JournalException(String.format("Failed to ensure durability of segment %s with descriptor %s, rolling back", segmentFile, descriptor), e);
        }
        try {
            FileUtil.flushDirectory((Path)this.directory.toPath());
        }
        catch (IOException e) {
            throw new JournalException(String.format("Failed to flush journal directory after creating segment %s", segmentFile), e);
        }
        return this.loadSegment(segmentFile, mappedSegment, descriptor);
    }

    private JournalSegment loadExistingSegment(File segmentFile) {
        MappedByteBuffer mappedSegment;
        JournalSegmentDescriptor descriptor = this.readDescriptor(segmentFile);
        try {
            mappedSegment = this.mapSegment(segmentFile, descriptor, Collections.emptySet());
        }
        catch (IOException e) {
            throw new JournalException(String.format("Failed to load existing segment %s", segmentFile), e);
        }
        return this.loadSegment(segmentFile, mappedSegment, descriptor);
    }

    private JournalSegment loadSegment(File file, MappedByteBuffer buffer, JournalSegmentDescriptor descriptor) {
        JournalSegmentFile segmentFile = new JournalSegmentFile(file);
        return new JournalSegment(segmentFile, descriptor, buffer, this.lastWrittenIndex, this.journalIndex);
    }

    private MappedByteBuffer mapNewSegment(File segmentFile, JournalSegmentDescriptor descriptor) throws IOException {
        try {
            return this.mapSegment(segmentFile, descriptor, Set.of(StandardOpenOption.CREATE_NEW));
        }
        catch (FileAlreadyExistsException e) {
            if (this.lastWrittenIndex >= descriptor.index()) {
                throw new JournalException(String.format("Failed to create journal segment %s, as it already exists, and the last written index %d indicates we've already written to it", segmentFile, this.lastWrittenIndex), e);
            }
            return this.mapSegment(segmentFile, descriptor, Set.of(StandardOpenOption.TRUNCATE_EXISTING));
        }
    }

    private MappedByteBuffer mapSegment(File segmentFile, JournalSegmentDescriptor descriptor, Set<OpenOption> extraOptions) throws IOException {
        HashSet<OpenOption> options = new HashSet<OpenOption>(extraOptions);
        options.add(StandardOpenOption.READ);
        options.add(StandardOpenOption.WRITE);
        try (FileChannel channel = FileChannel.open(segmentFile.toPath(), options, new FileAttribute[0]);){
            MappedByteBuffer mappedSegment = channel.map(FileChannel.MapMode.READ_WRITE, 0L, descriptor.maxSegmentSize());
            mappedSegment.order(ENDIANNESS);
            MappedByteBuffer mappedByteBuffer = mappedSegment;
            return mappedByteBuffer;
        }
    }
}

